/*
**************************************************************************
                               description
                             -------------------
    copyright            : (C) 2002 by Andreas Zehender
    email                : zehender@kde.org
**************************************************************************

**************************************************************************
*                                                                        *
*  This program is free software; you can redistribute it and/or modify  *
*  it under the terms of the GNU General Public License as published by  *
*  the Free Software Foundation; either version 2 of the License, or     *
*  (at your option) any later version.                                   *
*                                                                        *
**************************************************************************/

#include "pmblobsphere.h"

#include "pmoutputdevice.h"
#include "pmxmlhelper.h"
#include "pmblobsphereedit.h"
#include "pmmemento.h"
#include "pm3dcontrolpoint.h"
#include "pmdistancecontrolpoint.h"
#include "pmdefaults.h"

#include <kdebug.h>
#include "pmglobals.h"

#include <klocale.h>

/** default param for the sphere */
const double c_defaultRadius = 0.5;
const PMVector c_defaultCentre = PMVector( 0, 0, 0 );
const double c_defaultStrength = 1.0;

/** default sphere structure */
PMViewStructure* PMBlobSphere::s_pDefaultViewStructure = 0;

int PMBlobSphere::s_vStep = c_defaultBlobSphereVSteps;
int PMBlobSphere::s_uStep = c_defaultBlobSphereUSteps;
int PMBlobSphere::s_parameterKey = 0;

PMBlobSphere::PMBlobSphere( )
      : Base( )
{
   m_radius = c_defaultRadius;
   m_centre = c_defaultCentre;
   m_strength = c_defaultStrength;
}

PMBlobSphere::~PMBlobSphere( )
{
}


QString PMBlobSphere::description( ) const
{
   return i18n( "blob sphere" );
}

bool PMBlobSphere::isA( PMObjectType t ) const
{
   if( t == PMTBlobSphere )
      return true;
   return Base::isA( t );
}

void PMBlobSphere::serialize( PMOutputDevice& dev ) const
{
   dev.objectBegin( "sphere" );
   serializeName( dev );
   dev.writeLine( m_centre.serialize( ) + QString( ", %1," ).arg( m_radius ) );
   dev.writeLine( QString( "strength %1" ).arg( m_strength ) );
   Base::serialize( dev );
   dev.objectEnd( );
}

void PMBlobSphere::serialize( QDomElement& e, QDomDocument& doc ) const
{
   e.setAttribute( "centre", m_centre.serializeXML( ) );
   e.setAttribute( "radius", m_radius );
   e.setAttribute( "strength", m_strength );
   Base::serialize( e, doc );
}

void PMBlobSphere::readAttributes( const PMXMLHelper& h )
{
   m_centre = h.vectorAttribute( "centre", c_defaultCentre );
   m_radius = h.doubleAttribute( "radius", c_defaultRadius );
   m_strength = h.doubleAttribute( "strength", c_defaultStrength );
   Base::readAttributes( h );
}

PMDialogEditBase* PMBlobSphere::editWidget( QWidget* parent ) const
{
   return new PMBlobSphereEdit( parent );
}

void PMBlobSphere::restoreMemento( PMMemento* s )
{
   PMMementoDataIterator it( s );
   PMMementoData* data;

   for( ; it.current( ); ++it )
   {
      data = it.current( );
      if( data->objectType( ) == PMTBlobSphere )
      {
         switch( data->valueID( ) )
         {
            case PMRadiusID:
               setRadius( data->doubleData( ) );
               break;
            case PMCentreID:
               setCentre( data->vectorData( ) );
               break;
            case PMStrengthID:
               setStrength( data->doubleData( ) );
               break;
            default:
               kdError( PMArea ) << "Wrong ID in PBlobSphere::restoreMemento\n";
               break;
         }
      }
   }
   Base::restoreMemento( s );

}

void PMBlobSphere::controlPoints( PMControlPointList& list )
{
   PM3DControlPoint* p = new PM3DControlPoint( m_centre, PMCentreID,
                                               i18n( "Center" ) );
   list.append( p );
   list.append( new PMDistanceControlPoint( p, PMVector( 1.0, 0.0, 0.0 ),
                                            m_radius, PMRadiusID,
                                            i18n( "Radius (x)" ) ) );
   list.append( new PMDistanceControlPoint( p, PMVector( 0.0, 1.0, 0.0 ),
                                            m_radius, PMRadiusID,
                                            i18n( "Radius (y)" ) ) );
   list.append( new PMDistanceControlPoint( p, PMVector( 0.0, 0.0, 1.0 ),
                                            m_radius, PMRadiusID,
                                            i18n( "Radius (z)" ) ) );
}

void PMBlobSphere::controlPointsChanged( PMControlPointList& list )
{
   PMControlPoint* p;
   bool radiusChanged = false;

   for( p = list.first( ); p; p = list.next( ) )
   {
      if( p->changed( ) )
      {
         switch( p->id( ) )
         {
            case PMCentreID:
               setCentre( ( ( PM3DControlPoint* ) p )->point( ) );
               break;
            case PMRadiusID:
               setRadius( ( ( PMDistanceControlPoint* ) p )->distance( ) );
               radiusChanged = true;
               break;
            default:
               kdError( PMArea ) << "Wrong ID in PMBlobSphere::controlPointsChanged\n";
               break;
         }
      }
   }
   
   if( radiusChanged )
      for( p = list.first( ); p; p = list.next( ) )
         if( p->id( ) == PMRadiusID )
            ( ( PMDistanceControlPoint* ) p )->setDistance( m_radius );
}

bool PMBlobSphere::isDefault( )
{
  if( ( m_radius == c_defaultRadius ) && ( m_centre == c_defaultCentre ) )
      return true;
   return false;
}

void PMBlobSphere::createViewStructure( )
{
   if( !m_pViewStructure )
   {
      m_pViewStructure = new PMViewStructure( defaultViewStructure( ) );
      m_pViewStructure->points( ).detach( );
   }
   
   createPoints( m_pViewStructure->points( ), m_centre, m_radius );
}

PMViewStructure* PMBlobSphere::defaultViewStructure( ) const
{
   if( !s_pDefaultViewStructure )
   {
      int u, v;

      s_pDefaultViewStructure =
         new PMViewStructure( s_vStep * ( s_uStep - 1 ) + 2,
                              s_vStep * ( s_uStep - 1 ) * 2 + s_vStep );
      
      PMLineArray& lines = s_pDefaultViewStructure->lines( );
      
   
      // points
      createPoints( s_pDefaultViewStructure->points( ), c_defaultCentre,
                    c_defaultRadius );

      int offset = 0;

      // horizontal lines
      for( u = 0; u < ( s_uStep - 1 ); u++ )
      {
         for( v = 0; v < ( s_vStep - 1 ); v++ )
            lines[offset + v] =
               PMLine( u * s_vStep + v + 1, u * s_vStep + v + 2 );
         lines[offset + s_vStep - 1] =
            PMLine( u * s_vStep + 1, u * s_vStep + s_vStep );
         
         offset += s_vStep;
      }

      // vertical lines
      // lines at the "north pole"
      for( v = 0; v < s_vStep; v++ )
         lines[offset + v] = PMLine( 0, v + 1 );
      offset += s_vStep;

      for( v = 0; v < s_vStep; v++ )
      {
         for( u = 0; u < ( s_uStep - 2 ); u++ )
         {
            lines[offset + u] =
               PMLine( u * s_vStep + v + 1, ( u + 1 ) * s_vStep + v + 1 );
         }
         offset += ( s_uStep - 2 );
      }
      // lines at the "south pole"
      for( v = 0; v < s_vStep; v++ )
         lines[offset + v] = PMLine( ( s_uStep - 2 ) * s_vStep + v + 1,
                                     ( s_uStep - 1 ) * s_vStep + 1 );
      // offset += s_vStep;
   }
   
   return s_pDefaultViewStructure;
}

void PMBlobSphere::createPoints( PMPointArray& points, const PMVector& centre,
                             double radius )
{
   double l_UradStep = M_PI / s_uStep;
   double l_VradStep = ( 2.0 * M_PI ) / s_vStep;
   double l_u = l_UradStep;
   int u, v;
   
   points[0] = PMPoint( centre + PMVector( 0, radius, 0 ) );
   points[s_vStep * ( s_uStep - 1 ) + 1] =
      PMPoint( centre - PMVector( 0, radius, 0 ) );

   for( u = 0; u < ( s_uStep - 1 ); u++ )
   {
      double l_v = 0.0;
      double l_rcosu = radius * sin( l_u );
      double y = ( radius * cos( l_u ) ) + centre[1];
      for( v = 0; v < s_vStep ; v++ )
      {

         double x = ( l_rcosu * cos( l_v ) ) + centre[0];
         double z = ( l_rcosu * sin( l_v ) ) + centre[2];

         points[u * s_vStep + v + 1] = PMPoint( x, y, z );
         l_v = l_v + l_VradStep;
      }
      l_u = l_u + l_UradStep;
   }
}

void PMBlobSphere::setRadius( double radius )
{
   if( m_radius != radius )
   {
      if( m_pMemento )
         m_pMemento->addData( PMTBlobSphere, PMRadiusID, m_radius );
      m_radius = radius;
      setViewStructureChanged( );
   }
}

void PMBlobSphere::setCentre( const PMVector& centre ) 
{
   if( m_centre != centre )
   {
      if( m_pMemento )
         m_pMemento->addData( PMTBlobSphere, PMCentreID, m_centre );
      m_centre = centre;
      setViewStructureChanged( );
   }
}

void PMBlobSphere::setStrength( double s )
{
   if( m_strength != s )
   {
      if( m_pMemento )
         m_pMemento->addData( PMTBlobSphere, PMStrengthID, m_strength );
      m_strength = s;
   }
}

bool PMBlobSphere::canInsert( PMObjectType t, const PMObject*,
                              const PMObjectList* ) const
{
   switch( t )
   {
      case PMTComment:
      case PMTRaw:
      case PMTTranslate:
      case PMTRotate:
      case PMTScale:
      case PMTMatrix:
      case PMTTexture:
      case PMTPigment:
      case PMTNormal:
      case PMTFinish:
         return true;
         break;
      default:
         break;
   }
   return false;
}

void PMBlobSphere::setUSteps( int u )
{
   if( u >= 2 )
   {
      s_uStep = u;
      if( s_pDefaultViewStructure )
      {
         delete s_pDefaultViewStructure;
         s_pDefaultViewStructure = 0;
      }
   }
   else
      kdDebug( PMArea ) << "PMBlobSphere::setUSteps: U must be greater than 1\n";
   s_parameterKey++;
}

void PMBlobSphere::setVSteps( int v )
{
   if( v >= 4 )
   {
      s_vStep = v;
      if( s_pDefaultViewStructure )
      {
         delete s_pDefaultViewStructure;
         s_pDefaultViewStructure = 0;
      }
   }
   else
      kdDebug( PMArea ) << "PMBlobSphere::setVSteps: V must be greater than 3\n";
   s_parameterKey++;
}

void PMBlobSphere::cleanUp( ) const
{
   if( s_pDefaultViewStructure )
      delete s_pDefaultViewStructure;
   s_pDefaultViewStructure = 0;
   Base::cleanUp( );
}
