------------------------------------------------------------------------------
--                                                                          --
--                         GNAT COMPILER COMPONENTS                         --
--                                                                          --
--                             E X P _ U T I L                              --
--                                                                          --
--                                 B o d y                                  --
--                                                                          --
--                            $Revision: 1.111 $                            --
--                                                                          --
--        Copyright (c) 1992,1993,1994,1995 NYU, All Rights Reserved        --
--                                                                          --
-- GNAT is free software;  you can  redistribute it  and/or modify it under --
-- terms of the  GNU General Public License as published  by the Free Soft- --
-- ware  Foundation;  either version 2,  or (at your option) any later ver- --
-- sion.  GNAT is distributed in the hope that it will be useful, but WITH- --
-- OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
-- or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License --
-- for  more details.  You should have  received  a copy of the GNU General --
-- Public License  distributed with GNAT;  see file COPYING.  If not, write --
-- to the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA. --
--                                                                          --
------------------------------------------------------------------------------

with Atree;    use Atree;
with Einfo;    use Einfo;
with Elists;   use Elists;
with Expander; use Expander;
with Exp_Ch7;  use Exp_Ch7;
with Itypes;   use Itypes;
with Nlists;   use Nlists;
with Nmake;    use Nmake;
with Output;   use Output;
with Sem;      use Sem;
with Sem_Eval; use Sem_Eval;
with Sem_Res;  use Sem_Res;
with Sem_Util; use Sem_Util;
with Sinfo;    use Sinfo;
with Snames;   use Snames;
with Tbuild;   use Tbuild;
with Ttypes;   use Ttypes;
with Uintp;    use Uintp;
with Urealp;   use Urealp;

package body Exp_Util is

   -----------------------
   -- Local Subprograms --
   -----------------------

   function Make_Literal_Range
     (Loc         : Source_Ptr;
      Literal_Typ : Entity_Id;
      Index_Typ   : Entity_Id)
      return        Node_Id;
   --  Produce a Range node whose bounds are:
   --    Index_Typ'first .. Index_Typ'First + Length (Literal_Typ)
   --  this is used for expanding declarations like X : String := "sdfgdfg";

   function Make_Subtype_From_Expr
     (N       : Node_Id;
      E       : Multi_Use.Exp_Id;
      Unc_Typ : Entity_Id)
      return    Node_Id;
   --  Creates an appropriate Subtype Indication for unconstrained object
   --  declarations. Unc_Typ can be an unconstrained array or record, or
   --  a classwide type.

   -----------------------
   -- Package Multi_Use --
   -----------------------

   package body Multi_Use is

      -------------
      -- Prepare --
      -------------

      procedure Prepare
        (Exp  : Node_Id;
         Res  : out Exp_Id;
         Code : out List_Id)
      is
         Loc          : constant Source_Ptr := Sloc (Exp);
         Exp_Type     : constant Entity_Id := Etype (Exp);
         Ref_Type     : Entity_Id;
         Ptr_Typ_Decl : Node_Id;
         New_Exp      : Node_Id;
         E            : Node_Id;

         Volatile_Case : constant Boolean
           := Is_Entity_Name (Exp) and then Is_Volatile (Entity (Exp));

      begin
         --  First copy the flags that need to be inherited

         Res.Assignment_OK := Assignment_OK (Exp);

         if Is_Entity_Name (Exp) and not Volatile_Case then
            Res.Id        := Entity (Exp);
            Res.Is_Access := False;
            Code := No_List;

         --  If the expression has the form v.all then we already have a
         --  pointer to the expression, we don't need to create one

         elsif Nkind (Exp) = N_Explicit_Dereference
           and then Is_Entity_Name (Prefix (Exp))
         then
            Res.Id        := Entity (Prefix (Exp));
            Res.Is_Access := True;
            Code := No_List;

         --  For expressions that are names, we can use a renaming scheme

         elsif Is_Variable (Exp) and then not Volatile_Case then
            Code := New_List;

            Res.Id := Make_Defining_Identifier (Loc, New_Internal_Name ('R'));
            Res.Is_Access := False;

            Append_To (Code,
              Make_Object_Renaming_Declaration (Loc,
                Defining_Identifier => Res.Id,
                Subtype_Mark        => New_Reference_To (Exp_Type, Loc),
                Name                => Relocate_Node (Exp)));

         --  If it is a scalar type or a volatile object, just make a copy

         elsif Is_Elementary_Type (Exp_Type) or else Volatile_Case then
            Code := New_List;

            Res.Id := Make_Defining_Identifier (Loc, New_Internal_Name ('R'));
            Res.Is_Access := False;

            Append_To (Code,
              Make_Object_Declaration (Loc,
                Defining_Identifier => Res.Id,
                Object_Definition   => New_Reference_To (Exp_Type, Loc),
                Expression          => Relocate_Node (Exp)));

         --  Otherwise we generate a pointer to the value

         else
            Ref_Type :=
              Make_Defining_Identifier (Loc, New_Internal_Name ('A'));
            Code := New_List;

            Ptr_Typ_Decl :=
              Make_Full_Type_Declaration (Loc,
                Defining_Identifier => Ref_Type,
                Type_Definition =>
                  Make_Access_To_Object_Definition (Loc,
                    All_Present => True,
                    Subtype_Indication =>
                      New_Reference_To (Exp_Type, Loc)));

            --  If the type of the expression is an implicit type defined in
            --  the expression node itself it better be transfered to the
            --  access type declaration to avoid forward references.

            if Is_Itype (Exp_Type) and then Nkind (Exp) in N_Has_Itypes then
               Transfer_Itypes (From => Exp, To => Ptr_Typ_Decl);
            end if;

            --  Furthermore if the expression is an expression-actions then
            --  move actions upfront too since the type can depend on those
            --  actions

            if Nkind (Exp) = N_Expression_Actions then
               Append_List_To (Code, Actions (Exp));
               E := Expression (Exp);
            else
               E := Exp;
            end if;

            Append_To (Code, Ptr_Typ_Decl);

            Res.Id := Make_Defining_Identifier (Loc, New_Internal_Name ('R'));
            Res.Is_Access := True;

            if Nkind (E) = N_Explicit_Dereference then
               New_Exp := Relocate_Node (Prefix (E));
            else
               New_Exp := Make_Reference (Loc, Relocate_Node (E));
            end if;

            Append_To (Code,
              Make_Object_Declaration (Loc,
                Defining_Identifier => Res.Id,
                Object_Definition   => New_Reference_To (Ref_Type, Loc),
                Expression          => New_Exp));
         end if;
      end Prepare;

      ----------------
      -- New_Exp_Id --
      ----------------

      function New_Exp_Id (Exp : Node_Id; N : Node_Id) return Exp_Id is
         Res  : Exp_Id;
         Code : List_Id;

      begin
         Prepare (Exp, Res, Code);

         if Present (Code) then
            Insert_List_Before_And_Analyze (N, Code);
         end if;

         return Res;
      end New_Exp_Id;

      ----------------
      -- New_Exp_Id --
      ----------------

      function New_Exp_Id (Exp : Node_Id; L : List_Id) return Exp_Id is
         Res  : Exp_Id;
         Code : List_Id;

      begin
         Prepare (Exp, Res, Code);

         if Present (Code) then
            Append_List_To (L, Code);
         end if;

         return Res;
      end New_Exp_Id;

      -------------
      -- New_Ref --
      -------------

      function New_Ref (E : Exp_Id; Loc : Source_Ptr) return  Node_Id is
         R : Node_Id  := New_Reference_To (E.Id, Loc);

      begin
         if E.Is_Access then
            R := Make_Explicit_Dereference (Loc, R);
         end if;

         --  Copy the inherited flags

         Set_Assignment_OK (R, E.Assignment_OK);
         return R;
      end New_Ref;

      ----------
      -- Wrap --
      ----------

      function Wrap (Code : List_Id; N : Node_Id) return Node_Id is
      begin
         if No (Code) then
            return N;
         else
            return
              Make_Expression_Actions (Sloc (N),
                Actions    => Code,
                Expression => N);
         end if;
      end Wrap;

   end Multi_Use;

   --------------------------
   -- Append_Freeze_Action --
   --------------------------

   procedure Append_Freeze_Action (T : Entity_Id; N : Node_Id) is
      Fnode : constant Node_Id := Freeze_Node (T);

   begin
      if not Present (Actions (Fnode)) then
         Set_Actions (Fnode, New_List);
      end if;

      Append (N, Actions (Fnode));
   end Append_Freeze_Action;

   ---------------------------
   -- Append_Freeze_Actions --
   ---------------------------

   procedure Append_Freeze_Actions (T : Entity_Id; L : List_Id) is
      Fnode : constant Node_Id := Freeze_Node (T);

   begin
      if No (L) then
         return;

      else
         if No (Actions (Fnode)) then
            Set_Actions (Fnode, L);

         else
            Append_List (L, Actions (Fnode));
         end if;

      end if;
   end Append_Freeze_Actions;

   ------------------------
   -- Build_Runtime_Call --
   ------------------------

   function Build_Runtime_Call (Loc : Source_Ptr; RE : RE_Id) return Node_Id is
   begin
      return
        Make_Procedure_Call_Statement (Loc,
          Name => New_Reference_To (RTE (RE), Loc));
   end Build_Runtime_Call;

   --------------------------
   -- Compile_Time_Compare --
   --------------------------

   function Compile_Time_Compare (L, R : Node_Id) return Compare_Result is
      T : constant Entity_Id := Etype (L);

   begin
      --  Static expression case

      if Is_OK_Static_Expression (L)
        and then Is_OK_Static_Expression (R)
      then
         --  For the floating-point case, we have to be a little careful, since
         --  at compile time we are dealing with universal exact values, but at
         --  runtime, these will be in non-exact target form. That's why the
         --  returned results are LE and GE below instead of LT and GT.

         if Is_Floating_Point_Type (T) then
            declare
               Lo : constant Ureal := Expr_Value_R (L);
               Hi : constant Ureal := Expr_Value_R (R);

            begin
               if Lo < Hi then
                  return LE;
               elsif Lo = Hi then
                  return EQ;
               else
                  return GE;
               end if;
            end;

         --  For the integer case we know exactly (note that this includes the
         --  fixed-point case, where we know the run time integer values now)

         else
            declare
               Lo : constant Uint := Expr_Value (L);
               Hi : constant Uint := Expr_Value (R);

            begin
               if Lo < Hi then
                  return LT;
               elsif Lo = Hi then
                  return EQ;
               else
                  return GT;
               end if;
            end;
         end if;

      --  For now, say not known if non-static expressions, we will do better
      --  than this later on for some special cases where we can tell.

      else
         return Unknown;
      end if;
   end Compile_Time_Compare;

   ----------------
   -- Convert_To --
   ----------------

   function Convert_To (T : Entity_Id; Exp : Node_Id) return Node_Id is
   begin
      if Present (Etype (Exp))
        and then (Etype (Exp)) = T
      then
         return Exp;
      else
         return
           Make_Type_Conversion (Sloc (Exp),
             Subtype_Mark => New_Occurrence_Of (T, Sloc (Exp)),
             Expression => Exp);
      end if;
   end Convert_To;

   ----------------------------
   -- New_Class_Wide_Subtype --
   ----------------------------

   function New_Class_Wide_Subtype
     (CW_Typ : Entity_Id;
      N      : Node_Id)
      return   Entity_Id
   is
      Res      : Entity_Id := New_Itype (E_Void, N);
      Res_Name : constant Name_Id := Chars (Res);

   begin
      Set_Public_Status (Res);
      Copy_Node (CW_Typ, Res);
      Set_Chars (Res, Res_Name);
      Set_Ekind (Res, E_Class_Wide_Subtype);
      Set_Next_Entity (Res, Empty);
      Set_Etype (Res, Base_Type (CW_Typ));
      Set_Freeze_Node (Res, Empty);
      Set_Next_Itype  (Res, Empty);
      return (Res);
   end New_Class_Wide_Subtype;

   -------------------------------
   -- Expand_Class_Wide_Subtype --
   -------------------------------

   --  Create a record type used as an equivalent of any member
   --  of the class which takes its size from exp.

   --  Generate the following code:

   --   type Equiv_T is record
   --     _parent :  T (List of discriminant constaints taken from Exp);
   --     Ext__50 : Storage_Array (1 .. (Exp'size - Typ'size) / Storage_Unit);
   --   end Equiv_T;

   function Expand_Class_Wide_Subtype
     (N          : Node_Id;
      Class_Type : Entity_Id;
      E          : Multi_Use.Exp_Id)
      return       List_Id
   is
      Loc         : constant Source_Ptr := Sloc (N);
      Root_Typ    : constant Entity_Id  := Root_Type (Class_Type);
      Equiv_Type  : Entity_Id;
      Range_Type  : Entity_Id;
      Str_Type    : Entity_Id;
      List_Def    : List_Id := Empty_List;
      Constr_Root : Entity_Id;
      Sizexpr     : Node_Id;

   begin
      if not Has_Discriminants (Root_Typ) then
         Constr_Root := Root_Typ;
      else
         Constr_Root :=
           Make_Defining_Identifier (Loc, New_Internal_Name ('R'));

         --  subtype cstr__n is T (List of discr constraints taken from Exp)

         Append_To (List_Def,
           Make_Subtype_Declaration (Loc,
             Defining_Identifier => Constr_Root,
               Subtype_Indication =>
                 Make_Subtype_From_Expr (N, E, Root_Typ)));
      end if;

      --  subtype rg__xx is Storage_Offset range
      --                           (Expr'size - typ'size) / Storage_Unit

      Range_Type := Make_Defining_Identifier (Loc, New_Internal_Name ('G'));

      Sizexpr :=
        Make_Op_Subtract (Loc,
          Left_Opnd =>
            Make_Attribute_Reference (Loc,
              Prefix => Multi_Use.New_Ref (E, Loc),
              Attribute_Name => Name_Size),
          Right_Opnd =>
            Make_Attribute_Reference (Loc,
              Prefix => New_Reference_To (Constr_Root, Loc),
              Attribute_Name => Name_Size));

      Set_Paren_Count (Sizexpr, 1);

      Append_To (List_Def,
        Make_Subtype_Declaration (Loc,
          Defining_Identifier => Range_Type,
          Subtype_Indication =>
            Make_Subtype_Indication (Loc,
              Subtype_Mark => New_Reference_To (RTE (RE_Storage_Offset), Loc),
              Constraint => Make_Range_Constraint (Loc,
                Range_Expression =>
                  Make_Range (Loc,
                    Low_Bound => Make_Integer_Literal (Loc, Uint_1),
                    High_Bound =>
                      Make_Op_Divide (Loc,
                        Left_Opnd => Sizexpr,
                        Right_Opnd => Make_Integer_Literal (Loc,
                          Intval =>
                            UI_From_Int (System_Storage_Unit))))))));

      --  subtype str__nn is Storage_Array (rg__x);

      Str_Type := Make_Defining_Identifier (Loc, New_Internal_Name ('S'));
      Append_To (List_Def,
        Make_Subtype_Declaration (Loc,
          Defining_Identifier => Str_Type,
          Subtype_Indication =>
            Make_Subtype_Indication (Loc,
              Subtype_Mark => New_Reference_To (RTE (RE_Storage_Array), Loc),
              Constraint =>
                Make_Index_Or_Discriminant_Constraint (Loc,
                  Constraints =>
                    New_List (New_Reference_To (Range_Type, Loc))))));

      --  type Equiv_T is record
      --    _parent : Tnn;
      --    E : Str_Type;
      --  end Equiv_T;

      Equiv_Type := Make_Defining_Identifier (Loc, New_Internal_Name ('T'));

      --  Avoid the generation of an init procedure

      Set_Is_Frozen (Equiv_Type);

      Append_To (List_Def,
        Make_Full_Type_Declaration (Loc,
          Defining_Identifier => Equiv_Type,
          Type_Definition =>
            Make_Record_Definition (Loc,
              Component_List => Make_Component_List (Loc,
                Component_Items => New_List (
                  Make_Component_Declaration (Loc,
                    Defining_Identifier =>
                      Make_Defining_Identifier (Loc, Name_uParent),
                    Subtype_Indication => New_Reference_To (Constr_Root, Loc)),
                  Make_Component_Declaration (Loc,
                    Defining_Identifier =>
                      Make_Defining_Identifier (Loc,
                        Chars => New_Internal_Name ('X')),
                    Subtype_Indication => New_Reference_To (Str_Type, Loc))),
                Variant_Part => Empty))));

      Set_Equivalent_Type (Class_Type, Equiv_Type);
      return List_Def;
   end Expand_Class_Wide_Subtype;

   ------------------------------
   -- Expand_Subtype_From_Expr --
   ------------------------------

   --  This function is applicable for both static and dynamic allocation of
   --  objects which are constrained by an initial expression. Basically it
   --  transforms an unconstrained subtype indication into a constrained one.
   --  The expression may also be transformed in certain cases in order to
   --  avoid multiple evaulation. In the static allocation case, the general
   --  scheme is :
   --     Val : T := Expr;
   --        is transformed into
   --     Val : Constrained_Subtype_of_T := Maybe_Modified_Expr;
   --
   --  Here are the main cases :
   --
   --  <if Expr is a Slice>
   --    Val : T ([Index_Subtype (Expr)]) := Expr;
   --
   --  <elsif Expr is a String Literal>
   --    Val : T (T'First .. T'First + Length (string literal) - 1) := Expr;
   --
   --  <elsif Expr is Constrained>
   --    Val : Type_Of_Expr := Expr;
   --
   --  <elsif Expr is an entity_name>
   --    Val : T (contraints taken from Expr) := Expr;
   --
   --  <else>
   --    type Axxx is access all T;
   --    Rval : Axxx := Expr'ref;
   --    Val  : T (contraints taken from Rval) := Rval.all;
   --    ??? note: when the Expression is allocated in the secondary stack
   --              we could use it directly instead of copying it by declaring
   --              Val : T (...) renames Rval.all

   procedure Expand_Subtype_From_Expr
     (N             : Node_Id;
      Unc_Type      : Entity_Id;
      Subtype_Indic : Node_Id;
      Exp           : Node_Id)
   is
      Loc           : constant Source_Ptr := Sloc (N);
      Exp_Typ       : constant Entity_Id  := Etype (Exp);
      E             : Multi_Use.Exp_Id;

   begin
      if not Expander_Active then
         return;
      end if;

      if Nkind (Exp) = N_Slice then
         Rewrite_Substitute_Tree (Subtype_Indic,
           Make_Subtype_Indication (Loc,
             Subtype_Mark => New_Reference_To (Unc_Type, Loc),
             Constraint =>
               Make_Index_Or_Discriminant_Constraint (Loc,
                 Constraints => New_List
                   (New_Reference_To (Etype (First_Index (Exp_Typ)), Loc)))));

      elsif Ekind (Exp_Typ) = E_String_Literal_Subtype then
         Rewrite_Substitute_Tree (Subtype_Indic,
           Make_Subtype_Indication (Loc,
             Subtype_Mark => New_Reference_To (Unc_Type, Loc),
             Constraint =>
               Make_Index_Or_Discriminant_Constraint (Loc,
                 Constraints => New_List (
                   Make_Literal_Range (Loc,
                     Literal_Typ => Exp_Typ,
                     Index_Typ   => Etype (First_Index (Unc_Type)))))));

      elsif Is_Constrained (Exp_Typ)
        and then not Is_Class_Wide_Type (Unc_Type)
      then
         Rewrite_Substitute_Tree (Subtype_Indic,
           New_Reference_To (Exp_Typ, Loc));

      else
         E := Multi_Use.New_Exp_Id (Exp, N);

         Rewrite_Substitute_Tree (Subtype_Indic,
           Make_Subtype_From_Expr (N, E, Unc_Type));

         Rewrite_Substitute_Tree (Exp, Multi_Use.New_Ref (E, Loc));
         Analyze (Exp);
         Resolve (Exp, Exp_Typ);
      end if;
   end Expand_Subtype_From_Expr;

   ------------------
   -- Find_Prim_Op --
   ------------------

   function Find_Prim_Op (T : Entity_Id; Name : Name_Id) return Entity_Id is
      Prim : Elmt_Id;
      Typ  : Entity_Id := T;

   begin
      if Is_Class_Wide_Type (Typ) then
         Typ := Root_Type (Typ);
      end if;

      Typ := Underlying_Type (Typ);
      Prim := First_Elmt (Primitive_Operations (Typ));
      while Chars (Node (Prim)) /= Name loop
         Prim := Next_Elmt (Prim);
         pragma Assert (Present (Prim));
      end loop;

      return Node (Prim);
   end Find_Prim_Op;

   --------------------------------
   -- Make_Constraints_From_Expr --
   --------------------------------

   --  1. if Expr is an uncontrained array expression, creates
   --    Unc_Type(Expr'first(1)..Expr'Last(1),..., Expr'first(n)..Expr'last(n))

   --  2. if Expr is a unconstrained discriminated type expression, creates
   --    Unc_Type(Expr.Discr1, ... , Expr.Discr_n)

   --  3. if Expr is class-wide, creates an implicit class wide subtype

   function Make_Subtype_From_Expr
     (N       : Node_Id;
      E       : Multi_Use.Exp_Id;
      Unc_Typ : Entity_Id)
      return    Node_Id
   is
      Loc         : constant Source_Ptr := Sloc (N);
      List_Constr : List_Id := New_List;
      D           : Entity_Id;

   begin

      if Is_Array_Type (Unc_Typ) then
         for J in 1 .. Number_Dimensions (Unc_Typ) loop
            Append_To (List_Constr,
              Make_Range (Loc,
                Low_Bound =>
                  Make_Attribute_Reference (Loc,
                    Prefix => Multi_Use.New_Ref (E, Loc),
                    Attribute_Name => Name_First,
                    Expressions => New_List (
                      Make_Integer_Literal (Loc, Intval => UI_From_Int (J)))),
                High_Bound =>
                  Make_Attribute_Reference (Loc,
                    Prefix => Multi_Use.New_Ref (E, Loc),
                    Attribute_Name => Name_Last,
                    Expressions => New_List (
                      Make_Integer_Literal (Loc,
                        Intval => UI_From_Int (J))))));
         end loop;

      elsif Is_Class_Wide_Type (Unc_Typ) then
         declare
            CW_Subtype : Entity_Id;

         begin
            CW_Subtype := New_Class_Wide_Subtype (Unc_Typ, N);

            if Expander_Active then
               Insert_List_Before_And_Analyze (N,
                 Expand_Class_Wide_Subtype (N, CW_Subtype, E));
            end if;

            return New_Occurrence_Of (CW_Subtype, Loc);
         end;

      else
         D := First_Discriminant (Unc_Typ);
         while (Present (D)) loop

            Append_To (List_Constr,
              Make_Selected_Component (Loc,
                Prefix => Multi_Use.New_Ref (E, Loc),
                Selector_Name => New_Reference_To (D, Loc)));

            D := Next_Discriminant (D);
         end loop;
      end if;

      return
        Make_Subtype_Indication (Loc,
          Subtype_Mark => New_Reference_To (Unc_Typ, Loc),
          Constraint   =>
            Make_Index_Or_Discriminant_Constraint (Loc,
              Constraints => List_Constr));
   end Make_Subtype_From_Expr;

   ------------------------
   -- Make_Literal_Range --
   ------------------------

   function Make_Literal_Range
     (Loc         : Source_Ptr;
      Literal_Typ : Entity_Id;
      Index_Typ   : Entity_Id)
      return        Node_Id
   is
   begin
         return
           Make_Range (Loc,
             Low_Bound =>
               Make_Attribute_Reference (Loc,
                 Prefix => New_Occurrence_Of (Index_Typ, Loc),
                 Attribute_Name => Name_First),
             High_Bound =>
               Make_Op_Subtract (Loc,
                  Left_Opnd =>
                    Make_Op_Add (Loc,
                      Left_Opnd =>
                        Make_Attribute_Reference (Loc,
                          Prefix => New_Occurrence_Of (Index_Typ, Loc),
                          Attribute_Name => Name_First),
                      Right_Opnd => Make_Integer_Literal (Loc,
                        String_Literal_Length (Literal_Typ))),
                  Right_Opnd => Make_Integer_Literal (Loc, Uint_1)));
   end Make_Literal_Range;

   ----------------------
   -- Make_Tagged_Copy --
   ----------------------
   --  Generate :
   --
   --      Annn : Tag := Lhs'tag;
   --      Bnnn : Finalizable_Ptr := Finalizable!(lhs).Prev;
   --      Cnnn : Finalizable_Ptr := Finalizable!(lhs).Next;
   --
   --      lhs := rhs;
   --
   --      lhs._tag := Annn;
   --      Finalizable!(lhs).Prev := Bnnn;
   --      Finalizable!(lhs).Next := Cnnn;

   procedure Expand_Tagged_Copy
     (N   : Node_Id;
      Lhs : Multi_Use.Exp_Id;
      Rhs : Multi_Use.Exp_Id;
      Typ : Entity_Id)
   is
      Loc      : constant Source_Ptr := Sloc (N);
      A        : constant Node_Id
              := Make_Defining_Identifier (Loc, New_Internal_Name ('A'));
      B        : constant Node_Id
                   := Make_Defining_Identifier (Loc, New_Internal_Name ('B'));
      C        : constant Node_Id
                   := Make_Defining_Identifier (Loc, New_Internal_Name ('C'));
      L        : List_Id;
      Ctrl_Ref : Node_Id;

   begin
      --  First save the tags and the finalization pointers

      L := New_List;

      if Is_Tagged_Type (Typ) then
         Append_To (L,
           Make_Object_Declaration (Loc,
             Defining_Identifier => A,
             Object_Definition => New_Reference_To (RTE (RE_Tag), Loc),
             Expression =>
               Make_Selected_Component (Loc,
                 Prefix => Multi_Use.New_Ref (Lhs, Loc),
                 Selector_Name =>
                   New_Reference_To (Tag_Component (Typ), Loc))));
      end if;

      if Controlled_Type (Typ) then

         Ctrl_Ref := Multi_Use.New_Ref (Lhs, Loc);
         if Has_Controlled (Typ) then
            Ctrl_Ref :=
              Make_Selected_Component (Loc,
                Prefix => Ctrl_Ref,
                Selector_Name =>
                  New_Reference_To (Controller_Component (Typ), Loc));
         end if;

         Append_To (L,
           Make_Object_Declaration (Loc,
             Defining_Identifier => B,
             Object_Definition =>
               New_Reference_To (RTE (RE_Finalizable_Ptr), Loc),
             Expression =>
               Make_Selected_Component (Loc,
                 Prefix =>
                   Make_Unchecked_Type_Conversion (Loc,
                     Subtype_Mark =>
                       New_Reference_To (RTE (RE_Finalizable), Loc),
                     Expression => Ctrl_Ref),
                 Selector_Name => Make_Identifier (Loc, Name_Prev))));

         Append_To (L,
           Make_Object_Declaration (Loc,
             Defining_Identifier => C,
             Object_Definition =>
               New_Reference_To (RTE (RE_Finalizable_Ptr), Loc),
             Expression =>
               Make_Selected_Component (Loc,
                 Prefix =>
                   Make_Unchecked_Type_Conversion (Loc,
                     Subtype_Mark =>
                       New_Reference_To (RTE (RE_Finalizable), Loc),
                     Expression => New_Copy_Tree (Ctrl_Ref)),
                 Selector_Name => Make_Identifier (Loc, Name_Next))));
      end if;

      Insert_List_Before_And_Analyze (N, L);

      --  Do the copy

      Rewrite_Substitute_Tree (Name (N), Multi_Use.New_Ref (Lhs, Loc));
      Analyze (Name (N));

      Rewrite_Substitute_Tree (Expression (N), Multi_Use.New_Ref (Rhs, Loc));
      Analyze (Expression (N));

      --  Then restore tag and the finalization pointers

      L := New_List;

      if Is_Tagged_Type (Typ) then
         Append_To (L,
           Make_Assignment_Statement (Loc,
             Name =>
               Make_Selected_Component (Loc,
                 Prefix => Multi_Use.New_Ref (Lhs, Loc),
                 Selector_Name => New_Reference_To (Tag_Component (Typ), Loc)),
             Expression => New_Reference_To (A, Loc)));
      end if;

      if Controlled_Type (Typ) then

         Append_To (L,
           Make_Assignment_Statement (Loc,
             Name =>
               Make_Selected_Component (Loc,
                 Prefix =>
                   Make_Unchecked_Type_Conversion (Loc,
                     Subtype_Mark =>
                       New_Reference_To (RTE (RE_Finalizable), Loc),
                     Expression => New_Copy_Tree (Ctrl_Ref)),
                 Selector_Name => Make_Identifier (Loc, Name_Prev)),
             Expression => New_Reference_To (B, Loc)));

         Append_To (L,
           Make_Assignment_Statement (Loc,
             Name =>
               Make_Selected_Component (Loc,
                 Prefix =>
                   Make_Unchecked_Type_Conversion (Loc,
                     Subtype_Mark =>
                       New_Reference_To (RTE (RE_Finalizable), Loc),
                     Expression => New_Copy_Tree (Ctrl_Ref)),
                 Selector_Name => Make_Identifier (Loc, Name_Next)),
             Expression => New_Reference_To (C, Loc)));
      end if;

      Insert_List_After (N, L);
   end Expand_Tagged_Copy;

   --------------------------
   -- Unchecked_Convert_To --
   --------------------------

   function Unchecked_Convert_To
     (T    : Entity_Id;
      Exp  : Node_Id)
      return Node_Id
   is
   begin
      if Present (Etype (Exp))
        and then Base_Type (Etype (Exp)) = T
      then
         return Exp;

      else
         return
           Make_Unchecked_Type_Conversion (Sloc (Exp),
             Subtype_Mark => New_Occurrence_Of (T, Sloc (Exp)),
             Expression   => Exp);
      end if;
   end Unchecked_Convert_To;

   ----------------------------
   -- Wrap_Cleanup_Procedure --
   ----------------------------

   procedure Wrap_Cleanup_Procedure (N : Node_Id) is
      Loc   : constant Source_Ptr := Sloc (N);
      Stseq : constant Node_Id    := Handled_Statement_Sequence (N);
      Stmts : constant List_Id    := Statements (Stseq);

   begin
      Prepend_To (Stmts, Build_Runtime_Call (Loc, RE_Abort_Defer));
      Append_To  (Stmts, Build_Runtime_Call (Loc, RE_Abort_Undefer));
   end Wrap_Cleanup_Procedure;

end Exp_Util;
