/*
 * copyright (c) 1995 Danny Gasparovski
 *
 * Please read the file COPYRIGHT for the
 * terms and conditions of the copyright.
 */

/*
 * mbuf's in SLiRP are much simpler than the real mbufs in
 * FreeBSD.  Mbuf's cannot be chaned together.  If there's
 * more data than the mbuf could hold, an externel malloced
 * buffer is pointed to by m_ext (and the data pointers) 
 * and M_EXT is set in the flags
 */

#include "h/common.h"

struct	mbuf *mbutl;
char	*mclrefcnt;
int mbuf_alloced = 0;
struct mbuf m_freelist, m_usedlist;
int mbuf_thresh = 30;
int mbuf_max = 0;
int msize;

void
m_init()
{
	m_freelist.m_next = m_freelist.m_prev = &m_freelist;
	m_usedlist.m_next = m_usedlist.m_prev = &m_usedlist;
	/*
	 * Find a nice value for msize
	 * XXX if_maxlinkhdr allready in mtu
	 */
	msize = ifp.ifp_mtu + ifp.ifp_maxlinkhdr + sizeof(struct m_hdr);
}


/*
 * Get an mbuf from the free list, if there are none
 * malloc one
 * 
 * Because fragmentation can occur if we alloc new mbufs and
 * free old mbufs, we mark all mbufs above mbuf_thresh as M_DOFREE,
 * which tells m_free to actually free() it
 */
struct mbuf *
m_get()
{
	register struct mbuf *m;
	int flags = 0;
	
#ifdef DEBUG
	debug_call(dfd,"m_get() called ...\n");
	debug_call(dfd,"    mbuf_alloced = %d\n", mbuf_alloced);
	fflush_call(dfd);
#endif
	
	if (m_freelist.m_next == &m_freelist) {
		m = (struct mbuf *)malloc(msize);
		if (m == NULL)
			return m;
		mbuf_alloced++;
		if (mbuf_alloced > mbuf_thresh)
			flags = M_DOFREE;
		if (mbuf_alloced > mbuf_max)
			mbuf_max = mbuf_alloced;
	} else {
		m = m_freelist.m_next;
		remque(m);
	}
	
	/* Insert it in the used list */
	insque(m,&m_usedlist);
	m->m_flags = (flags | M_USEDLIST);
	
	/* Initialise it */
	m->m_size = msize - sizeof(struct m_hdr);
	m->m_data = m->m_dat;
	m->m_len = 0;
	m->m_nextpkt = 0;
	m->m_prevpkt = 0;
	return m;
}

void
m_free(m)
	struct mbuf *m;
{
	
#ifdef DEBUG
	debug_call(dfd,"m_free(m) called ...\n");
	debug_call(dfd,"    m = %d\n", (int)m);
	fflush_call(dfd);
#endif
	/* Remove from m_usedlist */
	if (m->m_flags & M_USEDLIST)
	   remque(m);
	
	/* If it's M_EXT, free() it */
	if (m->m_flags & M_EXT)
	   free(m->m_ext);
	
	/*
	 * Either free() it or put it on the free list
	 */
	if (m->m_flags & M_DOFREE) {
		free(m);
		mbuf_alloced--;
	} else if ((m->m_flags & M_FREELIST) == 0) {
		insque(m,&m_freelist);
		m->m_flags = M_FREELIST; /* Clobber other flags */
	}
	
}

/*
 * Copy data from one mbuf to the end of
 * the other.. if result is too big for one mbuf, malloc()
 * an M_EXT data segment
 */
void
m_cat(m, n)
	register struct mbuf *m, *n;
{

#ifdef DEBUG
	debug_call(dfd,"m_cat(m,n)	called ...\n");
	debug_call(dfd,"    m = %d\n    n = %d\n", (int)m, (int)n);
	fflush_call(dfd);
#endif
	
	/*
	 * If there's no room, realloc
	 */
	if (M_FREEROOM(m) < n->m_len)
		m_inc(m,MINCSIZE);
	
	memcpy(m->m_data+m->m_len, n->m_data, n->m_len);
	m->m_len += n->m_len;

	m_free(n);
}


void
m_inc(m, size)
        struct mbuf *m;
        int size;
{
        char *dat;
        int datasize;
 
#ifdef DEBUG
        debug_call(dfd,"m_inc(m,size) called ...\n");
        debug_call(dfd,"    m = %d\n    size = %d\n", (int)m, size);
        fflush_call(dfd);
#endif
 
        if (m->m_flags & M_EXT) {
                datasize = m->m_data - m->m_ext;
                m->m_ext = (char *)realloc(m->m_ext,size);
/*		if (m->m_ext == NULL)
 *			return (struct mbuf *)NULL;
 */		
		m->m_data = m->m_ext + datasize;
        } else {
                datasize = m->m_data - m->m_dat;
                dat = (char *)malloc(size);
/*		if (dat == NULL)
 *			return (struct mbuf *)NULL;
 */
                memcpy(dat, m->m_dat, m->m_size);
 
                m->m_ext = dat;
                m->m_data = m->m_ext + datasize;
                m->m_flags |= M_EXT;
        }
 
        m->m_size = size;
}



void
m_adj(m, len)
	struct mbuf *m;
	int len;
{
	
#ifdef DEBUG
	debug_call(dfd,"m_adj(m, len) called ...\n");
	debug_call(dfd,"    m = %d\n    len = %d\n", (int)m, len);
	fflush_call(dfd);
#endif
	
	if (m == NULL)
		return;
	if (len >= 0) {
		/* Trim from head */
		m->m_data += len;
		m->m_len -= len;
	} else {
		/* Trim from tail */
		len = -len;
		m->m_len -= len;
	}
}


/*
 * Copy len bytes from m, starting off bytes into n
 */
int
m_copy(n, m, off, len)
	struct mbuf *n, *m;
	int off, len;
{

#ifdef DEBUG
	debug_call(dfd,"m_copy(n, m, off, len) called ...\n");
	debug_call(dfd,"    n = %d\n    m = %d\n    off = %d\n    len = %d\n",
		(int)n, (int)m, off, len);
	fflush_call(dfd);
#endif

	if (len > M_FREEROOM(n))
		return -1;

	memcpy((n->m_data + n->m_len), (m->m_data + off), len);
	n->m_len += len;
	return 0;
}


/*
 * Given a pointer into an mbuf, return the mbuf
 * XXX This is a kludge, I should eliminate the need for it
 */
struct mbuf *
dtom(dat)
	void *dat;
{
	struct mbuf *m;
	
#ifdef DEBUG
	debug_call(dfd,"dtom(dat) called ...\n");
	debug_call(dfd,"    dat = %d\n", (int)dat);
	fflush_call(dfd);
#endif

	for (m = m_usedlist.m_next; m != &m_usedlist; m = m->m_next) {
		if ((char *)dat >= m->m_dat &&
		    (char *)dat < (m->m_dat + m->m_size))
			return m;
		if (m->m_flags & M_EXT) {
			if ((char *)dat >= m->m_ext &&
	                    (char *)dat < (m->m_ext + m->m_size))
        	                return m;
		}
	}
#ifdef DEBUG
	debug_error(dfd,"PANIC: couldn't dtom!\n");
#endif
	return (struct mbuf *)0;
}

