/* Hypertext file browser.
   Copyright (C) 1994 Miguel de Icaza.
   
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

   Implements the hypertext file viewer.
   The hypertext file is a file that may have one or more nodes.  Each
   node ends with a ^D character and starts with a bracket, then the
   name of the node and then a closing bracket.

   Links in the hypertext file are specified like this: the text that
   will be highlighted should have a leading ^A, then it comes the
   text, then a ^B indicating that highlighting is done, then the name
   of the node you want to link to and then a ^C.

   The file must contain a ^D at the beginning and at the end of the
   file or the program will not be able to detect the end of file.

*/
#include <ncurses.h>
#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <malloc.h>
#include <errno.h>
#include "mad.h"
#include "color.h"
#include "dialog.h"
#include "util.h"
#include "input.h"
#include "win.h"
#include "global.h"
#include "mouse.h"
#include "key.h"	/* For mi_getch() */
#include "help.h"


#ifdef USE_BSD_CURSES
    #define ACS_MAP(x) '*'
#else
    #define ACS_MAP(x) acs_map [x]
#endif

#define MAXLINKNAME 80
#define HISTORY_SIZE 20
#define HELP_WINDOW_WIDTH 60

static char rcsid [] = "$Id: help.c,v 1.14 1995/01/27 02:35:04 miguel Exp $";
static char *data;		/* Pointer to the loaded data file */
static int help_lines = 18;	/* Lines in help viewer */
WINDOW *whelp;			/* Help Window */
static int  history_ptr;	/* For the history queue */
static char *main;		/* The main node */
static char *last_shown = 0;	/* Last byte shown in a screen */
static int end_of_node = 0;	/* Flag: the last character of the node shown? */
static int quit;		/* Flag: quit? */
static char *current, *start, *selected_item;

static struct {
    char *page;			/* Pointer to the selected page */
    char *link;			/* Pointer to the selected link */
} history [HISTORY_SIZE];

/* Link areas for the mouse */
typedef struct Link_Area {
    int x1, y1, x2, y2;
    char *link_name;
    struct Link_Area *next;
} Link_Area;

static Link_Area *link_area = NULL;
static int inside_link_area = 0;

/* returns the position where text was found in the start buffer */
/* or 0 if not found */
char *search_string (char *start, char *text)
{
    char *d = text;
    char *e = start;

    /* fmt sometimes replaces a space with a newline in the help file */
    /* Replace the newlines in the link name with spaces to correct the situation */
    while (*d){
	if (*d == '\n')
	    *d = ' ';
	d++;
    }
    /* Do search */
    for (d = text; *e; e++){
	if (*d == *e)
	    d++;
	else
	    d = text;
	if (!*d)
	    return e+1;
    }
    return 0;
}

/* Searches text in the buffer pointed by start.  Search ends */
/* if the CHAR_NODE_END is found in the text.  Returns 0 on failure */
static char *search_string_node (char *start, char *text)
{
    char *d = text;
    char *e = start;

    if (!start)
	return 0;
    
    for (; *e && *e != CHAR_NODE_END; e++){
	if (*d == *e)
	    d++;
	else
	    d = text;
	if (!*d)
	    return e+1;
    }
    return 0;
}

/* Searches the_char in the buffer pointer by start and searches */
/* it can search forward (direction = 1) or backward (direction = -1) */
static char *search_char_node (char *start, char the_char, int direction)
{
    char *e;

    e = start;
    
    for (; *e && (*e != CHAR_NODE_END); e += direction){
	if (*e == the_char)
	    return e;
    }
    return 0;
}

/* Returns the new current pointer when moved lines lines */
static char *move_forward2 (char *current, int lines)
{
    char *p;
    int  line;

    for (line = 0, p = current; *p && *p != CHAR_NODE_END; p++){
	if (line == lines)
	    return p;
	if (*p == '\n')
	    line++;
    }
    return current;
}

static char *move_backward2 (char *current, int lines)
{
    char *p;
    int line;

    for (line = 0, p = current; *p && p >= data; p--){
	if (*p == CHAR_NODE_END)
	{
	    /* We reached the beginning of the node */
	    /* Skip the node headers */
	    while (*p != ']') p++;
	    return p + 2;
	}
	if (*(p - 1) == '\n')
	    line++;
	if (line == lines)
	    return p;
    }
    return current;
}

static void move_forward (int i)
{
    if (end_of_node)
	return;
    current = move_forward2 (current, i);
}

static void move_backward (int i)
{
    current = move_backward2 (current, ++i);
}

static void move_to_top (int dummy)
{
    while (current > data && *current != CHAR_NODE_END)
	current--;
    while (*current != ']')
	current++;
    current = current + 1;
    selected_item = NULL;
}

static void move_to_bottom (int dummy)
{
    while (*current && *current != CHAR_NODE_END)
	current++;
    current--;
    move_backward (help_lines - 1);
}

static char *follow_link (char *start, char *selected_item)
{
    char link_name [MAXLINKNAME];
    char *p;
    int  i = 0;

    if (!selected_item)
	return start;
    
    for (p = selected_item; *p && *p != CHAR_NODE_END && *p != CHAR_LINK_POINTER; p++)
	;
    if (*p == CHAR_LINK_POINTER){
	link_name [0] = '[';
	for (i = 1; *p != CHAR_LINK_END && *p && *p != CHAR_NODE_END && i < MAXLINKNAME-3; )
	    link_name [i++] = *++p;
	link_name [i-1] = ']';
	link_name [i] = 0;
	p = search_string (data, link_name);
	if (p)
	    return p;
    }
    return " Help file format error\n\x4";	/*  */
}

static char *select_next_link (char *start, char *current_link)
{
    char *p;

    if (!current_link)
	return 0;

    p = search_string_node (current_link, STRING_LINK_END);
    if (!p)
	return NULL;
    p = search_string_node (p, STRING_LINK_START);
    if (!p)
	return NULL;
    return p - 1;
}

static char *select_prev_link (char *start, char *current_link)
{
    char *p;

    if (!current_link)
	return 0;
    
    p = current_link - 1;
    if (p <= start)
	return 0;
    
    p = search_char_node (p, CHAR_LINK_START, -1);
    return p;
}

static void start_link_area (int x, int y, char *link_name)
{
    Link_Area *new;

    if (inside_link_area)
	message (0, " Warning ", " Internal bug: Double start of link area ");

    /* Allocate memory for a new link area */
    new = (Link_Area*) xmalloc (sizeof (Link_Area), "Help, link_area");
    new->next = link_area;
    link_area = new;

    /* Save the beginning coordinates of the link area */
    link_area->x1 = x;
    link_area->y1 = y;

    /* Save the name of the destination anchor */
    link_area->link_name = link_name;

    inside_link_area = 1;
}

static void end_link_area (int x, int y)
{
    if (inside_link_area){
	/* Save the end coordinates of the link area */
	link_area->x2 = x;
	link_area->y2 = y;

	inside_link_area = 0;
    }
}

static void clear_link_areas (void)
{
    Link_Area *current;

    while (link_area){
	current = link_area;
	link_area = current -> next;
	free (current);
    }
    inside_link_area = 0;
}

static void show (char *paint_start)
{
    char *p;
    int  col, line, c;
    int  painting = 1;
    int acs = 0;	/* Flag: Alternate character set active? */
    
    line = col = 0;
    wclrn (whelp, 2);
    
    clear_link_areas ();
    if (selected_item < paint_start)
	selected_item = NULL;
    
    for (p = paint_start; *p != CHAR_NODE_END && line < help_lines; p++){
	c = *p;
	switch (c){
	case CHAR_LINK_START:
	    if (selected_item == NULL)
		selected_item = p;
	    if (p == selected_item)
		wattrset (whelp, MARKED_COLOR | A_BOLD);
	    else
		wattrset (whelp, INPUT_COLOR);
	    start_link_area (col, line, p);
	    break;
	case CHAR_LINK_POINTER:
	    painting = 0;
	    end_link_area (col - 1, line);
	    break;
	case CHAR_LINK_END:
	    painting = 1;
	    wattrset (whelp, REVERSE_COLOR);
	    break;
	case CHAR_ALTERNATE:
	    acs = 1;
	    break;
	case CHAR_NORMAL:
	    acs = 0;
	    break;
	case CHAR_VERSION:
	    mvwaddstr (whelp, line+2, col+2, VERSION);
	    col += strlen (VERSION);
	    break;
	case '\n':
	    line++;
	    col = 0;
	    break;
	case '\t':
	    col = (col/8 + 1) * 8;
	    break;
	default:
	    if (!painting)
		continue;
	    if (col > HELP_WINDOW_WIDTH-1)
		continue;
	    
	    wmove (whelp, line+2, col+2);
	    if (acs){
		if (c == ' ' || c == '.')
		    waddch (whelp, c);
		else
		    waddch (whelp, ACS_MAP(c));
	    } else
		waddch (whelp, c);
	    col++;
	    break;
	}
    }
    last_shown = p;
    end_of_node = line < help_lines;
    wattrset (whelp, REVERSE_COLOR);
    if (selected_item >= last_shown){
	if (link_area != NULL){
	    selected_item = link_area->link_name;
	    show (paint_start);
	}
	else
	    selected_item = NULL;
    }
    wrefresh (whelp);
}

static int help_event (Gpm_Event *event)
{
    int hp;
    Link_Area *current_area;

    if (! (event->type & GPM_UP))
	return 0;

    event->y--;
    if (event->buttons & GPM_B_RIGHT){
	int hp;

	hp = --history_ptr % HISTORY_SIZE;
	current = start = history [hp].page;
	selected_item = history [hp].link;
	show (current);
	return 0;
    }

    /* Test whether the mouse click is inside one of the link areas */
    current_area = link_area;
    while (current_area)
    {
	/* Test one line link area */
	if (event->y == current_area->y1 && event->x >= current_area->x1 &&
	    event->y == current_area->y2 && event->x <= current_area->x2)
	    break;
	/* Test two line link area */
	if (current_area->y1 + 1 == current_area->y2){
	    /* The first line */
	    if (event->y == current_area->y1 && event->x >= current_area->x1)
		break;
	    /* The second line */
	    if (event->y == current_area->y2 && event->x <= current_area->x2)
		break;
	}
	/* Mouse will not work with link areas of more than two lines */

	current_area = current_area -> next;
    }

    /* Test whether a link area was found */
    if (current_area){
	/* The click was inside a link area -> follow the link */
	hp = history_ptr++ % HISTORY_SIZE;
	history [hp].page = current;
	history [hp].link = current_area->link_name;
	current = start = follow_link (current, current_area->link_name);
	selected_item = NULL;
    } else{
	if (event->y < 0)
	    move_backward (help_lines - 1);
	else if (event->y >= help_lines)
	    move_forward (help_lines - 1);
	else if (event->y < help_lines/2)
	    move_backward (1);
	else
	    move_forward (1);
    }

    /* Show the new node */
    show (current);

    return 0;
}

static int help_cmd (void)
{
    /* show help */
    {
	int hp;

	hp = history_ptr++ % HISTORY_SIZE;
	history [hp].page = current;
	history [hp].link = selected_item;
	current = start = search_string (data, "[Help]") + 1;
    }
    selected_item = NULL;
    return 1;
}

static int index_cmd (void)
{
    char *new_item;

    {
	int hp;

	hp = history_ptr++ % HISTORY_SIZE;
	history [hp].page = current;
	history [hp].link = selected_item;
	current = start = search_string (data, "[Help]") + 1;
    }
    if (!(new_item = search_string (data, "[Contents]")))
	message (1, " Error ", " Can't find node [Contents] in help file ");
    else
	current = start = new_item + 1;
    selected_item = NULL;
    return 1;
}

static int quit_cmd (void)
{
    return quit = 1;
}

static int prev_node_cmd (void)
{
    int hp;

    hp = --history_ptr % HISTORY_SIZE;
    current = start = history [hp].page;
    selected_item = history [hp].link;
    return 1;
}

void interactive_display (char *filename, char *node)
{
    WINDOW *fkeys;
    int c;
    char *new_item;

    quit = 0;
    if ((data = load_file (filename)) == 0){
	message (1, " Error ", " Can't open file %s \n %s ",
		 filename, unix_error_string (errno));
	return;
    }
    if (!(main = search_string (data, node))){
	message (1, " Error ", " Can't find node %s in help file ", node);
	return;
    }

    if (help_lines > LINES - 4)
	help_lines = LINES - 4;
    create_dialog (HELP_WINDOW_WIDTH, help_lines, " Help ", "", 0);
    whelp = get_top_text ();
    
    selected_item = search_string_node (main, STRING_LINK_START) - 1;
    current = start = main + 1;
    
    for (history_ptr = 0; history_ptr < HISTORY_SIZE; history_ptr++){
	history [history_ptr].page = current;
	history [history_ptr].link = selected_item;
    }

    push_event (3, 2, 60 + 2, help_lines + 3, (mouse_h) help_event,
		0, event_use_frame);

    fkeys = push_fkey (1);
    define_label_quit (fkeys, 1, "Help", help_cmd);
    define_label_quit (fkeys, 2, "Index", index_cmd);
    define_label_quit (fkeys, 3, "Prev", prev_node_cmd);
    define_label (fkeys, 4, "", 0);
    define_label (fkeys, 5, "", 0);
    define_label (fkeys, 6, "", 0);
    define_label (fkeys, 7, "", 0);
    define_label (fkeys, 8, "", 0);
    define_label (fkeys, 9, "", 0);
    define_label_quit (fkeys, 10, "Quit", quit_cmd);

    do {
	show (current);
	c = mi_getch ();
	if (check_fkeys (c))
	    /* Nothing */;
	else if (c != KEY_UP && c != KEY_DOWN &&
	    check_movement_keys (c, 1, help_lines, move_backward, move_forward,
				  move_to_top, move_to_bottom))
	    /* Nothing */;
	else switch (c){

	case 'l':
	case KEY_LEFT:
	    prev_node_cmd ();
	    break;
	    
	case '\n':
	case KEY_RIGHT:
	    /* follow link */
	    if (!selected_item){
		/* If there are no links, go backward in history */
		int hp = --history_ptr % HISTORY_SIZE;
		
		current = start = history [hp].page;
		selected_item   = history [hp].link;
	    } else {
		int hp;

		hp = history_ptr++ % HISTORY_SIZE;
		history [hp].page = current;
		history [hp].link = selected_item;
		current = start = follow_link (current, selected_item) + 1;
	    }
	    selected_item = NULL;
	    break;

	case KEY_DOWN:
	case '\t':
	    /* select next link */
	    new_item = select_next_link (start, selected_item);
	    if (new_item){
		selected_item = new_item;
		if (selected_item >= last_shown){
		    if (c == KEY_DOWN)
			move_forward (1);
		    else
			selected_item = NULL;
		}
	    } else if (c == KEY_DOWN)
		move_forward (1);
	    else
		selected_item = NULL;
	    break;

	case KEY_UP:
	case ALT ('\t'):
	    /* select previous link */
	    new_item = select_prev_link (start, selected_item);
	    selected_item = new_item;
	    if (selected_item < current || selected_item >= last_shown){
		if (c == KEY_UP)
		    move_backward (1);
		else{
		    if (link_area != NULL)
			selected_item = link_area->link_name;
		    else
			selected_item = NULL;
		}
	    }
	    break;

	case 'n':
	    /* Next node */
	    new_item = current;
	    while (*new_item && *new_item != CHAR_NODE_END)
		new_item++;
	    if (*++new_item == '['){
		while (*new_item != ']')
		    new_item++;
		current = new_item + 2;
		selected_item = NULL;
	    }
	    break;

	case 'p':
	    /* Previous node */
	    new_item = current;
	    while (new_item > data + 1 && *new_item != CHAR_NODE_END)
		new_item--;
	    new_item--;
	    while (new_item > data && *new_item != CHAR_NODE_END)
		new_item--;
	    while (*new_item != ']')
		new_item++;
	    current = new_item + 2;
	    selected_item = NULL;
	    break;

	case 'c':
	    index_cmd ();
	    break;

	case ESC_CHAR:
	    /* quit */
	    quit = 1;
	}
    } while (!quit);
    clear_link_areas ();
    pop_fkey (fkeys);
    pop_event ();
    free (data);
    destroy_dialog ();
    do_refresh ();
}
