/*
 * Get kernel symbol table(s).
 *
 * Bjorn Ekwall <bj0rn@blox.se> in 1994 (C)
 * See the file COPYING for your rights (GNU GPL)
 */

#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <sys/types.h>
#include <linux/unistd.h>
#include <linux/module.h>

/*
 * This is here as syscall.h and sys/syscall.h redefine the defines in
 * unistd.h why doesn't unistd #include them?
 */

extern int syscall(int, ...);

static int get_kernel_syms(struct kernel_sym *buffer)
{
	return syscall( __NR_get_kernel_syms, buffer);
}

static void *
ckalloc(size_t nbytes)
{
	void *p;

	if ((p = malloc(nbytes)) == NULL) {
		fputs("ksyms:  malloc failed\n", stderr);
		exit(2);
	}
	return p;
}

int
main(int argc, char **argv)
{
	struct kernel_sym *ksymtab = NULL;
	struct kernel_sym *ksym = NULL;
	int nksyms;
	int i;
	int allsyms = 0;
	int show_header = 1;
	char *p;
	char *module_name = "";

	while (argc > 1 && (argv[1][0] == '-')) {
		p = &(argv[1][1]);
		while (*p) {
			switch (*p) {
			case 'a':
				allsyms = 1;
				break;

			case 'h':
				show_header = 0;
				break;
			}
			++p;
		}
		--argc;
		++argv;
	}

	if (argc < 1) {
		fputs("Usage: ksyms [-a] [-h]\n", stderr);
		exit(2);
	}

	/* get the size of the current kernel symbol table */
	nksyms = get_kernel_syms(NULL);

	if (nksyms < 0) {
		fprintf(stderr, "get_kernel_sys failed: Cannot find Kernel symbols!\n");
		exit(2);
	}

	if (nksyms) {
		ksymtab = (struct kernel_sym *) ckalloc(nksyms * sizeof *ksymtab);
		/* NOTE!!! The order of the symbols is important */
		if (get_kernel_syms(ksymtab) != nksyms) {
			fprintf(stderr, "Kernel symbol problem\n");
			exit(2);
		}
	}

	if (show_header)
		printf("Address  Symbol    \tDefined by\n");

	for (ksym = ksymtab, i = 0 ; i < nksyms ; ++i, ksym++) {
		/* Magic in this version of the new get_kernel_syms:
		 * Every module is sent along as a symbol,
		 * where the module name is represented as "#module", and
		 * the adress of the module struct is stuffed into the value.
		 * The name "#" means that the symbols that follow are
		 * kernel resident.
		 */
		if (ksym->name[0] == '#') {
			module_name = ksym->name + 1;
			if (!allsyms && (*module_name == '\0'))
				break;
			else
				continue;
		}

		printf("%08lx %s", ksym->value, ksym->name);
		if (*module_name)
			printf("\t[%s]", module_name);
		printf("\n");
	}

	return 0;
}
