/* #module    GloTeX    "2-001"
 ***********************************************************************
 *                                                                     *
 * The software was developed at the Monsanto Company and is provided  *
 * "as-is".  Monsanto Company and the auther disclaim all warranties   *
 * on the software, including without limitation, all implied warran-  *
 * ties of merchantabilitiy and fitness.                               *
 *                                                                     *
 * This software does not contain any technical data or information    *
 * that is proprietary in nature.  It may be copied, modified, and     *
 * distributed on a non-profit basis and with the inclusion of this    *
 * notice.                                                             *
 *                                                                     *
 ***********************************************************************
 */
/*
 * Module Name:	GloTeX
 *
 * Author:	R L Aurbach	CR&DS MIS Group    17-Aug-1986
 *
 * Function:
 *	Main Line for the GloTeX program.  This program processes .glo files
 *	produced by LaTeX and generates files which will produce a properly
 *	formatted glossary, using definitions extracted from one or more
 *	glossary definition files.
 *
 * Modification History:
 *
 * Version     Initials	   Date		Description
 * ------------------------------------------------------------------------
 * 1-001	RLA	17-Aug-1986	Original Code
 * 1-001	RLA	25-Aug-1986	Improve handling of error when there
 *					  is no gdf file to process.
 * 2-001	F.H.	17-May-1991	converted to portable C
 */
/*
 * Module GloTeX - Module-Wide Data Description Section
 *
 * Include Files:
 */
#ifdef MSDOS
#include <stdlib.h>
#include <io.h>
#define F_OK		0	/* access(): File exists */
#else
#include <sys/file.h>
#endif
#include <stdio.h>
#include "GloDef.h"
/*
 * Module Definitions:
 */
/*
 * Global Declarations:
 */
#ifdef MSDOS
main(int argc, char *argv[]);
#else
main();
#endif
    STRING_PTR	    filelist = 0;   /* Listhead for glossary filespecs	*/
    NODE_PTR	    root = 0;	    /* Listhead for definitions		*/
    int		    flag = 0;	    /* /TOC options flag		*/
    char	    infile[256];    /* Input filespec string		*/
    STRING_PTR	    labels = 0;	    /* Listhead for labels		*/
/*
 * Static Declarations:
 */
/*
 * External References:
 */
#ifdef MSDOS
extern void Glo_Command(int argc, char *argv[]);
extern int Glo_Build_LabLst(void);
extern int Glo_Scan_File(STRING_PTR file);
extern int Glo_Report_Missing(void);
extern int Glo_Build_File(void);
#else
extern void Glo_Command();
extern int Glo_Build_LabLst();
extern int Glo_Scan_File();
extern int Glo_Report_Missing();
extern int Glo_Build_File();
#endif
/*
 * Functions Called:
 */
/*
 * Function GloTeX - Documentation Section
 *
 * Discussion:
 *	This is the main line of the GloTeX program.  
 *
 *	It reads the input file and creates from it a list of items to be 
 *	defined.  
 *
 *	It then reads one or more glossary definition files, searching for 
 *	definitions which match the items specified.  If it finds an item, it 
 *	copies the definition into an internal data structure and removes the 
 *	item from the list.  In this way, the first definition seen for an 
 *	item is always used.
 *
 *	Once the list of definition files is exhausted, the internal data
 *	structures hold a complete list of all definitions.  Any items remaining
 *	in the item list represent items for which no definitions were found.
 *	These are reported.
 *
 *	Finally, the internal data structures are used to build a text file
 *	which will be processed by LaTeX to generate the glossary.
 *
 * Calling Synopsis:
 *	$ GloTeX :== $Crl_Public:GloTeX
 *	$ GloTeX filespec [/STYLE:{ARTICLE | REPORT | SPECIAL}] 
 *			    [/GLOSSARY=(file1,file2,...)]
 *
 * Inputs:
 *	filespec    ->	Name of the input file to be processed.  The default
 *			filespec SYS$DISK:[].GLO is processed against this
 *			specification.  This is the file generated by LaTeX
 *			by the \makeglossary command.
 *
 *	/STYLE	    ->  The /STYLE qualifier indicates the type of document for
 *			which the glossary is to be generated.  Options are:
 *			    * ARTICLE - a standard LaTeX "article" documentstyle
 *			    * REPORT  - a standard LaTeX "report" or "book"
 *					documentstyle
 *			    * SPECIAL - any of the CR&DS special documentstyles
 *			If the qualifier is not specified, SPECIAL is assumed.
 *
 *	/GLOSSARY   ->	The /GLOSSARY qualifier defines one or more glossary 
 *			definition files.  Each value in the list is the file
 *			specification of a glossary definition file.  These
 *			files are the basis for the list of glossary files
 *			managed by the program.  
 *
 *			If using a document style which does not define the 
 *			\glossaryfile command (i.e., a standard LaTeX document
 *			style), then this qualifier is the only means 
 *			available to specify the files to be searched for 
 *			definitions.  
 *
 *			If using a CR&DS document style, then this qualifier 
 *			is used to specify definition files to be searched 
 *			before any specified by the \glossaryfile command.  
 *
 *			The qualifier is optional.  However, at least one 
 *			definition file must be specified to the program.
 *
 * Outputs:
 *	filespec    ->	The program produces an output file which is the 
 *			resultant filespec from processing .GLS as the primary
 *			file specification and the fully-qualified input 
 *			filespec as the default.
 *
 * Return Value:
 *	returns SUCCESS
 *
 * Global Data:
 *	All commonly-used data structures, such as the file list, the label 
 *	list, the node list, etc., are declared globally.
 *
 * Files Used:
 *	reads the specified input file and one or more glossary definition files
 *	as described above.  Produces an output file.
 *
 * Assumed Entry State:
 *	Called from DCL level
 *
 * Normal Exit State:
 *	Returns to DCL level
 *
 * Error Conditions:
 *	Error messages are generated for all errors.  Any significant ones
 *	cause the program to exit with an appropriate message.
 *
 * Algorithm:
 *	A. Process the command line.
 *	B. Process the input file.
 *	    1. Glossary label entries are placed on a list.
 *	    2. Glossary file entries are added to the file list.
 *	C. For each file in the file list,
 *	    1. Scan the glossary file for definitions which match a label in
 *	       the label list.
 *	    2. If found,
 *		a. Copy the definition to a node in the data structure.
 *	D. For any labels for which the definition was not found,
 *	    1. Report it.
 *	E. Generate the output file.
 *
 * Special Notes:
 *	none
 */
/*
 * Function GloTeX - Code Section
 */
main(argc,argv)
int argc;
char *argv[];
{
/*
 * Local Declarations
 */
  int		    status;
  STRING_PTR	    file;
/*
 * Module Body
 */
/* Process the command line.  The routine will exit on error */
  Glo_Command(argc, argv);
  (void)printf("\nGloTeX, the automatic glossary processor for LaTeX, version 1.0\n");
  if (argc < 2) {
    (void) printf("usage: glotex file [/STYLE={Article | Report | Special}] \\ \n");
    (void) printf("                   [/GLOSSARY=(def1[,def2[,...]])]\n");
    exit(1);
  }  
/*
 * Process the list of definitions referenced:
 *  Open the input file.
 *  For each entry record in the file,
 *      Build a label entry in the label list.
 *  For each file record in the file,
 *	Build a file entry in the file list.
 *  Close the input file.
 */
  status = Glo_Build_LabLst();
  if (!(status & TRUE)) {
    (void)printf("\nFatal error processing the input file...aborting\n");
    exit(1);
  }
  if (labels == 0) {
    (void)printf("\nFatal error.  No glossary entries found in the .glo file");
    (void)printf("...aborting\n");
    exit(1);
  }
/*
 * For each file in the file list,
 *  Scan the file for entries which match labels with label list entries.
 *  For each one, build a node containing the definition.
 */
  if (filelist == 0) {
    (void)Glo_Report_Missing();
    (void)printf("\nNo glossary definition files specified...aborting\n");
    exit(1);
  }
  file = filelist;
  while (file != 0) {
    status = Glo_Scan_File(file);
    if (!(status & TRUE)) {
      (void)printf("\nFatal error scanning glossary definition file...aborting\n");
      exit(1);
    }
    file = file->next;
  }
/*
 * Report all labels for which no definitions were found.
 *  Write the information to SYS$OUTPUT.
 *  Write the information to a glossary log file.
 */
  status = Glo_Report_Missing();
  if (!(status & TRUE)) {
    (void)printf("\nFatal error while reporting the list of undefined terms");
    (void)printf("...aborting\n");
    exit(1);
  }
/*
 * Generate a formatted output file containing the definitions.
 * The file format will be dependent on whether the glossary is being built
 *  for a LaTeX "article", a LaTeX "report" or "book", or a CR&DS defined
 *  documentstyle.
 */
  status = Glo_Build_File();
  if (!(status & TRUE))
    (void)printf("\nFatal error while building the output file\n");
  (void)printf("\nDone.\n");
}
