------------------------------------------------------------------------------
--                                                                          --
--                         GNAT COMPILER COMPONENTS                         --
--                                                                          --
--                              S E M _ R E S                               --
--                                                                          --
--                                 B o d y                                  --
--                                                                          --
--                            $Revision: 1.195 $                            --
--                                                                          --
--           Copyright (c) 1992,1993,1994 NYU, All Rights Reserved          --
--                                                                          --
-- GNAT is free software;  you can  redistribute it  and/or modify it under --
-- terms of the  GNU General Public License as published  by the Free Soft- --
-- ware  Foundation;  either version 2,  or (at your option) any later ver- --
-- sion.  GNAT is distributed in the hope that it will be useful, but WITH- --
-- OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
-- or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License --
-- for  more details.  You should have  received  a copy of the GNU General --
-- Public License  distributed with GNAT;  see file COPYING.  If not, write --
-- to the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA. --
--                                                                          --
------------------------------------------------------------------------------

with Atree;    use Atree;
with Debug;    use Debug;
with Debug_A;  use Debug_A;
with Einfo;    use Einfo;
with Elists;   use Elists;
with Errout;   use Errout;
with Expand;
with Exp_Util; use Exp_Util;
with Features; use Features;
with Itypes;   use Itypes;
with Lib;      use Lib;
with Namet;    use Namet;
with Nmake;    use Nmake;
with Nlists;   use Nlists;
with Opt;      use Opt;
with Output;   use Output;
with Rtsfind;  use Rtsfind;
with Sem;      use Sem;
with Sem_Aggr; use Sem_Aggr;
with Sem_Attr; use Sem_Attr;
with Sem_Ch3;  use Sem_Ch3;
with Sem_Ch4;  use Sem_Ch4;
with Sem_Ch5;  use Sem_Ch5;
with Sem_Ch8;  use Sem_Ch8;
with Sem_Ch13; use Sem_Ch13;
with Sem_Disp; use Sem_Disp;
with Sem_Eval; use Sem_Eval;
with Sem_Util; use Sem_Util;
with Sem_Type; use Sem_Type;
with Sinfo;    use Sinfo;
with Snames;   use Snames;
with Stand;    use Stand;
with Stringt;  use Stringt;
with Tbuild;   use Tbuild;
with Uintp;    use Uintp;

with Treepr;   use Treepr;

package body Sem_Res is

   -----------------------
   -- Local Subprograms --
   -----------------------

   --  Second pass (top-down) type checking and overload resolution procedures
   --  Typ is the type required by context. These procedures propagate the
   --  type information recursively to the descendants of N. If the node
   --  is not overloaded, its Etype is established in the first pass. If
   --  overloaded,  the Resolve routines set the correct type. For arith.
   --  operators, the Etype is the base type of the context.

   --  Note that Resolve_Attribute is separated off in Sem_Attr

   procedure Resolve_Allocator                 (N : Node_Id; Typ : Entity_Id);
   procedure Resolve_Arithmetic_Op             (N : Node_Id; Typ : Entity_Id);
   procedure Resolve_Call                      (N : Node_Id; Typ : Entity_Id);
   procedure Resolve_Character_Literal         (N : Node_Id; Typ : Entity_Id);
   procedure Resolve_Comparison_Op             (N : Node_Id; Typ : Entity_Id);
   procedure Resolve_Conditional_Expression    (N : Node_Id; Typ : Entity_Id);
   procedure Resolve_Equality_Op               (N : Node_Id; Typ : Entity_Id);
   procedure Resolve_Explicit_Dereference      (N : Node_Id; Typ : Entity_Id);
   procedure Resolve_Expression_Actions        (N : Node_Id; Typ : Entity_Id);
   procedure Resolve_Entity_Name               (N : Node_Id; Typ : Entity_Id);
   procedure Resolve_Indexed_Component         (N : Node_Id; Typ : Entity_Id);
   procedure Resolve_Integer_Literal           (N : Node_Id; Typ : Entity_Id);
   procedure Resolve_Logical_Op                (N : Node_Id; Typ : Entity_Id);
   procedure Resolve_Membership_Op             (N : Node_Id; Typ : Entity_Id);
   procedure Resolve_Null                      (N : Node_Id; Typ : Entity_Id);
   procedure Resolve_Operator_Symbol           (N : Node_Id; Typ : Entity_Id);
   procedure Resolve_Op_Concat                 (N : Node_Id; Typ : Entity_Id);
   procedure Resolve_Op_Expon                  (N : Node_Id; Typ : Entity_Id);
   procedure Resolve_Op_Not                    (N : Node_Id; Typ : Entity_Id);
   procedure Resolve_Qualified_Expression      (N : Node_Id; Typ : Entity_Id);
   procedure Resolve_Range                     (N : Node_Id; Typ : Entity_Id);
   procedure Resolve_Real_Literal              (N : Node_Id; Typ : Entity_Id);
   procedure Resolve_Reference                 (N : Node_Id; Typ : Entity_Id);
   procedure Resolve_Selected_Component        (N : Node_Id; Typ : Entity_Id);
   procedure Resolve_Shift                     (N : Node_Id; Typ : Entity_Id);
   procedure Resolve_Short_Circuit             (N : Node_Id; Typ : Entity_Id);
   procedure Resolve_Slice                     (N : Node_Id; Typ : Entity_Id);
   procedure Resolve_String_Literal            (N : Node_Id; Typ : Entity_Id);
   procedure Resolve_Type_Conversion           (N : Node_Id; Typ : Entity_Id);
   procedure Resolve_Unary_Op                  (N : Node_Id; Typ : Entity_Id);
   procedure Resolve_Unchecked_Type_Conversion (N : Node_Id; Typ : Entity_Id);

   procedure Make_Call_Into_Operator (N : Node_Id);
   --  Inverse transformation: if an operator is given in functional notation,
   --  then after resolving the node, transform into an operator node, so
   --  that operands are resolved properly. Recall that predefined operators
   --  do not have a full signature and special resolution rules apply.

   procedure Resolve_Entry_Call (N : Node_Id);
   --  Called from Resolve_Call, when the prefix denotes an entry or element
   --  of entry family. actuals are resolved as for subprograms, and node
   --  is rebuilt as an entry call.

   procedure Rewrite_Operator_As_Call (N : Node_Id; Nam : Entity_Id);
   --  If an operator node resolves to a call to a user-defined operator,
   --  rewrite the node as a function call.

   function Valid_Conversion (N : Node_Id) return Boolean;
   --  Verify legality rules given in 4.6 (8-23)

   -----------------------------
   -- Make_Call_Into_Operator --
   -----------------------------

   procedure Make_Call_Into_Operator (N : Node_Id) is
      Op_Name   : constant Name_Id := Chars (Entity (Name (N)));
      Act1      : constant Node_Id := First_Actual (N);
      Act2      : constant Node_Id := Next_Actual (Act1);
      Is_Binary : constant Boolean := Present (Act2);
      Kind      : Node_Kind;
      Op_Node   : Node_Id;

   begin
      --  Binary operators

      if Is_Binary then
         if    Op_Name =  Name_Op_And      then Kind := N_Op_And;
         elsif Op_Name =  Name_Op_Or       then Kind := N_Op_Or;
         elsif Op_Name =  Name_Op_Xor      then Kind := N_Op_Xor;
         elsif Op_Name =  Name_Op_Eq       then Kind := N_Op_Eq;
         elsif Op_Name =  Name_Op_Ne       then Kind := N_Op_Ne;
         elsif Op_Name =  Name_Op_Lt       then Kind := N_Op_Lt;
         elsif Op_Name =  Name_Op_Le       then Kind := N_Op_Le;
         elsif Op_Name =  Name_Op_Gt       then Kind := N_Op_Gt;
         elsif Op_Name =  Name_Op_Ge       then Kind := N_Op_Ge;
         elsif Op_Name =  Name_Op_Add      then Kind := N_Op_Add;
         elsif Op_Name =  Name_Op_Subtract then Kind := N_Op_Subtract;
         elsif Op_Name =  Name_Op_Concat   then Kind := N_Op_Concat;
         elsif Op_Name =  Name_Op_Multiply then Kind := N_Op_Multiply;
         elsif Op_Name =  Name_Op_Divide   then Kind := N_Op_Divide;
         elsif Op_Name =  Name_Op_Mod      then Kind := N_Op_Mod;
         elsif Op_Name =  Name_Op_Rem      then Kind := N_Op_Rem;
         elsif Op_Name =  Name_Op_Expon    then Kind := N_Op_Expon;

         else
            pragma Assert (False); null;
         end if;

         Op_Node := New_Node (Kind, Sloc (N));
         Remove (Act1);
         Remove (Act2);
         Set_Left_Opnd  (Op_Node, Act1);
         Set_Right_Opnd (Op_Node, Act2);

      --  Unary operators

      else
         if    Op_Name =  Name_Op_Add      then Kind := N_Op_Plus;
         elsif Op_Name =  Name_Op_Subtract then Kind := N_Op_Minus;
         elsif Op_Name =  Name_Op_Abs      then Kind := N_Op_Abs;
         elsif Op_Name =  Name_Op_Not      then Kind := N_Op_Not;
         else
            pragma Assert (False); null;
         end if;

         Op_Node := New_Node (Kind, Sloc (N));
         Remove (Act1);
         Set_Right_Opnd (Op_Node, Act1);
      end if;

      Set_Chars (Op_Node, Op_Name);
      Set_Etype (Op_Node,  Etype (N));
      Rewrite_Substitute_Tree (N,  Op_Node);
   end Make_Call_Into_Operator;

   procedure Resolve (N : Node_Id; Typ : Entity_Id) is
      I         : Interp_Index;
      It        : Interp;
      Found     : Boolean := False;
      Nam       : Node_Id;
      Seen      : Entity_Id;
      Ctx_Type  : Entity_Id := Typ;
      Expr_Type : Entity_Id;
      Ambiguous : Boolean := False;

   begin
      Debug_A_Entry ("resolving  ", N);

      --  Return if already analyzed

      if Analyzed (N) then
         Debug_A_Exit ("resolving  ", N, "  (done, already analyzed)");
         return;

      --  Return if type = Any_Type (previous error encountered)

      elsif Etype (N) = Any_Type then
         Debug_A_Exit ("resolving  ", N, "  (done, Etype = Any_Type)");
         return;
      end if;

      --  First deal with a parameterless function call, where the node must
      --  be rebuilt to be a function call (just looked like a name till now)
      --  The name may be that of an overloadable construct,  or it can be
      --  an explicit dereference of a prefix that denotes an access to sub-
      --  program. In that case, we want to convert the name into a call only
      --  if the context requires the return type of the subprogram.
      --  Finally, a parameterless protected function call appears as a
      --  selected component.

      if (Is_Entity_Name (N)
            and then Is_Overloadable (Entity (N))
            and then (Ekind (Entity (N)) /= E_Enumeration_Literal
                        or else Is_Overloaded (N)))
        or else
          (Nkind (N) = N_Explicit_Dereference
            and then Ekind (Etype (N)) = E_Subprogram_Type
            and then Base_Type (Etype (Etype (N))) = Base_Type (Typ))
        or else
          (Nkind (N) = N_Selected_Component
            and then Ekind (Entity (Selector_Name (N))) = E_Function)
      then
         Nam := New_Copy (N);

         --  If overloaded, overload set belongs to new copy.

         Save_Interps (N, Nam);

         --  Change node to parameterless function call (note that the
         --  Parameter_Associations associations field is left set to Empty,
         --  its normal default value since there are no parameters)

         Change_Node (N, N_Function_Call);
         Set_Name (N, Nam);
         Set_Sloc (N, Sloc (Nam));
         Analyze_Call (N);
      end if;

      --  If not overloaded, then we know the type, and all that needs doing
      --  is to check that this type is compatible with the context.

      if not Is_Overloaded (N) then
         Found := Covers (Typ, Etype (N));
         Expr_Type := Etype (N);

      --  In the overloaded case, we must select the interpretation that
      --  is compatible with the context (i.e. the type passed to Resolve)

      else
         Get_First_Interp (N, I, It);

         --  Loop through possible interpretations

         Interp_Loop : while Present (It.Typ) loop

            --  We are only interested in interpretations that are compatible
            --  with the expected type, any other interpretations are ignored

            if Covers (Typ, It.Typ) then

               --  First matching interpretation

               if not Found then
                  Found := True;
                  Seen  := It.Nam;
                  Expr_Type := It.Typ;

               --  Matching intepretation that is not the first, maybe an
               --  error, but there are some cases where other rules are
               --  used to choose between the two possibilities

               else -- Found = True

                  --  Could be a tag-indeterminate call which resolves
                  --  statically to the operation on the root of the class.
                  --  Keep the interpretation that is closest to the root type.

                  if Is_Overloadable (It.Nam)
                    and then Is_Dispatching_Operation (It.Nam)
                    and then Root_Type (Find_Dispatching_Type (It.Nam)) =
                             Root_Type (Find_Dispatching_Type (Seen))
                  then
                     declare
                        T1 : Entity_Id := Find_Dispatching_Type (It.Nam);
                        T2 : Entity_Id := Find_Dispatching_Type (Seen);
                        R  : Entity_Id := Root_Type (T1);

                     begin
                        while T1 /= R and then T2 /= R loop
                           T1 := Etype (T1);
                           T2 := Etype (T2);
                        end loop;

                        if T1 = R then
                           Seen := It.Nam;
                        end if;
                     end;

                  --  Case of more than one interpretation. Use preference
                  --  rules, and check operator visibility and hiding.

                  else
                     Error_Msg_Sloc := Sloc (Seen);
                     Seen := Disambiguate (It.Nam, Seen, Typ);

                     if Seen = Any_Id then

                        --  Before we issue an ambiguity complaint, check for
                        --  the case of a subprogram call where at least one
                        --  of the arguments is Any_Type, and if so, suppress
                        --  the message, since it is a cascaded message.

                        if Nkind (N) = N_Function_Call
                          or else Nkind (N) = N_Procedure_Call_Statement
                        then
                           declare
                              A : Node_Id := First_Actual (N);
                              E : Node_Id;

                           begin
                              while Present (A) loop
                                 E := A;

                                 if Nkind (E) = N_Parameter_Association then
                                    E := Explicit_Actual_Parameter (E);
                                 end if;

                                 if Etype (E) = Any_Type then
                                    if Debug_Flag_V then
                                       Write_Str ("Any_Type in call");
                                       Write_Eol;
                                    end if;

                                    exit Interp_Loop;
                                 end if;

                                 A := Next_Actual (A);
                              end loop;
                           end;
                        end if;

                        --  Not that special case, so issue message using the
                        --  flag Ambiguous to control printing of the header
                        --  message only at the start of an ambiguous set.

                        if not Ambiguous then
                           Error_Msg_NE
                             ("ambiguous expression (cannot resolve&)!",
                              N, It.Nam);
                           Error_Msg_N
                             ("possible interpretation#!", N);
                           Ambiguous := True;
                        end if;

                        Error_Msg_Sloc := Sloc (It.Nam);
                        Error_Msg_N ("possible interpretation#!", N);

                     --  Case where preference rules disambiguated. The
                     --  new value of Seen has already been set, so nothing
                     --  else to do at this point.

                     else
                        null;
                     end if;
                  end if;
               end if;

               --  We have a matching interpretation, and Expr_Type is the
               --  type from this interpretation, and Seen is the entity.

               --  For an operator, just set the entity name. The type will
               --  be set by the specific operator resolution routine.

               if Nkind (N) in N_Op then
                  Set_Entity (N, Seen);

               --  For an explicit dereference, attribute reference, range
               --  or a call with a name that is an explicit dereference,
               --  there is nothing to be done at this point.

               elsif     Nkind (N) = N_Explicit_Dereference
                 or else Nkind (N) = N_Attribute_Reference
                 or else Nkind (N) = N_Range
                 or else Nkind (Name (N)) = N_Explicit_Dereference
               then
                  null;

               --  For procedure or function calls, set the type of the
               --  name, and also the entity pointer for the prefix

               elsif (Nkind (N) = N_Procedure_Call_Statement
                       or else Nkind (N) = N_Function_Call)
                 and then (Is_Entity_Name (Name (N))
                            or else Nkind (Name (N)) = N_Operator_Symbol)
               then
                  Set_Etype  (Name (N), Expr_Type);
                  Set_Entity (Name (N), Seen);

               --  For all other cases, just set the type of the Name

               else
                  Set_Etype (Name (N), Expr_Type);
               end if;

            --  Here if interpetation is incompatible with context type

            else
               if Debug_Flag_V then
                  Write_Str ("    intepretation incompatible with context");
                  Write_Eol;
               end if;
            end if;

            --  Move to next interpretation

            Get_Next_Interp (I, It);
         end loop Interp_Loop;
      end if;

      --  At this stage Found indicates whether or not an acceptable
      --  interpretation exists. If not, then we have an error, except
      --  that if the context is Any_Type as a result of some other error,
      --  then we suppress the error report.

      if not Found then
         if Typ /= Any_Type then

            --  If type we are looking for is Void, then this is the
            --  procedure call case, and the error is simply that what
            --  we gave is not a procedure name (we think of procedure
            --  calls as expressions with types internally, but the user
            --  doesn't think of them this way!

            if Typ = Standard_Void_Type then
               Error_Msg_N ("expect procedure name in procedure call", N);

            --  Otherwise we do have a subexpression with the wrong type

            else
               --  Check for the case of an allocator which uses an access
               --  type instead of the designated type. This is a common
               --  error and we specialize the message, posting an error
               --  on the operand of the allocator, complaining that we
               --  expected the designated type of the alocator.

               if Nkind (N) = N_Allocator
                 and then Ekind (Typ) in Access_Kind
                 and then Ekind (Etype (N)) in Access_Kind
                 and then Designated_Type (Etype (N)) = Typ
               then
                  Wrong_Type (Expression (N), Designated_Type (Typ));

               --  Normal case of looking for Typ, found Etype (N)

               else
                  Wrong_Type (N, Typ);
               end if;
            end if;
         end if;

         --  Make believe it was right, to avoid cascaded errors.

         Set_Etype (N, Typ);
         Debug_A_Exit ("resolving  ", N, " (done, resolution failed)");
         Set_Analyzed (N, True);
         return;

      --  Here we have an acceptable interpretation for the context

      else
         --  A user-defined operator is tranformed into a function call
         --  at this point, so that further processing knows that operators
         --  are really operators (i.e. are predefined operators)

         if Nkind (N) in N_Op
           and then Present (Entity (N))
           and then Ekind (Entity (N)) /= E_Operator
         then
            Rewrite_Operator_As_Call (N, Entity (N));
         end if;

         --  Propagate type information and normalize tree for various
         --  predefined operations. If the context only imposes a class of
         --  types, rather than a specific type, propagate the actual type
         --  downward.

         if Typ = Any_Integer or else Typ = Any_Boolean then
            Ctx_Type := Expr_Type;
         end if;

         case N_Subexpr'(Nkind (N)) is

            when N_Aggregate => Resolve_Aggregate                (N, Ctx_Type);

            when N_Allocator => Resolve_Allocator                (N, Ctx_Type);

            when N_Attribute_Reference
                             => Resolve_Attribute                (N, Ctx_Type);

            when N_Character_Literal
                             => Resolve_Character_Literal        (N, Ctx_Type);

            when N_Concat_Multiple => null;  --  ??? needs explanation

            when N_Conditional_Expression
                             => Resolve_Conditional_Expression   (N, Ctx_Type);

            when N_Expanded_Name
                             => Resolve_Entity_Name              (N, Ctx_Type);

            when N_Extension_Aggregate
                             => Resolve_Extension_Aggregate      (N, Ctx_Type);

            when N_Explicit_Dereference
                             => Resolve_Explicit_Dereference     (N, Ctx_Type);

            when N_Expression_Actions
                             => Resolve_Expression_Actions       (N, Ctx_Type);

            when N_Function_Call
                             => Resolve_Call                     (N, Ctx_Type);

            when N_Identifier
                             => Resolve_Entity_Name              (N, Ctx_Type);

            when N_Indexed_Component
                             => Resolve_Indexed_Component        (N, Ctx_Type);

            when N_Integer_Literal
                             => Resolve_Integer_Literal          (N, Ctx_Type);

            when N_Null      => Resolve_Null                     (N, Ctx_Type);

            when N_Op_And_Then | N_Op_Or_Else
                             => Resolve_Short_Circuit            (N, Ctx_Type);

            when N_Op_And | N_Op_Or | N_Op_Xor
                             => Resolve_Logical_Op               (N, Ctx_Type);

            when N_Op_In | N_Op_Not_In
                             => Resolve_Membership_Op            (N, Ctx_Type);

            when N_Op_Eq | N_Op_Ne
                             => Resolve_Equality_Op              (N, Ctx_Type);

            when N_Op_Lt | N_Op_Le | N_Op_Gt | N_Op_Ge
                             => Resolve_Comparison_Op            (N, Ctx_Type);

            when N_Op_Not    => Resolve_Op_Not                   (N, Ctx_Type);

            when N_Op_Add    | N_Op_Subtract | N_Op_Multiply |
                 N_Op_Divide | N_Op_Mod      | N_Op_Rem

                             => Resolve_Arithmetic_Op            (N, Ctx_Type);

            when N_Op_Concat => Resolve_Op_Concat                (N, Ctx_Type);

            when N_Op_Expon  => Resolve_Op_Expon                 (N, Ctx_Type);

            when N_Op_Plus | N_Op_Minus  | N_Op_Abs
                             => Resolve_Unary_Op                 (N, Ctx_Type);

            when N_Op_Shift  => Resolve_Shift                    (N, Ctx_Type);

            when N_Operator_Symbol
                             => Resolve_Operator_Symbol          (N, Ctx_Type);

            when N_Procedure_Call_Statement
                             => Resolve_Call                     (N, Ctx_Type);

            when N_Qualified_Expression
                             => Resolve_Qualified_Expression     (N, Ctx_Type);

            when N_Raise_Constraint_Error
                             => null;

            when N_Range     => Resolve_Range                    (N, Ctx_Type);

            when N_Real_Literal
                             => Resolve_Real_Literal             (N, Ctx_Type);

            when N_Reference => Resolve_Reference                (N, Ctx_Type);

            when N_Selected_Component
                             => Resolve_Selected_Component       (N, Ctx_Type);

            when N_Slice     => Resolve_Slice                    (N, Ctx_Type);

            when N_String_Literal
                             => Resolve_String_Literal           (N, Ctx_Type);

            when N_Type_Conversion
                             => Resolve_Type_Conversion          (N, Ctx_Type);

            when N_Unchecked_Type_Conversion =>
               Resolve_Unchecked_Type_Conversion                 (N, Ctx_Type);

         end case;

         --  Freeze the type of the Expression and its entity if it is a name
         --  (RM 13.14(9,10)). Note that calls that turn out to be entry calls
         --  have been converted into such, and are not subexprs any longer.

         if Nkind (N) in N_Subexpr then
            Freeze_Expression (N);
         end if;

         --  Now that the resolution of the type of the node is complete,
         --  and we did not detect an error, we can expand this node.

         Debug_A_Exit ("resolving  ", N, "  (done)");
         Expand (N);
      end if;

   end Resolve;

   -----------------------
   -- Resolve_Allocator --
   -----------------------

   procedure Resolve_Allocator (N : Node_Id; Typ : Entity_Id) is
      E : constant Node_Id := Expression (N);

   begin
      --  Replace general access with specific type

      if Ekind (Etype (N)) = E_Allocator_Type then
         Set_Etype (N, Typ);
      end if;

      if Is_Abstract (Typ) then
         Error_Msg_N ("type of allocator cannot be abstract",  N);
      end if;

      --  For qualified expression, resolve the expression using the
      --  given subtype (nothing to do for type mark, subtype indication)

      if Nkind (E) = N_Qualified_Expression then
         Resolve (Expression (E), Entity (Subtype_Mark (E)));
      end if;
   end Resolve_Allocator;

   ---------------------------
   -- Resolve_Arithmetic_Op --
   ---------------------------

   --  Used for resolving all arithmetic operators except exponentiation

   procedure Resolve_Arithmetic_Op (N : Node_Id; Typ : Entity_Id) is
      L : constant Node_Id := Left_Opnd (N);
      R : constant Node_Id := Right_Opnd (N);
      T : Entity_Id;

      B_Typ : constant Entity_Id := Base_Type (Typ);
      --  We do the resolution using the base type, because intermediate values
      --  in expressions always are of the base type, not a subtype of it.

      procedure Set_Operand_Type (N : Node_Id);
      --  Set operand type to T if universal

      procedure Set_Operand_Type (N : Node_Id) is
      begin
         if Etype (N) = Universal_Integer
           or else Etype (N) = Universal_Real
         then
            Set_Etype (N, T);
         end if;
      end Set_Operand_Type;

   --  Start of processing for Resolve_Arithmetic_Op

   begin
      --  Special-case for mixed-mode universal expressions or fixed point type
      --  operation : each argument is resolved separately.

      if (B_Typ = Universal_Real
          or else Etype (N) = Universal_Fixed
          or else (Is_Fixed_Point_Type (B_Typ)
            and then (Etype (L) = Standard_Integer
              or else Etype (R) = Standard_Integer)))
        and then (Nkind (N) = N_Op_Multiply or else Nkind (N) = N_Op_Divide)
      then
         Resolve (L, Etype (L));
         Resolve (R, Etype (R));

         if Etype (N) = Universal_Fixed then

            if B_Typ = Universal_Fixed
              and then Nkind (Parent (N)) /= N_Type_Conversion
            then
               Error_Msg_N
                 ("result type can''t be determined from context", N);
            end if;

            Set_Etype (N, B_Typ);
         end if;

      else
         Resolve (L, B_Typ);
         Resolve (R, B_Typ);

         --  If one of the arguments was resolved to a non-universal type.
         --  label the result of the operation itself with the same type.
         --  Do the same for the universal argument, if any.

         T := Intersect_Types (L, R);
         Set_Etype (N, Base_Type (T));
         Set_Operand_Type (L);
         Set_Operand_Type (R);
      end if;

      Eval_Arithmetic_Op (N);

      --  Set overflow checking bit. Much cleverer code needed here eventually
      --  and perhaps the Resolve routines should be separated for the various
      --  arithmetic operations, since they will need different processing. ???

      if Nkind (N) in N_Op then
         if not Overflow_Checks_Suppressed (Etype (N)) then
            Set_Do_Overflow_Check (N, True);
         end if;
      end if;

   end Resolve_Arithmetic_Op;

   ---------------------
   -- Resolve_Actuals --
   ---------------------

   procedure Resolve_Actuals (N : Node_Id; Nam : Entity_Id) is
      A     : Node_Id;
      F     : Entity_Id;
      A_Typ : Entity_Id;
      F_Typ : Entity_Id;

      function OK_Variable_For_Out_Formal (AV : Node_Id) return Boolean;
      pragma Inline (OK_Variable_For_Out_Formal);
      --  Used to test if AV is an acceptable formal for an OUT or IN OUT
      --  formal. Note that the Is_Variable function is not quite the right
      --  test because this is a case in which conversions whose expression
      --  is a variable (in the Is_Variable sense) with a non-tagged type
      --  target are considered view conversions and hence variables.

      function OK_Variable_For_Out_Formal (AV : Node_Id) return Boolean is
      begin
         if Is_Variable (AV) then
            return True;

         elsif Nkind (AV) = N_Identifier then
            return True;

         elsif Nkind (AV) /= N_Type_Conversion
           and then Nkind (AV) /= N_Unchecked_Type_Conversion
         then
            return False;

         else
            return Is_Variable (Expression (AV));
         end if;
      end OK_Variable_For_Out_Formal;

   --  Start of processing for Resolve_Actuals;

   begin
      A := First_Actual (N);
      F := First_Formal (Nam);

      while Present (A) loop
         if Nkind (Parent (A)) /= N_Parameter_Association
           or else Chars (Selector_Name (Parent (A))) = Chars (F)
         then
            Resolve (A, Etype (F));
            A_Typ := Etype (A);
            F_Typ := Etype (F);

            if Ekind (F) /= E_In_Parameter
              and then not OK_Variable_For_Out_Formal (A)
            then
               Error_Msg_NE ("actual for& must be a variable", A, F);
            end if;

            --  For in or in-out parameters, check non-static context and
            --  for all cases apply a possible range check. Gigi looks at
            --  the flag and uses the appropriate types. For now since one
            --  flag is used there is an optimization which might not be
            --  done in the In Out case since Gigi does not do any analysis.

            if Ekind (F) = E_In_Parameter then
               Check_Non_Static_Context (A);
               Apply_Range_Check (A, A_Typ, F_Typ);

            elsif Ekind (F) = E_Out_Parameter then
               Apply_Range_Check (A, F_Typ, A_Typ);

            else  --  E_In_Out_Parameter
               Check_Non_Static_Context (A);
               Apply_Range_Check (A, F_Typ, A_Typ);
               Apply_Range_Check (A, A_Typ, F_Typ);
            end if;

            --  Check if the actual is a controlling argument, which is
            --  illegal if the operation is not primitive (RM 3.9.2 (9))
            --  for this particular tagged type

            if Is_Class_Wide_Type (A_Typ)
              and then not Is_Class_Wide_Type (F_Typ)
            then
               if Is_Dispatching_Operation (Nam)
                 and then Scope (Nam) = Scope (F_Typ)
               then
                  Set_Is_Controlling_Actual (A);
               else
                  Error_Msg_N ("class-wide argument not allowed here!", A);
                  Error_Msg_Node_2 := F_Typ;
                  Error_Msg_NE
                    ("& is not a primitive operation of &!", A, Nam);
               end if;

            elsif Is_Access_Type (A_Typ)
              and then Is_Class_Wide_Type (Designated_Type (A_Typ))
              and then Ekind (F_Typ) = E_Anonymous_Access_Type
              and then not Is_Class_Wide_Type (Designated_Type (F_Typ))
            then
               if Is_Dispatching_Operation (Nam)
                 and then Scope (Nam) = Scope (Designated_Type (F_Typ))
               then
                  Set_Is_Controlling_Actual (A);
               else
                  Error_Msg_N
                    ("access to class-wide argument not allowed here!", A);
                  Error_Msg_Node_2 := Designated_Type (F_Typ);
                  Error_Msg_NE
                    ("& is not a primitive operation of &!", A, Nam);
               end if;
            end if;

            Eval_Actual (A);
            A := Next_Actual (A);

         else
            --  There is a default value for this formal, which will be
            --  inserted during expansion. Current actual corresponds
            --  to some subsequent formal.

            null;
         end if;

         F := Next_Formal (F);
      end loop;

   end Resolve_Actuals;

   ------------------
   -- Resolve_Call --
   ------------------

   procedure Resolve_Call (N : Node_Id; Typ : Entity_Id) is
      Loc     : constant Source_Ptr := Sloc (N);
      Subp    : constant Node_Id    := Name (N);
      Nam     : Entity_Id;
      I       : Interp_Index;
      It      : Interp;
      Norm_OK : Boolean;

      function Is_Predefined_Op (Nam : Entity_Id) return Boolean;
      --  Utility to check whether the name in the call is a predefined
      --  operator, in which case the call is made into an operator node.

      procedure Validate_Non_Static_Call;
      --  Non-static calls are not allowed during the elaboration of a
      --  preelaborated unit. A call from inside a subprogram is however
      --  always fine (RM 10.2.1(7)). This procedure validates this rule.

      function Is_Predefined_Op (Nam : Entity_Id) return Boolean is
      begin
         return Is_Intrinsic_Subprogram (Nam)
           and then Chars (Nam) in Any_Operator_Name;
      end Is_Predefined_Op;

      procedure Validate_Non_Static_Call is
      begin
         if not Inside_Subprogram_Unit (N)
           and then Inside_Preelaborated_Unit (N)
           and then Comes_From_Source (Entity (Name (N)))
         then
            Error_Msg_N
              ("?non-static call not allowed in preelaborated unit", N);
         end if;
      end Validate_Non_Static_Call;

   --  Start of processing for Resolve_Call

   begin
      --  The context imposes a unique interpretation with type Typ on
      --  a procedure or function call. Find the entity of the subprogram
      --  that yields the expected type, and propagate the corresponding
      --  formal constraints on the actuals. The caller has established
      --  that an interpretation exists, and emitted an error if not unique.

      if Ekind (Etype (Subp)) = E_Subprogram_Type then

         --  Call to an access to subprogram, dereference made explicit in
         --  Analyze_Call.

         if not Is_Overloaded (Subp) then
            Nam := Etype (Subp);

         else
            --  Find the interpretation whose type (a subprogram type)
            --  has a return type that is compatible with the context.

            Get_First_Interp (Subp,  I, It);

            while Present (It.Typ) loop

               if Covers (Typ, Etype (It.Typ)) then
                  Nam := It.Typ;
                  exit;
               end if;

               Get_Next_Interp (I, It);
            end loop;
         end if;

         if not Is_Entity_Name (Subp) then
            Resolve (Subp, Nam);
         end if;

      elsif Nkind (Subp) = N_Selected_Component
        or else Nkind (Subp) = N_Indexed_Component
        or else (Is_Entity_Name (Subp)
        and then Ekind (Entity (Subp)) = E_Entry)
      then
         Resolve_Entry_Call (N);
         Validate_Non_Static_Call;
         return;

      elsif not (Is_Type (Entity (Subp))) then

         --  Name correctly established in Resolve

         Nam := Entity (Subp);
         Set_Entity_With_Style_Check (Subp, Nam);

      else
         pragma Assert (Is_Overloaded (Subp));
         Get_First_Interp (Subp,  I, It);

         while Present (It.Typ) loop
            if Covers (Typ, It.Typ) then
               Nam := It.Nam;
               Set_Entity_With_Style_Check (Subp, Nam);
               exit;
            end if;

            Get_Next_Interp (I, It);
         end loop;

      end if;

      --  The type of the call is the type returned by the subprogram.

      if Is_Predefined_Op (Nam) then
         Set_Etype (N, Typ);

      --  if the subprogram returns an array type, and the context
      --  requires the component type of that array type, the node is
      --  really an indexing of the parameterless call. Resolve as such.

      elsif Needs_No_Actuals (Nam)
        and then Is_Array_Type (Etype (Nam))
        and then Covers (Typ, Component_Type (Etype (Nam)))
      then
         declare
            Index_Node : Node_Id;

         begin
            Validate_Non_Static_Call;
            Index_Node :=
              Make_Indexed_Component (Loc,
                Prefix =>
                  Make_Function_Call (Loc,
                    Name => New_Occurrence_Of (Nam, Loc)),
                Expressions => Parameter_Associations (N));

            Rewrite_Substitute_Tree (N, Index_Node);
            Set_Etype (Prefix (N), Etype (Nam));
            Set_Etype (N, Typ);
            Resolve_Indexed_Component (N, Typ);
            return;
         end;

      else
         Set_Etype (N, Etype (Nam));
      end if;

      --  In the case where the call is to an overloaded subprogram, Analyze
      --  calls Normalize_Actuals once per overloaded subprogram. Therefore in
      --  such a case Normalize_Actuals needs to be called once more to order
      --  the actuals correctly. Otherwise the call will have the ordering
      --  given by the last overloaded subprogram whether this is the correct
      --  one being called or not.

      if Is_Overloaded (Subp) then
         Normalize_Actuals (N, Nam, False, Norm_OK);
         pragma Assert (Norm_OK);
      end if;

      --  In any case, call is fully resolved now. Reset Overload flag, to
      --  prevent subsequent overload resolution if node is analyzed again

      Set_Is_Overloaded (Subp, False);
      Set_Is_Overloaded (N, False);

      --  If subprogram name is a predefined operator, it was given in
      --  functional notation. Replace call node with operator node, so
      --  that actuals can be resolved appropriately.

      if Is_Predefined_Op (Nam)
        or else (Chars (Nam) = Name_Op_Concat
                  and then not Comes_From_Source (Nam))
      then
         Make_Call_Into_Operator (N);
         Resolve (N, Typ);
         return;

      elsif Present (Alias (Nam))
        and then Is_Predefined_Op (Alias (Nam))
      then
         Set_Entity (Subp, Alias (Nam));
         Make_Call_Into_Operator (N);
         Resolve (N, Typ);
         return;
      end if;

      --  Propagate interpretation to actuals. Skip over those formals
      --  for which there are no matching actuals (named notation and/or
      --  default values).

      if Present (Parameter_Associations (N)) then
         Resolve_Actuals (N, Nam);

         --  Overloaded literals are rewritten as function calls, for
         --  purpose of resolution.  After resolution, we can replace
         --  the call with the literal itself.

      elsif Ekind (Nam) = E_Enumeration_Literal then
         Copy_Node (Subp, N);
         Resolve_Entity_Name (N, Typ);

         --  Avoid validation, since it is a static function call.

         return;
      end if;

      --  If the subprogram is a primitive operation, check whether or not
      --  it is a correct dispatching call.

      if Is_Overloadable (Nam) and then Is_Dispatching_Operation (Nam) then
         Check_Dispatching_Call (N);

            --  if the subprogram is abstract, check that the call has a
            --  controlling argument (i.e. is dispatching) or is disptaching on
            --  result

         if Is_Abstract (Nam)
           and then No (Controlling_Argument (N))
           and then not Is_Class_Wide_Type (Typ)
         then
            Error_Msg_N ("call to abstract subprogram must be dispatching", N);
         end if;
      end if;

      --  If we fall through we definitely have a non-static call

      Validate_Non_Static_Call;

   end Resolve_Call;

   -------------------------------
   -- Resolve_Character_Literal --
   -------------------------------

   procedure Resolve_Character_Literal (N : Node_Id; Typ : Entity_Id) is
      B_Typ : constant Entity_Id := Base_Type (Typ);
      C     : Entity_Id;

   begin
      --  Verify that the character does belong to the type of the context

      Set_Etype (N, B_Typ);
      Eval_Character_Literal (N);

      --  Wide_Character literals must always be defined, since the set of
      --  wide character literals is complete, i.e. if a character literal
      --  is accepted by the parser, then it is OK for wide character.

      if Root_Type (B_Typ) = Standard_Wide_Character then
         return;

      --  Always accept character literal for type Any_Character, which
      --  occurs in error situations and in comparisons of literals, both
      --  of which should accept all literals.

      elsif B_Typ = Any_Character then
         return;

      --  For Standard.Character or a type derived from it, check that
      --  the literal is in range

      elsif Root_Type (B_Typ) = Standard_Character then
         if In_Character_Range (Char_Literal_Value (N)) then
            return;
         end if;

      --  Otherwise we have a user defined character type, and we can use
      --  the standard visibility mechanisms to locate the referenced entity

      else
         C := Current_Entity (N);

         while Present (C) loop
            if Etype (C) = B_Typ then
               Set_Entity_With_Style_Check (N, C);
               return;
            end if;

            C := Homonym (C);
         end loop;
      end if;

      --  If we fall through, then the literal does not match any of the
      --  entries of the enumeration type. This isn't just a constraint
      --  error situation, it is an illegality (see RM 4.2).

      Error_Msg_NE ("Character is not defined for type&", N, B_Typ);

   end Resolve_Character_Literal;

   ---------------------------
   -- Resolve_Comparison_Op --
   ---------------------------

   --  Context requires a boolean type, and plays no role in resolution.
   --  Processing identical to that for equality operators.

   procedure Resolve_Comparison_Op (N : Node_Id; Typ : Entity_Id) is
      L : constant Node_Id   := Left_Opnd (N);
      R : constant Node_Id   := Right_Opnd (N);
      T : constant Entity_Id := Find_Unique_Type (L, R);

   begin
      if T /= Any_Type then
         Resolve (L, T);
         Resolve (R, T);
         Eval_Relational_Op (N);
      end if;
   end Resolve_Comparison_Op;

   ------------------------------------
   -- Resolve_Conditional_Expression --
   ------------------------------------

   procedure Resolve_Conditional_Expression (N : Node_Id; Typ : Entity_Id) is
      Condition : constant Node_Id := First (Expressions (N));
      Then_Expr : constant Node_Id := Next (Condition);
      Else_Expr : constant Node_Id := Next (Then_Expr);

   begin
      Resolve (Condition, Standard_Boolean);
      Resolve (Then_Expr, Typ);
      Resolve (Else_Expr, Typ);

      Set_Etype (N, Typ);
      Eval_Conditional_Expression (N);
   end Resolve_Conditional_Expression;

   -------------------------
   -- Resolve_Entity_Name --
   -------------------------

   --  Used to resolve identifiers and expanded names

   procedure Resolve_Entity_Name (N : Node_Id; Typ : Entity_Id) is
      E : constant Entity_Id := Entity (N);

   begin
      --  If it's a named number replace it by an appropriate literal node
      --  representing its value, and emit a constraint check if needed. The
      --  tree rewrite procedure is used so that the original reference to
      --  the named number is preserved for debugging purposes, etc.

      if Is_Named_Number (E) then
         Rewrite_Substitute_Tree (N, New_Copy (Expression (Parent (E))));
         Set_Etype (N, Typ);

         if Is_Integer_Type (Typ) then
            Eval_Integer_Literal (N);
         else
            Eval_Real_Literal (N);
         end if;

      elsif Is_Type (E) then

         --  Allow use of subtype only if it is a task type where we are
         --  currently inside the body. This will eventually be expanded
         --  into a call to Self.

         if Ekind (E) in Task_Kind
           and then In_Open_Scopes (E)
         then
            null;

         --  Any other use of a subtype is invalid

         else
            Error_Msg_N
               ("Invalid use of subtype mark in expression or call", N);
         end if;

      else
         Eval_Entity_Name (N);
      end if;
   end Resolve_Entity_Name;

   ------------------------
   -- Resolve_Entry_Call --
   ------------------------

   procedure Resolve_Entry_Call (N : Node_Id) is
      Entry_Name       : constant Node_Id := Name (N);
      Actuals          : constant List_Id := Parameter_Associations (N);
      First_Named      : constant Node_Id := First_Named_Actual (N);
      Loc              : constant Source_Ptr := Sloc (Entry_Name);
      Nam              : Entity_Id;
      New_N            : Node_Id;
      S                : Entity_Id;
      Tsk              : Entity_Id;

   begin
      --  Find name of entry being called,  and resolve prefix of name
      --  with its own type. For now we assume that the prefix cannot be
      --  overloaded and the name of the entry plays no role in the resolution.

      if Is_Entity_Name (Entry_Name)
        and then (Nkind (Entry_Name) = N_Identifier
                   or else (Nkind (Entry_Name) = N_Expanded_Name
                             and then Is_Type (Entity (Prefix (Entry_Name)))))
      then

         --  Entry call to an entry in the current task. This is legal
         --  even though the task will deadlock. Rewrite as call to
         --  current task.
         --  This can also be a call to an entry in  an enclosing task.
         --  If this is a single task, we have to retrieve its name,
         --  because the scope of the entry is the task type, not the
         --  object. If the enclosing task is a task type, the identity
         --  of the task is given by its own self variable.

         S := Scope (Entity (Entry_Name));

         for J in reverse 0 .. Scope_Stack.Last loop

            if S = Scope_Stack.Table (J).Entity then

               --  call to current task. Will be transformed into
               --  call to Self.

               exit;

            elsif Is_Task_Type (Scope_Stack.Table (J).Entity)
              and then not Comes_From_Source (S)
            then

               --  S is an enclosing task. The task declaration has
               --  been converted into a type declaration, and the
               --  task object itself has an object declaration that
               --  follows the type in the same declarative part.

               Tsk := Next_Entity (S);

               while Etype (Tsk) /= S loop
                  Tsk := Next_Entity (Tsk);
               end loop;

               S := Tsk;
               exit;
            end if;
         end loop;

         New_N :=
           Make_Selected_Component (Loc,
             Prefix => New_Occurrence_Of (S, Loc),
             Selector_Name => New_Occurrence_Of (Entity (Entry_Name), Loc));
         Rewrite_Substitute_Tree (Entry_Name, New_N);
         Analyze (Entry_Name);
      end if;

      if Nkind (Entry_Name) = N_Selected_Component then
         Nam := Entity (Selector_Name (Entry_Name));
         Resolve (Prefix (Entry_Name), Etype (Prefix (Entry_Name)));

      elsif Nkind (Entry_Name) = N_Indexed_Component then
         Nam := Entity (Selector_Name (Prefix (Entry_Name)));
         Resolve (Prefix (Prefix (Entry_Name)),
                   Etype (Prefix (Prefix (Entry_Name))));

         Resolve (First (Expressions (Entry_Name)), Entry_Index_Type (Nam));

      else
         pragma Assert (False); null;
      end if;

      Resolve_Actuals (N, Nam);

      --  After resolution,  change node to an entry call, for expansion.
      --  The structure of the node does not change,  so it can safely
      --  be done in place.

      Change_Node (N, N_Entry_Call_Statement);
      Set_Name (N, Entry_Name);
      Set_Parameter_Associations (N, Actuals);
      Set_First_Named_Actual (N, First_Named);

   end Resolve_Entry_Call;

   -------------------------
   -- Resolve_Equality_Op --
   -------------------------

   --  Both arguments must have the same type, and the boolean context
   --  does not participate in the resolution. The first pass verifies
   --  that the interpretation is not ambiguous, and the type of the left
   --  argument is correctly set, or is Any_Type in case of ambiguity.

   --  Equality may be dispatching (???).

   procedure Resolve_Equality_Op (N : Node_Id; Typ : Entity_Id) is
      L : constant Node_Id   := Left_Opnd (N);
      R : constant Node_Id   := Right_Opnd (N);
      T : constant Entity_Id := Find_Unique_Type (L, R);

   begin
      if T /= Any_Type then
         Resolve (L, T);
         Resolve (R, T);
         Eval_Relational_Op (N);
      end if;
   end Resolve_Equality_Op;

   ----------------------------------
   -- Resolve_Explicit_Dereference --
   ----------------------------------

   procedure Resolve_Explicit_Dereference (N : Node_Id; Typ : Entity_Id) is
      P  : Node_Id := Prefix (N);
      I  : Interp_Index;
      It : Interp;

   begin
      if Is_Overloaded (P) then

         --  Use the context type to select the prefix that has the
         --  correct designated type.

         Get_First_Interp (P, I, It);
         while Present (It.Typ) loop
            exit when Covers (Typ, Designated_Type (It.Typ));

            Get_Next_Interp (I, It);
         end loop;

         Resolve (P, It.Typ);
         Set_Etype (N, Designated_Type (It.Typ));

      else
         Resolve (P, Etype (P));
      end if;

      if Is_Access_Type (Etype (P)) then
         Apply_Access_Check (N, Etype (P));
      end if;

      --  Note: there is no Eval processing required for an explicit
      --  deference, because the type is known to be an allocators, and
      --  allocator expressions can never be static.

   end Resolve_Explicit_Dereference;

   --------------------------------
   -- Resolve_Expression_Actions --
   --------------------------------

   procedure Resolve_Expression_Actions (N : Node_Id; Typ : Entity_Id) is
   begin
      --  Note: Expression (N) cannot be replaced by a constant in this
      --  procedure because if it happens to be an N_Allocator, then in
      --  certain conditions this expression can be modified during its
      --  expansion called by the Resolve call below (see exp_ch4.adb,
      --  Expand_Allocator)

      Resolve (Expression (N), Typ);
      Set_Etype (N, Etype (Expression (N)));

      --  Move up implicit types

      Transfer_Itypes (From => Expression (N), To => N);

      --  Note: any Eval actions required have already been performed on
      --  the expression that is contained in the expression actions node.

   end Resolve_Expression_Actions;

   -------------------------------
   -- Resolve_Indexed_Component --
   -------------------------------

   procedure Resolve_Indexed_Component (N : Node_Id; Typ : Entity_Id) is
      Name       : Node_Id := Prefix  (N);
      Expr       : Node_Id := First (Expressions (N));
      Array_Type : Entity_Id;
      Index      : Node_Id;

   begin
      if Is_Overloaded (Name) then

         --  Use the context type to select the prefix that yields the
         --  correct component type.

         Unimplemented (N, "Overloaded prefixes ");

      else
         Array_Type := Etype (Name);
         Resolve (Name, Etype (Name));

         if Is_Access_Type (Array_Type) then
            Apply_Access_Check (N, Array_Type);
            Array_Type := Designated_Type (Array_Type);
         end if;

         Index := First_Index (Array_Type);

         while Present (Index) loop
            Resolve (Expr, Etype (Index));
            Apply_Range_Check (Expr, Etype (Expr), Etype (Index));
            Index := Next_Index (Index);
            Expr  := Next (Expr);
         end loop;
      end if;

      Eval_Indexed_Component (N);

   end Resolve_Indexed_Component;

   -----------------------------
   -- Resolve_Integer_Literal --
   -----------------------------

   procedure Resolve_Integer_Literal (N : Node_Id; Typ : Entity_Id) is
   begin
      Set_Etype (N, Typ);
      Eval_Integer_Literal (N);
   end Resolve_Integer_Literal;

   ------------------------
   -- Resolve_Logical_Op --
   ------------------------

   procedure Resolve_Logical_Op (N : Node_Id; Typ : Entity_Id) is
      B_Typ : Entity_Id;

   begin
      --  Predefined operations on  scalar types yield the base type. On
      --  the other hand, logical operations on arrays yield the type of
      --  the arguments (and the context).

      if Is_Array_Type (Typ) then
         B_Typ := Typ;
      else
         B_Typ := Base_Type (Typ);
      end if;

      Resolve (Left_Opnd (N), B_Typ);
      Resolve (Right_Opnd (N), B_Typ);
      Set_Etype (N, B_Typ);
      Eval_Logical_Op (N);
   end Resolve_Logical_Op;

   ---------------------------
   -- Resolve_Membership_Op --
   ---------------------------

   --  The context can only be a boolean type, and does not determine
   --  the arguments. Take the non-universal type of either argument,
   --  if any, to complete the resolution of both. If the second argument
   --  is a subtype name, there is nothing to resolve.

   procedure Resolve_Membership_Op (N : Node_Id; Typ : Entity_Id) is
      L : constant Node_Id   := Left_Opnd (N);
      R : constant Node_Id   := Right_Opnd (N);
      T : constant Entity_Id := Intersect_Types (L, R);

   begin
      Resolve (L, T);

      if Is_Entity_Name (R) then
         null;
      else
         Resolve (R, T);
      end if;

      Eval_Membership_Op (N);
   end Resolve_Membership_Op;

   ------------------
   -- Resolve_Null --
   ------------------

   procedure Resolve_Null (N : Node_Id; Typ : Entity_Id) is
   begin
      --   The literal NULL takes its type from the context.
      --   (null = null) is now ambiguous in all cases ???.

      Set_Etype (N, Typ);
   end Resolve_Null;

   --------------------
   -- Resolve_Op_Not --
   --------------------

   procedure Resolve_Op_Not (N : Node_Id; Typ : Entity_Id) is
      B_Typ : Entity_Id;

   begin
      --  Predefined operations on  scalar types yield the base type. On
      --  the other hand, logical operations on arrays yield the type of
      --  the arguments (and the context).

      if Is_Array_Type (Typ) then
         B_Typ := Typ;
      else
         B_Typ := Base_Type (Typ);
      end if;

      Resolve (Right_Opnd (N), B_Typ);
      Set_Etype (N, B_Typ);
      Eval_Op_Not (N);
   end Resolve_Op_Not;

   -----------------------------
   -- Resolve_Operator_Symbol --
   -----------------------------

   procedure Resolve_Operator_Symbol (N : Node_Id; Typ : Entity_Id) is
   begin
      null;
   end Resolve_Operator_Symbol;

   -----------------------
   -- Resolve_Op_Concat --
   -----------------------

   procedure Resolve_Op_Concat (N : Node_Id; Typ : Entity_Id) is
      Btyp : constant Entity_Id := Base_Type (Typ);

      --  Internal procedure to resolve one argument of concatenation operator.
      --  The argument is either of the array type or of the component type.

      procedure Resolve_Concatenation_Arg (Arg : Node_Id) is
      begin
         if Has_Compatible_Type (Arg, Component_Type (Typ)) then
            Resolve (Arg, Component_Type (Typ));
         else
            Resolve (Arg, Btyp);
         end if;
      end Resolve_Concatenation_Arg;

   begin
      Set_Etype (N, Btyp);
      Resolve_Concatenation_Arg (Left_Opnd (N));
      Resolve_Concatenation_Arg (Right_Opnd (N));
      Eval_Concatenation (N);
   end Resolve_Op_Concat;

   ----------------------
   -- Resolve_Op_Expon --
   ----------------------

   procedure Resolve_Op_Expon (N : Node_Id; Typ : Entity_Id) is
      B_Typ : constant Entity_Id := Base_Type (Typ);

   begin
      --  We do the resolution using the base type, because intermediate values
      --  in expressions always are of the base type, not a subtype of it.

      Resolve (Left_Opnd (N), B_Typ);
      Resolve (Right_Opnd (N), Standard_Integer);
      Set_Etype (N, B_Typ);
      Eval_Op_Expon (N);

      --  Set overflow checking bit. Much cleverer code needed here eventually
      --  and perhaps the Resolve routines should be separated for the various
      --  arithmetic operations, since they will need different processing. ???

      if Nkind (N) in N_Op then
         if not Overflow_Checks_Suppressed (Etype (N)) then
            Set_Do_Overflow_Check (N, True);
         end if;
      end if;

   end Resolve_Op_Expon;

   ----------------------------------
   -- Resolve_Qualified_Expression --
   ----------------------------------

   procedure Resolve_Qualified_Expression (N : Node_Id; Typ : Entity_Id) is
      Mark_Typ : constant Entity_Id := Entity (Subtype_Mark (N));
      Expr     : constant Node_Id   := Expression (N);

   begin
      Resolve (Expr, Mark_Typ);

      --  In general, a specific tagged type can be given when class-wide type
      --  is expected but this is not the case for Qualified Expressions.

      if Is_Class_Wide_Type (Mark_Typ)
        and then not Is_Class_Wide_Type (Etype (Expr))
      then
         Wrong_Type (Expr, Mark_Typ);
      end if;

      Eval_Qualified_Expression (N);
   end Resolve_Qualified_Expression;

   -------------------
   -- Resolve_Range --
   -------------------

   procedure Resolve_Range (N : Node_Id; Typ : Entity_Id) is
      L : constant Node_Id := Low_Bound (N);
      R : constant Node_Id := High_Bound (N);

   begin
      Set_Etype (N, Typ);
      Resolve (L, Typ);
      Resolve (R, Typ);
   end Resolve_Range;

   --------------------------
   -- Resolve_Real_Literal --
   --------------------------

   procedure Resolve_Real_Literal (N : Node_Id; Typ : Entity_Id) is
   begin
      Set_Etype (N, Typ);
      Eval_Real_Literal (N);
   end Resolve_Real_Literal;

   -----------------------
   -- Resolve_Reference --
   -----------------------

   procedure Resolve_Reference (N : Node_Id; Typ : Entity_Id) is
   begin
      null; -- TBD ???
   end Resolve_Reference;

   --------------------------------
   -- Resolve_Selected_Component --
   --------------------------------

   procedure Resolve_Selected_Component (N : Node_Id; Typ : Entity_Id) is
      P  : constant Node_Id   := Prefix  (N);
      T  : constant Entity_Id := Etype (P);

   begin
      --  The prefix may be overloaded, or may involve expressions that
      --  must be further resolved.

      if Is_Overloaded (P) then

         --  Use the context type to select the prefix that has a selector
         --  of the correct name and type.

         Unimplemented (N, "Overloaded prefixes");
      else
         Resolve (P, T);
      end if;

      if Is_Access_Type (Etype (P)) then
         Apply_Access_Check (N, Etype (P));
      end if;

      --  Note: No Eval processing is required, because the prefix is of a
      --  record type, or protected type, and neither can possibly be static.

   end Resolve_Selected_Component;

   -------------------
   -- Resolve_Shift --
   -------------------

   procedure Resolve_Shift (N : Node_Id; Typ : Entity_Id) is
      B_Typ : constant Entity_Id := Base_Type (Typ);

   begin
      --  We do the resolution using the base type, because intermediate values
      --  in expressions always are of the base type, not a subtype of it.

      Resolve (Left_Opnd (N), B_Typ);
      Resolve (Right_Opnd (N), Standard_Natural);
      Set_Etype (N, B_Typ);
      Eval_Shift (N);

   end Resolve_Shift;

   -------------------
   -- Resolve_Slice --
   -------------------

   procedure Resolve_Slice (N : Node_Id; Typ : Entity_Id) is
      Name       : constant Node_Id := Prefix (N);
      Drange     : constant Node_Id := Discrete_Range (N);
      Array_Type : Entity_Id;
      Index      : Node_Id;

   begin
      if Is_Overloaded (Name) then

         --  Use the context type to select the prefix that yields the
         --  correct component type.

         Unimplemented (N, "Overloaded prefixes");

      else
         Array_Type := Etype (Name);
         Resolve (Name, Array_Type);

         if Is_Access_Type (Array_Type) then
            Apply_Access_Check (N, Array_Type);
            Array_Type := Designated_Type (Array_Type);
         end if;

         Index := First_Index (Array_Type);
         Resolve (Drange, Base_Type (Etype (Index)));
         Eval_Slice (N);
      end if;

   end Resolve_Slice;

   ---------------------------
   -- Resolve_Short_Circuit --
   ---------------------------

   procedure Resolve_Short_Circuit (N : Node_Id; Typ : Entity_Id) is
   begin
      Resolve (Left_Opnd (N), Standard_Boolean);
      Resolve (Right_Opnd (N), Standard_Boolean);
      Eval_Short_Circuit (N);
   end Resolve_Short_Circuit;

   ----------------------------
   -- Resolve_String_Literal --
   ----------------------------

   procedure Resolve_String_Literal (N : Node_Id; Typ : Entity_Id) is
      B_Typ      : constant Entity_Id  := Base_Type (Component_Type (Typ));
      Loc        : constant Source_Ptr := Sloc (N);
      Str        : constant String_Id  := Strval (N);
      Strlen     : constant Nat        := String_Length (Str);
      Subtype_Id : Entity_Id;

   begin
      --  Create a special subtype for the string node which becomes its Etype

      Subtype_Id := New_Itype (E_String_Literal_Subtype, N);
      Set_Component_Type (Subtype_Id, Component_Type (Typ));
      Set_String_Literal_Length
        (Subtype_Id, UI_From_Int (String_Length (Strval (N))));
      Set_Etype (Subtype_Id, Base_Type (Typ));
      Set_Etype (N, Subtype_Id);
      Eval_String_Literal (N);

      --  The null string is always valid

      if Strlen = 0 then
         return;

      --  For Standard.Wide_String, or any other type whose component type is
      --  Standard.Wide_Character, we know that all the characters in the
      --  string must be acceptable, since the parser accepted the characters
      --  as valid character literals.

      elsif B_Typ = Standard_Wide_Character then
         return;

      --  Always accept string literal with component type Any_Character,
      --  which occurs in error situations and in comparisons of literals,
      --  both of which should accept all literals.

      elsif B_Typ = Any_Character then
         return;

      --  For the case of Standard.String, or any other type whose component
      --  type is Standard.Character, we must make sure that there are no
      --  wide characters in the string, i.e. that it is entirely composed
      --  of characters in range of type String.

      elsif B_Typ = Standard_Character then
         for J in 1 .. Strlen loop
            if not In_Character_Range (Get_String_Char (Str, J)) then

               --  If we are out of range, post error. This is one of the
               --  very few places that we place the flag in the middle of
               --  a token, right under the offending wide character.

               Error_Msg
                 ("literal out of range of type Character",
                   Source_Ptr (Int (Loc) + J));
               return;
            end if;
         end loop;

         return;

      --  For all other component types, we transform the string literal into
      --  the equivalent qualified positional array aggregate. This is rather
      --  heavy artillery for this situation, but it is hard work to avoid.

      else
         declare
            Lits : List_Id    := New_List;
            P    : Source_Ptr := Loc + 1;
            C    : Char_Code;

         begin
            --  Build the character literals, we give them source locations
            --  that correspond to the string positions, which is a bit tricky
            --  given the possible presence of wide character escape sequences.

            for J in 1 .. Strlen loop
               C := Get_String_Char (Str, J);
               Set_Character_Literal_Name (C);

               Append_To (Lits,
                 Make_Character_Literal (P, Name_Find, C));

               if In_Character_Range (C) then
                  P := P + 1;
--  ???        else
--                Skip_Wide (P);
               end if;
            end loop;

            Rewrite_Substitute_Tree (N,
              Make_Qualified_Expression (Loc,
                Subtype_Mark => New_Reference_To (Typ, Loc),
                Expression   =>
                  Make_Aggregate (Loc, Expressions => Lits)));

            Analyze (N);
            Resolve (N, Typ);
         end;
      end if;

   end Resolve_String_Literal;

   -----------------------------
   -- Resolve_Type_Conversion --
   -----------------------------

   procedure Resolve_Type_Conversion (N : Node_Id; Typ : Entity_Id) is
      Target_Type : Entity_Id := Etype (N);
      Operand     : Node_Id   := Expression (N);
      Opnd_Type   : Entity_Id := Etype (Operand);

   begin
      if not Valid_Conversion (N) then
         return;
      end if;

      Resolve (Operand, Opnd_Type);

      if Is_Tagged_Type (Typ) then
         Note_Feature (Tagged_Type_Conversion, Sloc (N));
      end if;

      --  If no suppression of range checking is specified, enable flags to do
      --  range checking for type conversion. An overflow check is necessary
      --  in the case where the source target is larger than the target type
      --  of the conversion. Currently this check only occurrs for signed
      --  integer or enumeration types since it is premature to do this for
      --  fixed point, floating point and modular types.

      if Is_Discrete_Type (Target_Type)
        and then not Is_Modular_Integer_Type (Target_Type)
        and then not Range_Checks_Suppressed (Target_Type)
      then
         if Esize (Etype (Operand)) > Esize (Target_Type) then
            Set_Do_Overflow_Check (N, True);
         end if;

         Apply_Range_Check (Operand, Etype (Operand), Target_Type);
      end if;

      Eval_Type_Conversion (N);
   end Resolve_Type_Conversion;

   ----------------------
   -- Resolve_Unary_Op --
   ----------------------

   procedure Resolve_Unary_Op (N : Node_Id; Typ : Entity_Id) is
   begin
      Set_Etype (N, Base_Type (Typ));
      Resolve (Right_Opnd (N), Typ);
      Eval_Unary_Op (N);

      --  Set overflow checking bit. Much cleverer code needed here eventually
      --  and perhaps the Resolve routines should be separated for the various
      --  arithmetic operations, since they will need different processing ???

      if Nkind (N) in N_Op then
         if not Overflow_Checks_Suppressed (Etype (N)) then
            Set_Do_Overflow_Check (N, True);
         end if;
      end if;

   end Resolve_Unary_Op;

   ---------------------------------------
   -- Resolve_Unchecked_Type_Conversion --
   ---------------------------------------

   procedure Resolve_Unchecked_Type_Conversion
     (N   : Node_Id;
      Typ : Entity_Id)
   is
      Target_Type : Entity_Id := Etype (N);
      Operand     : Node_Id   := Expression (N);
      Opnd_Type   : Entity_Id := Etype (Operand);

   begin
      --  Unlike normal typed conversions, unchecked conversions cannot be
      --  done on operands of universal type, since the size of such operands
      --  is unclear. Source level unchecked conversions (from instantiations
      --  of Unchecked_Conversion) could never generate such situations, so
      --  this check is a defence against incompetent expander actions.
      --  Similarly, overloaded operands are invalid, but could only arise
      --  from improper expanded code.

      pragma Assert
        (Opnd_Type /= Universal_Integer
          and then Opnd_Type /= Universal_Real
          and then not Is_Overloaded (Operand));

      --  Resolve operand using its own type.

      Resolve (Operand, Opnd_Type);
      Eval_Unchecked_Conversion (N);

   end Resolve_Unchecked_Type_Conversion;

   ------------------------------
   -- Rewrite_Operator_As_Call --
   ------------------------------

   procedure Rewrite_Operator_As_Call (N : Node_Id; Nam : Entity_Id) is
      Actuals :  List_Id := New_List;

   begin
      if Nkind (N) in  N_Binary_Op then
         Append (Left_Opnd (N), Actuals);
      end if;

      Append (Right_Opnd (N), Actuals);

      Change_Node (N, N_Function_Call);
      Set_Etype   (N, Etype (Nam));
      Set_Name    (N, New_Occurrence_Of (Nam, Sloc (N)));
      Set_Parameter_Associations (N, Actuals);
   end Rewrite_Operator_As_Call;

   ----------------------
   -- Valid_Conversion --
   ----------------------

   function Valid_Conversion (N : Node_Id) return Boolean is
      Target_Type : Entity_Id := Base_Type (Etype (N));
      Operand     : Node_Id   := Expression (N);
      Opnd_Type   : Entity_Id := Etype (Operand);

      function Conversion_Check
        (Valid : Boolean;
         Msg   : String)
         return  Boolean
      is
      begin
         if not Valid then
            Error_Msg_N (Msg, Operand);
         end if;
         return Valid;
      end Conversion_Check;

   --  Start of processing for Valid_Conversion

   begin
      if Is_Overloaded (Operand) then

         declare
            I   : Interp_Index;
            It  : Interp;
            N1  : Entity_Id;
            N2  : Entity_Id;
            Nam : Entity_Id;
            T1  : Entity_Id;
            T2  : Entity_Id;

         begin
            Get_First_Interp (Operand, I,  It);
            N1 := It.Nam;
            T1 := It.Typ;
            Get_Next_Interp (I, It);
            N2 := It.Nam;
            T2 := It.Typ;
            Nam :=  Disambiguate (N1, N2, Any_Type);

            if Nam = Any_Id then
               Error_Msg_N ("ambiguous operand in conversion", Operand);
               return False;
            else
               if Nam = N1 then
                  Set_Etype (Operand, T1);
               else
                  Set_Etype (Operand, T2);
               end if;

               Set_Is_Overloaded (Operand, False);
            end if;
         end;
      end if;

      if Chars (Current_Scope) = Name_Unchecked_Conversion then
         --  This check is dubious, what if there were a user defined
         --  scope whose name was Unchecked_Conversion ???
         return True;

      elsif Is_Numeric_Type (Target_Type)  then
         if Opnd_Type = Universal_Fixed then
            return True;
         else
            return Conversion_Check (Is_Numeric_Type (Opnd_Type),
                             "illegal operand for numeric conversion");
         end if;

      elsif Is_Array_Type (Target_Type) then
         --  additional tests on index and component types ???
         return Conversion_Check
                  (Is_Array_Type (Opnd_Type)
                    and then Number_Dimensions (Target_Type)
                               = Number_Dimensions (Opnd_Type),
                  "illegal operand for array conversion");

      elsif Ekind (Target_Type) = E_General_Access_Type then
         --  additional tests on designated types, tagged types ???
         return Conversion_Check (Is_Access_Type (Opnd_Type),
                  "illegal operand for access type conversion");

      elsif Ekind (Target_Type) = E_Access_Subprogram_Type then
         --  check that designated types are type conformant ???
         return Conversion_Check
                  (Ekind (Opnd_Type) = E_Access_Subprogram_Type,
                  "illegal operand for access subprogram conversion");

      --  Types derived from the same root type are convertible

      elsif Root_Type (Target_Type) = Root_Type (Opnd_Type) then

         if Is_Tagged_Type (Target_Type) then
            if Covers (Target_Type, Opnd_Type)
              or else Is_Ancestor (Target_Type, Opnd_Type)
            then
               return True;

            elsif Is_Class_Wide_Type (Opnd_Type)
              and then Covers (Opnd_Type, Target_Type)
            then
               return True;
            else
               return Conversion_Check (False,
                        "downward conversion of tagged objects not allowed");
            end if;
         else
            return True;
         end if;

      else
         return
           Conversion_Check (False, "illegal operand for conversion");
      end if;
   end Valid_Conversion;

end Sem_Res;


----------------------
-- REVISION HISTORY --
----------------------

--  ----------------------------
--  revision 1.193
--  date: Wed Aug 24 14:02:20 1994;  author: schonber
--  (Resolve): handle calls to parameterless protected functions.
--  ----------------------------
--  revision 1.194
--  date: Wed Aug 24 19:37:43 1994;  author: dewar
--  (Resolve_Slice): Resolve range with base type of index type, not index
--   type (because of null ranges).
--  Minor reformatting
--  ----------------------------
--  revision 1.195
--  date: Sun Aug 28 08:50:05 1994;  author: comar
--  (Resolve_String_Literal):  Use new protocol for New_Itype
--  ----------------------------
--  New changes after this line.  Each line starts with: "--  "
