------------------------------------------------------------------------------
--                                                                          --
--                         GNAT COMPILER COMPONENTS                         --
--                                                                          --
--                             S E M _ E V A L                              --
--                                                                          --
--                                 B o d y                                  --
--                                                                          --
--                            $Revision: 1.114 $                            --
--                                                                          --
--           Copyright (c) 1992,1993,1994 NYU, All Rights Reserved          --
--                                                                          --
-- GNAT is free software;  you can  redistribute it  and/or modify it under --
-- terms of the  GNU General Public License as published  by the Free Soft- --
-- ware  Foundation;  either version 2,  or (at your option) any later ver- --
-- sion.  GNAT is distributed in the hope that it will be useful, but WITH- --
-- OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
-- or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License --
-- for  more details.  You should have  received  a copy of the GNU General --
-- Public License  distributed with GNAT;  see file COPYING.  If not, write --
-- to the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA. --
--                                                                          --
------------------------------------------------------------------------------

with Atree;    use Atree;
with Einfo;    use Einfo;
with Errout;   use Errout;
with Itypes;   use Itypes;
with Lib;      use Lib;
with Namet;    use Namet;
with Nmake;    use Nmake;
with Nlists;   use Nlists;
with Opt;      use Opt;
with Output;   use Output;
with Sem;      use Sem;
with Sem_Attr; use Sem_Attr;
with Sem_Ch5;  use Sem_Ch5;
with Sem_Res;  use Sem_Res;
with Sem_Util; use Sem_Util;
with Sinfo;    use Sinfo;
with Snames;   use Snames;
with Stand;    use Stand;
with Stringt;  use Stringt;
with Tbuild;   use Tbuild;
with Ttypes;   use Ttypes;
with Uintp;    use Uintp;
with Urealp;   use Urealp;

package body Sem_Eval is

   -----------------------------------------
   -- Handling of Compile Time Evaluation --
   -----------------------------------------

   --  The compile time evaluation of expressions is distributed over several
   --  Eval_xxx procedures. These procedures are called immediatedly after
   --  a subexpression is resolved and is therefore accomplished in a bottom
   --  up fashion. The flags are synthesized using the following approach.

   --    Potentially_Static is determined by following the detailed rules
   --    in RM 4.9(4-14). This involves testing the Potentially_Static
   --    flag of the operands in many cases.

   --    Raises_Constraint_Error is set if any of the operands have the flag
   --    set or if an attempt to compute the value of the current expression
   --    results in detection of a runtime constraint error.

   --  As described in the spec, the requirement is that Potentially_Static
   --  be accurately set, and in addition for nodes for which this flag is set,
   --  Raises_Constraint_Error must also be set. Furthermore a node which has
   --  Potentially_Static set, and Raises_Constraint_Error clear, then the
   --  requirement is that the expression value must be precomputed, and the
   --  node is either a literal, or the name of a constant entity whose value
   --  is a static expression.

   --  The general approach is as follows. First compute Potentially_Static.
   --  If the node is not potentially static, then the flag is left off in the
   --  node and we are all done. Otherwise for a potentially static node, we
   --  test if any of the operands will raise constraint error, and if so set
   --  the Raises_Constraint_Error flag of the result node and we are done.
   --  For the case of a potentially static node whose operands are static,
   --  i.e. do not raise constraint error, we then attempt to evaluate the
   --  node. Either this evaluation succeeds, in which case the node is
   --  replaced by the result of this computation, or we discover that the
   --  evaluation raises constraint error, in which case we use the routine
   --  Create_Raise_Expression to rewrite the node to raise the exception,
   --  and set Raise_Constraint_Error on the node.

   ----------------
   -- Local Data --
   ----------------

   type Bits is array (Nat range <>) of Boolean;
   --  Used to convert unsigned (modular) values for folding logical ops

   -----------------------
   -- Local Subprograms --
   -----------------------

   function Compare_Strings (S1, S2 : String_Id) return Int;
   --  Compares two strings, used for folding string comparisons. Returns
   --  negative, zero or positive for S1 < S2, S1 = S2, S1 > S2 respectively.

   function Expression_Is_Not_Foldable
     (N    : Node_Id;
      Op1  : Node_Id)
      return Boolean;
   --  Copies the Potentially_Static and Raises_Constraint_Error flags
   --  of operand Op1 to result node N, and returns True if the resulting
   --  expression is not static and therefore cannot be folded (i.e. it is
   --  not potentially static, or else it has Raises_Constraint_Error set).

   function Expression_Is_Not_Foldable
     (N    : Node_Id;
      Op1  : Node_Id;
      Op2  : Node_Id)
      return Boolean;
   --  Sets the Potentially_Static and Raises_Constraint_Error flags of
   --  node N from the settings of these flags in nodes Op1 and Op2. The
   --  Potentially_Static flag is set if both of the operands have the
   --  flag set, and the Raises_Constraint_Error flag is set if either of
   --  the operands have the flag set and the result is potentially static.
   --  The value returned is true if the resulting expression is non-static
   --  and therefore cannot be folded (i.e. it is not potentially static, or
   --  else it has Raises_Constraint_Error set).

   function From_Bits (B : Bits) return Uint;
   --  Converts a bit string of length B'Length to a Uint value

   function Get_String_Val (N : Node_Id) return Node_Id;
   --  Given a tree node for a folded string or character value, returns
   --  the corresponding string literal or character literal (one of the
   --  two must be available, or the operand would not have been marked
   --  as folded in the earlier analysis of the operands).

   function Potentially_Static_Range (N : Node_Id) return Boolean;
   --  Determine if range is potentially static, as defined in RM 4.9(26).
   --  The only allowed argument is an N_Range node (but note that the
   --  semantic analysis of equivalent range attribute references already
   --  turned them into the equivalent range).

   function Test (Cond : Boolean) return Uint;
   pragma Inline (Test);
   --  This function simply returns the appropriate Boolean'Pos value
   --  corresponding to the value of Cond as a universal integer. It is
   --  used for producing the result of the static evaluation of the
   --  logical operators

   procedure To_Bits (U : Uint; B : out Bits);
   --  Converts a Uint value to a bit string of length B'Length

   ------------------------------
   -- Check_Non_Static_Context --
   ------------------------------

   --  A static expression is said to appear in a non-static context if it
   --  appears as a subexpression of a non-static expression at the highest
   --  level, i.e. it is not itself part of a static expression (RM 4.9(35)).

   --  This situation represents a transition from the static world to the
   --  non-static world, and there are two special rules (RM 4.9(36)).

   --  First, the value of a static scalar expression must be within the
   --  base range of its type if the subtype is non-static. We also issue
   --  a warning if it is inside the base range, but outside the range of
   --  its subtype if the subtype is static.

   --  Third, for a value of a decimal fixed type, the value must be an
   --  exact multiple of the SMALL value, i.e. extra precision is not allowed.

   procedure Check_Non_Static_Context (N : Node_Id) is
      T : Entity_Id := Etype (N);

   begin
      --  This check is only made for static scalar expressions

      if Is_Static_Expression (N) and then Is_Scalar_Type (T) then

         --  Case of outside base range

         if not Is_In_Range (N, Base_Type (T)) then

            --  In 9X mode, outside base range is an error

            if Ada_9X then
               Error_Msg_NE
                 ("value is outside range of type&", N, Base_Type (T));

            --  In 83 mode, outside base range is just a warning

            else
               Constraint_Error_Warning (N, "static value out of range?!");
            end if;

            --  Preserve the Sloc of the node,  because the bound we just
            --  copied may be a node in Standard.

            Set_Sloc (N, Sloc (Parent (N)));

         --  Give warning if outside subtype and subtype is static

         elsif T /= Base_Type (T)
           and then Is_Static_Subtype (T)
           and then not Is_In_Range (N, T)
         then
            Constraint_Error_Warning (N, "static value out of range?!");
         end if;

         --  Do special check for decimal fixed point value. We don't need a
         --  special 9X test here (there are no decimal types in Ada 83!)

         if Is_Decimal_Fixed_Point_Type (Etype (N)) then
            declare
               Val : constant Ureal := Expr_Value (N);

            begin
               --  Do the check by dividing the value by the small value,
               --  normalizing, and checking that resulting denominator = 1.

               if UI_Ne (
                   Uint_1,
                   Norm_Den (UR_Quotient (Val, Small_Value (Etype (N)))))
               then
                  Error_Msg_N ("value has extraneous low order digits", N);
               end if;
            end;
         end if;
      end if;
   end Check_Non_Static_Context;

   -----------------------------
   -- Check_Static_Expression --
   -----------------------------

   --  This routine is called at the outer level on expressions which appear
   --  in a context requiring a static expression. If the expression is not
   --  static, an error message is issued.

   procedure Check_Static_Expression (N : Node_Id) is
   begin
      if not Is_Static_Expression (N) then
         Error_Msg_N ("static expression required in this context", N);
         Set_Etype (N, Any_Type);
      end if;
   end Check_Static_Expression;

   ---------------------
   -- Compare_Strings --
   ---------------------

   function Compare_Strings (S1, S2 : String_Id) return Int is
      L1 : constant Nat := String_Length (S1);
      L2 : constant Nat := String_Length (S2);
      LM : Nat;

      C1 : Char_Code;
      C2 : Char_Code;

   begin
      if L1 <= L2 then
         LM := L1;
      else
         LM := L2;
      end if;

      for J in 1 .. LM loop
         C1 := Get_String_Char (S1, J);
         C2 := Get_String_Char (S2, J);

         if C1 /= C2 then
            return Int (C1) - Int (C2);
         end if;
      end loop;

      return Int (L1) - Int (L2);
   end Compare_Strings;

   -----------------
   -- Eval_Actual --
   -----------------

   --  This is only called for actuals of functions that are not predefined
   --  operators (which have already been rewritten as operators at this
   --  stage), so the call can never be folded, and all that needs doing for
   --  the actual is to do the check for a non-static context.

   procedure Eval_Actual (N : Node_Id) is
   begin
      Check_Non_Static_Context (N);
   end Eval_Actual;

   --------------------
   -- Eval_Aggregate --
   --------------------

   procedure Eval_Aggregate (N : Node_Id) is
   begin
      null;          --  ???
   end Eval_Aggregate;

   --------------------
   -- Eval_Allocator --
   --------------------

   --  Allocators are never static, so all we have to do is to do the
   --  check for a non-static context if an expression is present.

   procedure Eval_Allocator (N : Node_Id) is
      Expr : constant Node_Id := Expression (N);

   begin
      if Nkind (Expr) = N_Qualified_Expression then
         Check_Non_Static_Context (Expression (Expr));
      end if;
   end Eval_Allocator;

   ------------------------
   -- Eval_Arithmetic_Op --
   ------------------------

   --  Arithmetic operations are static functions, so the result is potentially
   --  static if both operands are potentially static (RM 4.9(7), 4.9(20)).

   procedure Eval_Arithmetic_Op (N : Node_Id) is
      Left  : constant Node_Id   := Left_Opnd (N);
      Right : constant Node_Id   := Right_Opnd (N);
      Ltype : constant Entity_Id := Etype (Left);
      Rtype : constant Entity_Id := Etype (Right);

   begin
      if Expression_Is_Not_Foldable (N, Left, Right) then
         Check_Non_Static_Context (Left);
         Check_Non_Static_Context (Right);
         return;
      end if;

      --  Fold for cases where both operands are of integer type

      if Is_Integer_Type (Ltype) and then Is_Integer_Type (Rtype) then
         declare
            Left_Int  : constant Uint := Expr_Value (Left);
            Right_Int : constant Uint := Expr_Value (Right);
            Result    : Uint;

         begin
            case Nkind (N) is

               when N_Op_Add =>
                  Result := UI_Sum (Left_Int, Right_Int);

               when N_Op_Subtract =>
                  Result := UI_Difference (Left_Int, Right_Int);

               when N_Op_Multiply =>
                  Result := UI_Product (Left_Int, Right_Int);

               when N_Op_Divide =>

                  --  The exception Constraint_Error is raised by integer
                  --  division, rem and mod if the right operand is zero.

                  if UI_Is_Zero (Right_Int) then
                     Constraint_Error_Warning (N, "division by zero?!");
                     Set_Potentially_Static (N, True);
                     Check_Non_Static_Context (Left);
                     Check_Non_Static_Context (Right);
                     return;
                  else
                     Result := UI_Quotient (Left_Int, Right_Int);
                  end if;

               when N_Op_Mod =>

                  --  The exception Constraint_Error is raised by integer
                  --  division, rem and mod if the right operand is zero.

                  if UI_Is_Zero (Right_Int) then
                     Constraint_Error_Warning (N, "mod with zero divisor?!");
                     Set_Potentially_Static (N, True);
                     Check_Non_Static_Context (Left);
                     Check_Non_Static_Context (Right);
                     return;
                  else
                     Result := UI_Mod (Left_Int, Right_Int);
                  end if;

               when N_Op_Rem =>

                  --  The exception Constraint_Error is raised by integer
                  --  division, rem and mod if the right operand is zero.

                  if UI_Is_Zero (Right_Int) then
                     Constraint_Error_Warning (N, "rem with zero divisor?!");
                     Set_Potentially_Static (N, True);
                     Check_Non_Static_Context (Left);
                     Check_Non_Static_Context (Right);
                     return;
                  else
                     Result := UI_Rem (Left_Int, Right_Int);
                  end if;

               when others =>
                  pragma Assert (False); null;
            end case;

            --  Adjust the result by the modulus if the type is a modular type

            if Is_Modular_Integer_Type (Ltype) then
               Result := UI_Mod (Result, Modulus (Ltype));
            end if;

            Fold_Uint (N, Result);
         end;

      --  Cases where at least one operand is a real. We handle the cases
      --  of both reals, or mixed/real integer cases (the latter happen
      --  only for divide and multiply, and the result is always real).

      elsif Is_Real_Type (Ltype) or else Is_Real_Type (Rtype) then
         declare
            Left_Real  : Ureal;
            Right_Real : Ureal;
            Result     : Ureal;

         begin
            if Is_Real_Type (Ltype) then
               Left_Real := Expr_Value (Left);
            else
               Left_Real := UR_From_Uint (Expr_Value (Left));
            end if;

            if Is_Real_Type (Rtype) then
               Right_Real := Expr_Value (Right);
            else
               Right_Real := UR_From_Uint (Expr_Value (Right));
            end if;

            if Nkind (N) = N_Op_Add then
               Result := UR_Sum (Left_Real, Right_Real);

            elsif Nkind (N) = N_Op_Subtract then
               Result := UR_Difference (Left_Real, Right_Real);

            elsif Nkind (N) = N_Op_Multiply then
               Result := UR_Product (Left_Real, Right_Real);

            elsif Nkind (N) = N_Op_Divide then
               if UR_Is_Zero (Right_Real) then
                  Constraint_Error_Warning (N, "division by zero?!");
                  Set_Potentially_Static (N, True);
                  Check_Non_Static_Context (Left);
                  Check_Non_Static_Context (Right);
                  return;
               end if;

               Result := UR_Quotient (Left_Real, Right_Real);

            else
               pragma Assert (False); null;
            end if;

            Fold_Ureal (N, Result);
         end;
      end if;

   end Eval_Arithmetic_Op;

   ----------------------------
   -- Eval_Character_Literal --
   ----------------------------

   procedure Eval_Character_Literal (N : Node_Id) is
   begin
      Set_Potentially_Static (N);
   end Eval_Character_Literal;

   ------------------------
   -- Eval_Concatenation --
   ------------------------

   --  Concatenation is a a static functions, so the result is potentially
   --  static if both operands are potentially static (RM 4.9(7), 4.9(21)).

   procedure Eval_Concatenation (N : Node_Id) is
      Left  : constant Node_Id := Left_Opnd (N);
      Right : constant Node_Id := Right_Opnd (N);

   begin
      --  Concatenation is never static in Ada 83

      if Ada_83
        or else Expression_Is_Not_Foldable (N, Left, Right)
      then
         Check_Non_Static_Context (Left);
         Check_Non_Static_Context (Right);
         return;
      end if;

      --  Compile time string concatenation. Note that operands that are
      --  aggregates were never marked as potentially static, so we don't
      --  attempt to fold concatenations involving such aggregates
      --  (see Eval_Aggregate). Needs some more thought ???

      declare
         Left_Str  : constant Node_Id := Get_String_Val (Left);
         Right_Str : constant Node_Id := Get_String_Val (Right);

      begin
         --  Establish new string literal, and store left operand. We make
         --  sure to use the special Start_String that takes an operand if
         --  the left operand is a string literal. Since this is optimized
         --  in the case where that is the most recently created string
         --  literal, we ensure efficient time/space behavior for the
         --  case of a concatenation of a series of string literals.

         if Nkind (Left_Str) = N_String_Literal then
            Start_String (Strval (Left_Str));
         else
            Start_String;
            Store_String_Char (Char_Literal_Value (Left_Str));
         end if;

         --  Now append the characters of the right operand

         if Nkind (Right_Str) = N_String_Literal then
            declare
               S : constant String_Id := Strval (Right_Str);

            begin
               for J in 1 .. String_Length (S) loop
                  Store_String_Char (Get_String_Char (S, J));
               end loop;
            end;
         else
            Store_String_Char (Char_Literal_Value (Right_Str));
         end if;

         Fold_Str (N, End_String);
      end;
   end Eval_Concatenation;

   ---------------------------------
   -- Eval_Conditional_Expression --
   ---------------------------------

   --  This GNAT internal construct can never be statically folded, so the
   --  only required processing is to do the check for non-static context
   --  for the two expression operands.

   procedure Eval_Conditional_Expression (N : Node_Id) is
      Condition : constant Node_Id := First (Expressions (N));
      Then_Expr : constant Node_Id := Next (Condition);
      Else_Expr : constant Node_Id := Next (Then_Expr);

   begin
      Check_Non_Static_Context (Then_Expr);
      Check_Non_Static_Context (Else_Expr);
   end Eval_Conditional_Expression;

   ----------------------
   -- Eval_Entity_Name --
   ----------------------

   --  This procedure is used for identifiers and expanded names. These are
   --  static if they denote a named number or static constant (RM 4.9(6)).
   --  or if the name denotes an enumeration literal (RM 4.9(22)).

   procedure Eval_Entity_Name (N : Node_Id) is
      Def_Id    : constant Entity_Id := Entity (N);
      Val       : Node_Id;

      --  Return True if N is on the left hand side of an assignment statement,
      --  or is the defining id in an object declaration.

      function Assignment_Left_Hand_Side (N : Node_Id) return Boolean is
      begin
         if (Nkind (Parent (N)) = N_Assignment_Statement
             and then N = Name (Parent (N)))
           or else (Nkind (Parent (N)) = N_Object_Declaration
             and then N = Defining_Identifier (Parent (N)))
         then
            return True;
         end if;

         return False;
      end Assignment_Left_Hand_Side;

   begin
      --  Enumeration literals are always considered to be constants
      --  and cannot raise constraint error (RM 4.9(22)).

      if Ekind (Def_Id) = E_Enumeration_Literal then
         Set_Potentially_Static (N);
         return;

      --  A name is potentially static if it denotes a static constant
      --  (RM 4.9(5)), and we also copy Raise_Constraint_Error. Notice
      --  that even if non-static, it does not violate 10.2.1(8) here,
      --  since this is not a variable.

      elsif Ekind (Def_Id) = E_Constant then
         Val := Constant_Value (Def_Id);

         if Present (Val) then
            Set_Potentially_Static      (N, Potentially_Static   (Val));
            Set_Raises_Constraint_Error (N, Raises_Constraint_Error (Val));
            return;
         end if;
      end if;

      --  Fall through if the name is not static.

      --  In the elaboration code of a preelaborated library unit, check
      --  that we do not have the evaluation of a primary that is a name of
      --  an object, unless the name is a static expression (RM 10.2.1(8)).
      --  Non-static constant and variable are the targets, generic parameters
      --  are not included because the generic declaration and body are
      --  preelaborable.

      --  Filter out cases that default primary is in a record type component
      --  decl., record type discriminant specification or primary is a param.
      --  in a record type implicit init. procedure call.

      if Inside_Preelaborated_Unit (N)
        and then not Inside_Subprogram_Unit (N)
        and then Comes_From_Source (Entity (N))
        and then Nkind (Parent (N)) /= N_Component_Declaration
        and then Nkind (Parent (N)) /= N_Discriminant_Specification
        and then ((Ekind (Entity (N)) = E_Variable
                           and then not Assignment_Left_Hand_Side (N)
                           and then not
                             (Is_Record_Type (Etype (N))
                               and then Nkind (Parent (N)) =
                                            N_Procedure_Call_Statement
                               and then not Comes_From_Source
                                            (Entity (Name (Parent (N))))))
                   or else (not Potentially_Static (N)
                           and then Ekind (Entity (N)) = E_Constant))
      then
         Error_Msg_N ("?non-static object name in preelaborated unit", N);
      end if;

   end Eval_Entity_Name;

   ----------------------------
   -- Eval_Indexed_Component --
   ----------------------------

   --  Indexed components are never static, so the only required processing
   --  is to perform the check for non-static context on the index values.

   procedure Eval_Indexed_Component (N : Node_Id) is
      Index : Node_Id;

   begin
      return; -- for now ???
      Index := First_Index (Etype (Prefix (N)));

      while Present (Index) loop
         Check_Non_Static_Context (Index);
         Index := Next_Index (Index);
      end loop;

   end Eval_Indexed_Component;

   --------------------------
   -- Eval_Integer_Literal --
   --------------------------

   --  Numeric literals are potentially static (RM 4.9(1))

   procedure Eval_Integer_Literal (N : Node_Id) is
   begin
      Set_Potentially_Static (N);
   end Eval_Integer_Literal;

   ---------------------
   -- Eval_Logical_Op --
   ---------------------

   --  Logical operations are static functions, so the result is potentially
   --  static if both operands are potentially static (RM 4.9(7), 4.9(20)).

   procedure Eval_Logical_Op (N : Node_Id) is
      Left      : constant Node_Id := Left_Opnd (N);
      Right     : constant Node_Id := Right_Opnd (N);

   begin
      if Expression_Is_Not_Foldable (N, Left, Right) then
         Check_Non_Static_Context (Left);
         Check_Non_Static_Context (Right);
         return;
      end if;

      --  Compile time evaluation of logical operation

      declare
         Left_Int  : constant Uint := Expr_Value (Left);
         Right_Int : constant Uint := Expr_Value (Right);

      begin
         if Is_Modular_Integer_Type (Etype (N)) then
            declare
               Left_Bits  : Bits (0 .. UI_To_Int (Esize (Etype (N))) - 1);
               Right_Bits : Bits (0 .. UI_To_Int (Esize (Etype (N))) - 1);

            begin
               To_Bits (Left_Int, Left_Bits);
               To_Bits (Right_Int, Right_Bits);

               --  Note: should really be able to use array ops instead of
               --  these loops, but they weren't working at the time ???

               if Nkind (N) = N_Op_And then
                  for J in Left_Bits'range loop
                     Left_Bits (J) := Left_Bits (J) and Right_Bits (J);
                  end loop;

               elsif Nkind (N) = N_Op_Or then
                  for J in Left_Bits'range loop
                     Left_Bits (J) := Left_Bits (J) or Right_Bits (J);
                  end loop;

               else
                  pragma Assert (Nkind (N) = N_Op_Xor);

                  for J in Left_Bits'range loop
                     Left_Bits (J) := Left_Bits (J) xor Right_Bits (J);
                  end loop;
               end if;

               Fold_Uint (N, From_Bits (Left_Bits));
            end;

         else
            pragma Assert (Is_Boolean_Type (Etype (N)));

            if Nkind (N) = N_Op_And then
               Fold_Uint (N,
                 Test (Is_True (Left_Int) and then Is_True (Right_Int)));

            elsif Nkind (N) = N_Op_Or then
               Fold_Uint (N,
                 Test (Is_True (Left_Int) or else Is_True (Right_Int)));

            else
               pragma Assert (Nkind (N) = N_Op_Xor);
               Fold_Uint (N,
                 Test (Is_True (Left_Int) xor Is_True (Right_Int)));
            end if;
         end if;
      end;
   end Eval_Logical_Op;

   ------------------------
   -- Eval_Membership_Op --
   ------------------------

   --  A membership test is potentially static if the expression is static,
   --  and the range is a potentially static range, or is a subtype mark
   --  denoting a static subtype (RM 4.9(12)).

   procedure Eval_Membership_Op (N : Node_Id) is
      Left   : constant Node_Id := Left_Opnd (N);
      Right  : constant Node_Id := Right_Opnd (N);
      Def_Id : Entity_Id;
      Lo     : Uint;
      Hi     : Uint;

   begin
      --  Case of right operand is a subtype name

      if Is_Entity_Name (Right) then
         Def_Id := Entity (Right);

         if (Is_Scalar_Type (Def_Id) or else Is_String_Type (Def_Id))
           and then Is_Static_Subtype (Def_Id)
         then
            if Expression_Is_Not_Foldable (N, Left) then
               return;
            end if;
         else
            Check_Non_Static_Context (Left);
            return;
         end if;

         --  Here we deal with the bizarre case of a string type
         --  For now, just never fold, we will worry about this later ???

         if Is_String_Type (Def_Id) then
            Check_Non_Static_Context (Left);
            return;
         end if;

         Lo := Expr_Value (Type_Low_Bound (Def_Id));
         Hi := Expr_Value (Type_High_Bound (Def_Id));

      --  Case of right operand is a range

      else
         if Potentially_Static_Range (Right) then
            if Expression_Is_Not_Foldable (N, Left) then
               return;
            end if;
         else
            Check_Non_Static_Context (Left);
            return;
         end if;

         Lo := Expr_Value (Low_Bound (Right));
         Hi := Expr_Value (High_Bound (Right));
      end if;

      --  Fold the membership test. We know we have a static range and Lo
      --  and Hi are set to the values of the end points of this range.

      declare
         Left_Int : constant Uint := Expr_Value (Left);
         Result   : Boolean;

      begin
         Result := UI_Le (Lo, Left_Int) and then UI_Le (Left_Int, Hi);

         if Nkind (N) = N_Op_Not_In then
            Result := not Result;
         end if;

         Fold_Uint (N, Test (Result));
      end;
   end Eval_Membership_Op;

   -----------------
   -- Eval_Op_Not --
   -----------------

   --  The not operation is a  static functions, so the result is potentially
   --  static if the operand is potentially static (RM 4.9(7), 4.9(20)).

   procedure Eval_Op_Not (N : Node_Id) is
      Right : constant Node_Id := Right_Opnd (N);
      Rint  : Uint;

   begin
      if Expression_Is_Not_Foldable (N, Right) then
         Check_Non_Static_Context (Right);
         return;
      end if;

      --  Fold not operation

      declare
         Rint : constant Uint := Expr_Value (Right);

      begin
         if Is_Modular_Integer_Type (Etype (N)) then
            declare
               Right_Bits : Bits (0 .. UI_To_Int (Esize (Etype (N))) - 1);

            begin
               To_Bits (Rint, Right_Bits);

               for J in Right_Bits'range loop
                  Right_Bits (J) := not Right_Bits (J);
               end loop;

               Fold_Uint (N, From_Bits (Right_Bits));
            end;

         else
            pragma Assert (Is_Boolean_Type (Etype (N)));
            Fold_Uint (N, Test (not Is_True (Rint)));
         end if;
      end;
   end Eval_Op_Not;

   -------------------
   -- Eval_Op_Expon --
   -------------------

   --  Exponentiation is a static functions, so the result is potentially
   --  static if both operands are potentially static (RM 4.9(7), 4.9(20)).

   procedure Eval_Op_Expon (N : Node_Id) is
      Left   : constant Node_Id := Left_Opnd (N);
      Right  : constant Node_Id := Right_Opnd (N);

   begin
      if Expression_Is_Not_Foldable (N, Left, Right) then
         Check_Non_Static_Context (Left);
         Check_Non_Static_Context (Right);
         return;
      end if;

      --  Fold exponentiation operation

      declare
         Right_Int : constant Uint := Expr_Value (Right);

      begin
         --  Integer case

         if Is_Integer_Type (Etype (Left)) then
            declare
               Left_Int : constant Uint := Expr_Value (Left);
               Result   : Uint;

            begin
               --  Exponentiation of an integer raises the exception
               --  Constraint_Error for a negative exponent (RM 4.5.6)

               if UI_Is_Negative (Right_Int) then
                  Constraint_Error_Warning (N, "integer exponent negative?!");
                  Set_Potentially_Static (N, True);
                  Check_Non_Static_Context (Left);
                  Check_Non_Static_Context (Right);
                  return;

               else
                  Result := UI_Exponentiate (Left_Int, Right_Int);

                  if Is_Modular_Integer_Type (Etype (N)) then
                     Result := UI_Mod (Result, Modulus (Etype (N)));
                  end if;

                  Fold_Uint (N, Result);
               end if;
            end;

         --  Real case

         else
            declare
               Left_Real : constant Ureal := Expr_Value (Left);

            begin
               --  Cannot have a zero base with a negative exponent

               if UI_Is_Negative (Right_Int)
                 and then UR_Is_Zero (Left_Real)
               then
                  Constraint_Error_Warning (N, "zero ** negative integer?!");
                  Set_Potentially_Static (N, True);
                  Check_Non_Static_Context (Left);
                  Check_Non_Static_Context (Right);
                  return;
               else
                  Fold_Ureal (N, UR_Exponentiate (Left_Real, Right_Int));
               end if;
            end;
         end if;
      end;

   end Eval_Op_Expon;

   -------------------------------
   -- Eval_Qualified_Expression --
   -------------------------------

   --  A qualified expression is potentially static if its subtype mark denotes
   --  a static subtype and its expression is potentially static (RM 4.9 (11)).

   procedure Eval_Qualified_Expression (N : Node_Id) is
      Operand     : Node_Id   := Expression (N);
      Target_Type : Entity_Id := Etype (N);

   begin
      if (not Is_Scalar_Type (Target_Type)
            and then not Is_String_Type (Target_Type))
        or else not Is_Static_Subtype (Target_Type)
        or else Expression_Is_Not_Foldable (N, Operand)
      then
         Check_Non_Static_Context (Operand);
         return;
      end if;

      --  Fold the result of qualification

      if Is_Discrete_Type (Target_Type) then
         Fold_Uint (N, Expr_Value (Operand));

      elsif Is_Real_Type (Target_Type) then
         Fold_Ureal (N, Expr_Value (Operand));

      else
         Fold_Str (N, Strval (Get_String_Val (Operand)));
      end if;

      if not Is_In_Range (N, Etype (N)) then
         Check_Non_Static_Context (Operand);
         Constraint_Error_Warning (N, "value out of range");
      end if;

   end Eval_Qualified_Expression;

   -----------------------
   -- Eval_Real_Literal --
   -----------------------

   --  Numeric literals are potentially static (RM 4.9(1))

   procedure Eval_Real_Literal (N : Node_Id) is
   begin
      Set_Potentially_Static (N);
   end Eval_Real_Literal;

   ------------------------
   -- Eval_Relational_Op --
   ------------------------

   --  Relational operations are static functions, so the result is potentially
   --  static if both operands are potentially static (RM 4.9(7), 4.9(20)).

   procedure Eval_Relational_Op (N : Node_Id) is
      Left      : constant Node_Id   := Left_Opnd (N);
      Right     : constant Node_Id   := Right_Opnd (N);
      Typ       : constant Entity_Id := Etype (Left);
      Result    : Boolean;

   begin
      if not Is_Scalar_Type (Typ)
        or else Expression_Is_Not_Foldable (N, Left, Right)
      then
         Check_Non_Static_Context (Left);
         Check_Non_Static_Context (Right);
         return;
      end if;

      --  Integer and Enumeration (discrete) type cases

      if Is_Discrete_Type (Typ) then
         declare
            Left_Int  : constant Uint := Expr_Value (Left);
            Right_Int : constant Uint := Expr_Value (Right);

         begin
            case Nkind (N) is
               when N_Op_Eq => Result := UI_Eq (Left_Int, Right_Int);
               when N_Op_Ne => Result := UI_Ne (Left_Int, Right_Int);
               when N_Op_Lt => Result := UI_Lt (Left_Int, Right_Int);
               when N_Op_Le => Result := UI_Le (Left_Int, Right_Int);
               when N_Op_Gt => Result := UI_Gt (Left_Int, Right_Int);
               when N_Op_Ge => Result := UI_Ge (Left_Int, Right_Int);

               when others => pragma Assert (False); null;
            end case;

            Fold_Uint (N, Test (Result));
         end;

      --  Real type case

      else
         pragma Assert (Is_Real_Type (Typ));

         declare
            Left_Real  : constant Ureal := Expr_Value (Left);
            Right_Real : constant Ureal := Expr_Value (Right);

         begin
            case Nkind (N) is
               when N_Op_Eq => Result := UR_Eq (Left_Real, Right_Real);
               when N_Op_Ne => Result := UR_Ne (Left_Real, Right_Real);
               when N_Op_Lt => Result := UR_Lt (Left_Real, Right_Real);
               when N_Op_Le => Result := UR_Le (Left_Real, Right_Real);
               when N_Op_Gt => Result := UR_Gt (Left_Real, Right_Real);
               when N_Op_Ge => Result := UR_Ge (Left_Real, Right_Real);

               when others => pragma Assert (False); null;
            end case;

            Fold_Uint (N, Test (Result));
         end;
      end if;

   end Eval_Relational_Op;

   ----------------
   -- Eval_Shift --
   ----------------

   --  Shift operations are intrinsic operations that can never be static,
   --  so the only processing required is to perform the required check for
   --  a non static context for the two operands.

   procedure Eval_Shift (N : Node_Id) is
   begin
      Check_Non_Static_Context (Left_Opnd (N));
      Check_Non_Static_Context (Right_Opnd (N));
   end Eval_Shift;

   ------------------------
   -- Eval_Short_Circuit --
   ------------------------

   --  A short circuit operation is potentially static if both operands
   --  are potentially static (RM 4.9 (13))

   procedure Eval_Short_Circuit (N : Node_Id) is
      Kind      : constant Node_Kind := Nkind (N);
      Left      : constant Node_Id   := Left_Opnd (N);
      Right     : constant Node_Id   := Right_Opnd (N);
      Left_Int  : Uint;

   begin
      --  Short circuit operations are never static in Ada 83

      if Ada_83 then
         return;
      end if;

      --  In Ada 9X, short circuit operations are potentially static if
      --  both operands are potentially static, and as usual we are all
      --  done if the result is not potentially static.

      Set_Potentially_Static (N,
        Potentially_Static (Left) and Potentially_Static (Right));

      if not Potentially_Static (N) then
         Check_Non_Static_Context (Left);
         Check_Non_Static_Context (Right);
         return;
      end if;

      --  Note that we did not use the usual call to Exprssion_Is_Not_Foldable
      --  in the above code. That's because short circuit operations are a
      --  special case, they can end up not raising constraint error even if
      --  the right operand can raise constraint error (and they can thus be
      --  static, even if they have such an operand). However if the left
      --  operand raises constraint error, then so does the result.

      if Raises_Constraint_Error (Left) then
         Set_Raises_Constraint_Error (N);
         Check_Non_Static_Context (Right);
         return;
      end if;

      --  It does not matter if the right operand raises constraint error if
      --  it will not be evaluated. So deal specially with the cases where
      --  the right operand is not evaluated. Note that we will fold these
      --  cases even if the right operand is non-static, which is fine, but
      --  of course in these cases the result is not potentially static.

      Left_Int := Expr_Value (Left);

      if (Kind = N_Op_And_Then and then Is_False (Left_Int))
        or else (Kind = N_Op_Or_Else and Is_True (Left_Int))
      then
         Fold_Uint (N, Left_Int);
         return;
      end if;

      --  If first operand not decisive, then it does matter if the right
      --  operand raises constraint error, since it will be evaluated

      if Raises_Constraint_Error (Right) then
         Set_Raises_Constraint_Error (N);
         Check_Non_Static_Context (Left);
         return;
      end if;

      --  Otherwise the result depends on the right operand

      Fold_Uint (N, Expr_Value (Right));
      return;

   end Eval_Short_Circuit;

   ----------------
   -- Eval_Slice --
   ----------------

   --  Slices can never be static, so the only processing required is to
   --  check for non-static context if an explicit range is given.

   procedure Eval_Slice (N : Node_Id) is
      Drange : constant Node_Id := Discrete_Range (N);

   begin
      if Nkind (Drange) = N_Range then
         Check_Non_Static_Context (Low_Bound (Drange));
         Check_Non_Static_Context (High_Bound (Drange));
      end if;
   end Eval_Slice;

   -------------------------
   -- Eval_String_Literal --
   -------------------------

   --  String literals are potentially static if the subtype is static
   --  (RM 4.9(2)). Note string literals are not static in Ada 83.

   procedure Eval_String_Literal (N : Node_Id) is
   begin
      if Is_Static_Subtype (Component_Type (Etype (N))) then
         if Ada_9X then
            Set_Potentially_Static (N);
         end if;
      end if;
   end Eval_String_Literal;

   --------------------------
   -- Eval_Type_Conversion --
   --------------------------

   --  A type conversion is potentially static if its subtype mark is for a
   --  static scalar subtype, and its operand expression is potentially static
   --  (RM 4.9 (10))

   procedure Eval_Type_Conversion (N : Node_Id) is
      Operand     : constant Node_Id   := Expression (N);
      Source_Type : constant Entity_Id := Etype (Operand);
      Target_Type : constant Entity_Id := Etype (N);

   begin
      if not Is_Scalar_Type (Target_Type)
        or else not Is_Static_Subtype (Target_Type)
        or else Expression_Is_Not_Foldable (N, Operand)
      then
         Check_Non_Static_Context (Operand);
         return;
      end if;

      --  Fold conversion, case of integer target type

      if Is_Integer_Type (Target_Type) then
         declare
            Result : Uint;

         begin
            if Is_Integer_Type (Source_Type) then
               Result := Expr_Value (Operand);
            else
               pragma Assert (Is_Real_Type (Source_Type));
               Result := UR_To_Uint (Expr_Value (Operand));
            end if;

            Fold_Uint (N, Result);
         end;

      --  Fold conversion, case of real target type

      elsif Is_Real_Type (Target_Type) then
         declare
            Result : Ureal;

         begin
            if Is_Real_Type (Source_Type) then
               Result := Expr_Value (Operand);
            else
               Result := UR_From_Uint (Expr_Value (Operand));
            end if;

            Fold_Ureal (N, Result);
         end;

      --  Enumeration types

      else
         Fold_Uint (N, Expr_Value (Operand));
      end if;

      if not Is_In_Range (N, Etype (N)) then
         Check_Non_Static_Context (Operand);
         Constraint_Error_Warning (N, "Value out of range");
      end if;

   end Eval_Type_Conversion;

   -------------------------------
   -- Eval_Unchecked_Conversion --
   -------------------------------

   --  Unchecked conversions can never be static, so the only required
   --  processing is to check for a non-static context for the operand.

   procedure Eval_Unchecked_Conversion (N : Node_Id) is
   begin
      Check_Non_Static_Context (Expression (N));
   end Eval_Unchecked_Conversion;

   -------------------
   -- Eval_Unary_Op --
   -------------------

   --  Predefined unary operators are static functions (RM 4.9(20)) and thus
   --  are potentially static if the operand is potentially static (RM 4.9(7))

   procedure Eval_Unary_Op (N : Node_Id) is
      Right : constant Node_Id := Right_Opnd (N);

   begin
      if Expression_Is_Not_Foldable (N, Right) then
         Check_Non_Static_Context (Right);
         return;
      end if;

      --  Fold for integer case

      if Is_Integer_Type (Etype (N)) then
         declare
            Rint   : constant Uint := Expr_Value (Right);
            Result : Uint;

         begin
            --  In the case of modular unary plus and abs there is no need
            --  to adjust the result of the operation since if the original
            --  operand was in bounds the result will be in the bounds of the
            --  modular type. However, in the case of modular unary minus the
            --  result may go out of the bounds of the modular type and needs
            --  adjustment.

            if Nkind (N) = N_Op_Plus then
               Result := Rint;

            elsif Nkind (N) = N_Op_Minus then
               if Is_Modular_Integer_Type (Etype (N)) then
                  Result := UI_Mod (UI_Negate (Rint), Modulus (Etype (N)));
               else
                  Result := UI_Negate (Rint);
               end if;

            else
               pragma Assert (Nkind (N) = N_Op_Abs);
               Result := UI_Abs (Rint);
            end if;

            Fold_Uint (N, Result);
         end;

      --  Fold for real case

      elsif Is_Floating_Point_Type (Etype (N)) then
         declare
            Rreal  : constant Ureal := Expr_Value (Right);
            Result : Ureal;

         begin
            if Nkind (N) = N_Op_Plus then
               Result := Rreal;

            elsif Nkind (N) = N_Op_Minus then
               Result := UR_Negate (Rreal);

            else
               Result := UR_Abs (Rreal);
            end if;

            Fold_Ureal (N, Result);
         end;
      end if;

   end Eval_Unary_Op;

   ----------------
   -- Expr_Value --
   ----------------

   --  Uint case (integers, characters, enumeration literals)

   function Expr_Value (N : Node_Id) return Uint is
      Kind : constant Node_Kind := Nkind (N);
      Ent  : Entity_Id;

   begin
      if Is_Entity_Name (N) then
         Ent := Entity (N);

         --  An enumeration literal that was either in the source or
         --  created as a result of static evaluation.

         if Ekind (Ent) = E_Enumeration_Literal then
            return Enumeration_Pos (Ent);

         --  A user defined static constant

         else
            pragma Assert (Ekind (Ent) = E_Constant);
            return Expr_Value (Constant_Value (Ent));
         end if;

      --  An integer literal that was either in the source or created
      --  as a result of static evaluation.

      elsif Kind = N_Integer_Literal then
         return Intval (N);

      else
         pragma Assert (Kind = N_Character_Literal);
         Ent := Entity (N);

         --  Since Character literals of type Standard.Character don't
         --  have any defining character literals built for them, they
         --  do not have their Entity set, so just use their Char
         --  code. Otherwise for user-defined character literals use
         --  their Pos value as usual.

         if No (Ent) then
            return UI_From_Int (Int (Char_Literal_Value (N)));
         else
            return Enumeration_Pos (Ent);
         end if;
      end if;

   end Expr_Value;

   --  Ureal case (fixed, float)

   function Expr_Value (N : Node_Id) return Ureal is
      Kind : constant Node_Kind := Nkind (N);
      Ent  : Entity_Id;

   begin
      if Kind = N_Identifier or else Kind = N_Expanded_Name then
         Ent := Entity (N);
         pragma Assert (Ekind (Ent) = E_Constant);
         return Expr_Value (Constant_Value (Ent));

      else
         pragma Assert (Kind = N_Real_Literal);
         return Realval (N);
      end if;
   end Expr_Value;

   --  Entity_Id case (enumeration)

   function Expr_Value (N : Node_Id) return Entity_Id is
      Kind : constant Node_Kind := Nkind (N);
      Ent  : constant Entity_Id := Entity (N);

   begin
      if Ekind (Ent) = E_Enumeration_Literal then
         return Ent;
      else
         pragma Assert (Ekind (Ent) = E_Constant);
         return Expr_Value (Constant_Value (Ent));
      end if;
   end Expr_Value;

   --------------------------------
   -- Expression_Is_Not_Foldable --
   --------------------------------

   --  One operand case

   function Expression_Is_Not_Foldable
     (N    : Node_Id;
      Op1  : Node_Id)
      return Boolean
   is
   begin
      Set_Potentially_Static (N, Potentially_Static (Op1));

      if not Potentially_Static (N) then
         return True;

      else
         Set_Raises_Constraint_Error (N, Raises_Constraint_Error (Op1));
         return Raises_Constraint_Error (N);
      end if;
   end Expression_Is_Not_Foldable;

   --  Two operand case

   function Expression_Is_Not_Foldable
     (N    : Node_Id;
      Op1  : Node_Id;
      Op2  : Node_Id)
      return Boolean
   is
   begin
      Set_Potentially_Static (N,
        Potentially_Static (Op1) and Potentially_Static (Op2));

      if not Potentially_Static (N) then
         return True;

      else
         Set_Raises_Constraint_Error (N,
           Raises_Constraint_Error (Op1) or Raises_Constraint_Error (Op2));
         return Raises_Constraint_Error (N);
      end if;
   end Expression_Is_Not_Foldable;

   --------------
   -- Fold_Str --
   --------------

   procedure Fold_Str (N : Node_Id; Val : String_Id) is
      Loc      : constant Source_Ptr := Sloc (N);
      Typ      : constant Entity_Id  := Etype (N);

   begin
      Rewrite_Substitute_Tree (N, Make_String_Literal (Loc, Strval => Val));
      Analyze (N);
      Resolve (N, Typ);
   end Fold_Str;

   ---------------
   -- Fold_Uint --
   ---------------

   procedure Fold_Uint (N : Node_Id; Val : Uint) is
      Loc         : constant Source_Ptr := Sloc (N);
      Typ         : constant Entity_Id  := Etype (N);
      Int_Literal : Node_Id;
      Lit         : Entity_Id;
      Pos         : Int;

   begin
      --  For a result of type integer, subsitute an N_Integer_Literal node
      --  for the result of the compile time evaluation of the expression.

      if Is_Integer_Type (Etype (N)) then
         Rewrite_Substitute_Tree (N, Make_Integer_Literal (Loc, Val));

      --  Otherwise we have an enumeration type, and we substitute either
      --  an N_Identifier or N_Character_Literal to represent the enumeration
      --  literal corresponding to the given value, which must always be in
      --  range, because appropriate tests have already been made for this.

      elsif Is_Enumeration_Type (Etype (N)) then
         Pos := UI_To_Int (Val);

         --  In the case where the literal is either of type Wide_Character
         --  or Character or of a type derived from them, there needs to be
         --  some special handling since there is no explicit chain of
         --  literals to search. Instead, an N_Character_Literal node is
         --  created with the appropriate Char_Code and Chars fields.

         if Root_Type (Etype (N)) = Standard_Character
           or else Root_Type (Etype (N)) = Standard_Wide_Character
         then
            Set_Character_Literal_Name (Char_Code (Pos));

            Rewrite_Substitute_Tree (N,
              Make_Character_Literal (Loc,
                Chars => Name_Find,
                Char_Literal_Value => Char_Code (Pos)));

         --  For all other cases, we have a complete table of literals, and
         --  we simply iterate through the chain of literal until the one
         --  with the desired position value is found.
         --

         else
            Lit := First_Literal (Base_Type (Etype (N)));
            for J in 1 .. Pos loop
               Lit := Next_Literal (Lit);
            end loop;

            Rewrite_Substitute_Tree (N, New_Occurrence_Of (Lit, Loc));
         end if;

      --  Anything other than an integer type or enumeration type is wrong

      else
         pragma Assert (False); null;
      end if;

      --  Note that we don't call Analyze on the node, since that would for
      --  one thing set Potentially_Static to True, which is not right.

      Set_Potentially_Static (N, True);
      Set_Etype              (N, Typ);
   end Fold_Uint;

   ----------------
   -- Fold_Ureal --
   ----------------

   procedure Fold_Ureal (N : Node_Id; Val : Ureal) is
      Loc      : constant Source_Ptr := Sloc (N);
      Typ      : constant Entity_Id  := Etype (N);

   begin
      Rewrite_Substitute_Tree (N, Make_Real_Literal (Loc, Realval => Val));
      Set_Potentially_Static  (N, True);
      Set_Etype               (N, Typ);

   end Fold_Ureal;

   ---------------
   -- From_Bits --
   ---------------

   function From_Bits (B : Bits) return Uint is
      T : Uint := Uint_0;

   begin
      for J in 0 .. B'Last loop
         if B (J) then
            T := UI_Sum (T, UI_Exponentiate (Uint_2, UI_From_Int (J)));
         end if;
      end loop;

      return T;
   end From_Bits;

   --------------------
   -- Get_String_Val --
   --------------------

   function Get_String_Val (N : Node_Id) return Node_Id is
   begin
      if Nkind (N) = N_String_Literal then
         return N;

      elsif Nkind (N) = N_Character_Literal then
         return N;

      else
         pragma Assert (Is_Entity_Name (N));
         return Get_String_Val (Constant_Value (Entity (N)));
      end if;
   end Get_String_Val;

   ---------------------------------
   -- Potentially_Static_Range --
   ---------------------------------

   --  A potentially static range is a range whose bounds are potentially
   --  static expressions, or a range_attribute_reference that is equivalent
   --  to such a range (RM 4.9(26)).

   function Potentially_Static_Range (N : Node_Id) return Boolean is
   begin
      pragma Assert (Nkind (N) = N_Range);
      return Potentially_Static (Low_Bound (N))
        and Potentially_Static (High_Bound (N));
   end Potentially_Static_Range;

   -----------------
   -- Is_In_Range --
   -----------------

   function Is_In_Range (N : Node_Id; Typ : Entity_Id) return Boolean is
      Val : Uint;

   begin
      --  Universal types have no range limites, so always in range

      if Typ = Universal_Integer or else Typ = Universal_Real then
         return True;

      --  For now, all real types are considered to be in range, TBSL ???

      elsif Is_Real_Type (Typ) then
         return True;

      --  For discrete types, do the check against the bounds

      elsif Is_Discrete_Type (Typ) then
         Val := Expr_Value (N);

         return UI_Le (Expr_Value (Type_Low_Bound (Typ)), Val)
           and then UI_Le (Val, Expr_Value (Type_High_Bound (Typ)));

      --  For all other types, must be in range

      else
         return True;
      end if;
   end Is_In_Range;

   --------------------------
   -- Is_Static_Expression --
   --------------------------

   --  A static expression is a potentially static expression that does
   --  not propagate an exception upon (static) evaluation (RM 4.9(2))

   function Is_Static_Expression (N : Node_Id) return Boolean is
   begin
      return Potentially_Static (N) and not Raises_Constraint_Error (N);
   end Is_Static_Expression;

   -----------------------
   -- Is_Static_Subtype --
   -----------------------

   --  A static subtype is either a scalar base type, other than a generic
   --  formal type; or a scalar subtype formed by imposing on a static
   --  subtype either a static range constraint, or a floating or fixed
   --  point constraint whose range constraint, if any, is static. [LRM 4.9]

   --  Is this definition right???

   function Is_Static_Subtype (Typ : Entity_Id) return Boolean is
      Base_T : constant Entity_Id := Base_Type (Typ);

   begin
      if Is_Generic_Type (Base_T) or else not Is_Scalar_Type (Base_T) then
         return False;

      elsif Base_T = Typ then
         return True;

      else
         return Is_Static_Subtype (Base_T)
           and then Potentially_Static (Type_Low_Bound (Typ))
           and then Potentially_Static (Type_High_Bound (Typ));
      end if;
   end Is_Static_Subtype;

   ----------
   -- Test --
   ----------

   function Test (Cond : Boolean) return Uint is
   begin
      if Cond then
         return Uint_1;
      else
         return Uint_0;
      end if;
   end Test;

   --------------
   -- To_Bits --
   --------------

   procedure To_Bits (U : Uint; B : out Bits) is
   begin
      for J in 0 .. B'Last loop
         B (J) := not UI_Is_Zero (UI_Mod (UI_Quotient (U,
           UI_Exponentiate (Uint_2, UI_From_Int (J))), Uint_2));
      end loop;
   end To_Bits;

end Sem_Eval;


----------------------
-- REVISION HISTORY --
----------------------

--  ----------------------------
--  revision 1.112
--  date: Wed Aug 10 05:16:13 1994;  author: dewar
--  Set Raises_Constraint_Error on static values that are out of range. This
--   avoids some cases of duplicated error messages
--  ----------------------------
--  revision 1.113
--  date: Fri Aug 12 14:22:45 1994;  author: banner
--  (Fold_Uint): Modify check for Standard_Character and Wide_Character to
--   use the Root_Type since derivations from these types need to be
--   treated in a similar manner.
--  ----------------------------
--  revision 1.114
--  date: Wed Aug 24 19:37:29 1994;  author: dewar
--  (Check_Non_Static_Context): Remove replacement by lower bound, no longer
--   needed now we generate proper N_Constraint_Error nodes.
--  ----------------------------
--  New changes after this line.  Each line starts with: "--  "
