/* Input line implementation
   Copyright (C) 1994 Miguel de Icaza
   
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  */

#include <ncurses.h>
#include <ctype.h>
#include <sys/types.h>
#include <string.h>
#include <malloc.h>
#include "input.h"
#include "util.h"
#include "global.h"
#include "main.h"

static char rcsid [] = "$Id: input.c,v 1.13 1994/10/13 04:27:37 miguel Exp $";

Input *create_input (int x, int y, WINDOW *win, int color, int field_len,
		     char *def_text)
{
    Input *in = xmalloc (sizeof (Input), "create_input: in");
    
    in->start_x = x;
    in->start_y = y;
    in->current_max_len = field_len+1;
    in->window = win;
    in->buffer = xmalloc (field_len+1, "create_input: in->buffer");
    in->color = color;
    in->field_len = field_len;
    in->first = 1;
    in->first_shown = 0;
    in->mark = 0;
    in->kill_save = 0;
    in->need_push = 1;
    leaveok (win, FALSE);

    /* history setup */
    in->history = xmalloc (sizeof (Hist), "create_input: in->history");
    in->history->next = 0;
    in->history->prev = 0;
    in->history->text = strdup ("");
    strcpy (in->buffer, def_text);
    in->point = strlen (in->buffer);
    update_input (in);
    in->first = 1;
    return in;
}

void destroy_input (Input *in, int flags)
{
    if (!in){
	fprintf (stderr, "Internal error: null Input *\n");
	exit (1);
    }
    if (in->kill_save)
	free (in->kill_save);
    if (in->buffer)
	if (flags != IN_KEEP_BUFFER)
	    free (in->buffer);
    free (in);
}

void update_input (Input *in)
{
    int    x = in->start_x;
    int    y = in->start_y;
    WINDOW *w = in->window;
    int    i, j;
    int    buf_len = strlen (in->buffer);

    if ((in->point < in->first_shown)
 	|| (in->point >= in->first_shown+in->field_len))
	in->first_shown = (in->point / in->field_len) * in->field_len;

    /* Adjust the mark */
    if (in->mark > buf_len)
	in->mark = buf_len;
    wstandend (w);
    wattron (w, in->color);
    wmove (w, y, x);
    for (i = 0; i < in->field_len; i++)
	waddch (w, ' ');
    wmove (w, y, x);
    for (i = 0, j = in->first_shown; i < in->field_len && in->buffer [j]; i++)
	waddch (w, in->buffer [j++]);
    wmove (w, y, x + (in->point - in->first_shown));
    wrefresh (w);
    in->first = 0;
}

void push_history (Input *in, char *text)
{
    Hist *new;

    new = xmalloc (sizeof (Hist), "push_history");

    if (in->history){
	while (in->history->next)
	    in->history = in->history->next;
	in->history->next = new;
    }
    in->need_push = 1;
    new->next = 0;
    new->prev = in->history;
    new->text = strdup (text);
    in->history = new;
}

/* Cleans the input line and adds the current text to the history */
void new_input (Input *in)
{
    push_history (in, in->buffer);
    in->buffer [0] = 0;
    in->point = 0;
    in->mark = 0;
    update_input (in);
}

/*                                    */
/* Private routines for Input objects */
/*                                    */

static void insert_char (Input *in, int c_code)
{
    int i;

    if (strlen (in->buffer)+1 == in->current_max_len){
	/* Expand the buffer */
	char *narea = realloc(in->buffer, in->current_max_len + in->field_len);
	if (narea){
	    in->buffer = narea;
	    in->current_max_len += in->field_len;
	}
    }
    if (strlen (in->buffer)+1 < in->current_max_len){
	int l = strlen (&in->buffer [in->point]);
	for (i = l+2; i > 0; i--)
	    in->buffer [in->point+i] = in->buffer [in->point+i-1];
	in->buffer [in->point] = c_code;
	in->point++;
    }
}

static void beginning_of_line (Input *in)
{
    in->point = 0;
}

static void end_of_line (Input *in)
{
    in->point = strlen (in->buffer);
}

static void backward_char (Input *in)
{
    if (in->point)
	in->point--;
}

static void forward_char (Input *in)
{
    if (in->buffer [in->point])
	in->point++;
}

static void forward_word (Input *in)
{
    char *p = in->buffer+in->point;

    while (*p && isspace (*p))
	p++;
    while (*p && isalnum (*p))
	p++;
    in->point = p - in->buffer;
}

static void backward_word (Input *in)
{
    char *p = in->buffer+in->point;

    while (p-1 > in->buffer-1 && (isspace (*(p-1)) || ispunct (*(p-1))))
	p--;
    while (p-1 > in->buffer-1 && isalnum (*(p-1)))
	p--;
    in->point = p - in->buffer;
}

static void backward_delete (Input *in)
{
    int i;
    
    if (!in->point)
	return;
    for (i = in->point; in->buffer [i-1]; i++)
	in->buffer [i-1] = in->buffer [i];
    in->point--;
}

static void delete_char (Input *in)
{
    int i;

    for (i = in->point; in->buffer [i]; i++)
	in->buffer [i] = in->buffer [i+1];
}

static void copy_region (Input *in, int x_first, int x_last)
{
    int first = min (x_first, x_last);
    int last  = max (x_first, x_last);
    
    if (last == first)
	return;
    
    if (in->kill_save)
	free (in->kill_save);
    
    in->kill_save = malloc (last-first + 1);
    strncpy (in->kill_save, in->buffer+first, last-first);
    in->kill_save [last-first] = 0;
}

static void delete_region (Input *in, int x_first, int x_last)
{
   int first = min (x_first, x_last);
   int last  = max (x_first, x_last);

   in->point = first;
   in->mark  = first;
   strcpy (&in->buffer [first], &in->buffer [last]);
}

static void kill_word (Input *in)
{
    int old_point = in->point;
    int new_point;

    forward_word (in);
    new_point = in->point;
    in->point = old_point;

    copy_region (in, old_point, new_point);
    delete_region (in, old_point, new_point);
}

static void back_kill_word (Input *in)
{
    int old_point = in->point;
    int new_point;

    backward_word (in);
    new_point = in->point;
    in->point = old_point;

    copy_region (in, old_point, new_point);
    delete_region (in, old_point, new_point);
}

static void set_mark (Input *in)
{
    in->mark = in->point;
}

static void kill_save (Input *in)
{
    copy_region (in, in->mark, in->point);
}

static void kill_region (Input *in)
{
    kill_save (in);
    delete_region (in, in->point, in->mark);
}

static void yank (Input *in)
{
    char *p;
    
    for (p = in->kill_save; *p; p++)
	insert_char (in, *p);
}

static void kill_line (Input *in)
{
    if (in->kill_save)
	free (in->kill_save);
    in->kill_save = strdup (&in->buffer [in->point]);
    in->buffer [in->point] = 0;
}

static void assign_text (Input *in, char *text)
{
    in->buffer = strdup (in->history->text);
    in->current_max_len = strlen (in->buffer)+1;
    in->point = strlen (in->buffer);
    in->mark = 0;
}

static void hist_prev (Input *in)
{
    if (!in->history)
	return;

    if (!in->history->prev)
	return;

    if (in->need_push){
	push_history (in, in->buffer);
	in->need_push = 0;
    }
    free (in->history->text);
    in->history->text = strdup (in->buffer);
    in->history = in->history->prev;
    assign_text (in, in->history->text);
}

static void hist_next (Input *in)
{
    if (!in->history)
	return;

    if (!in->history->next)
	return;
    
    free (in->history->text);
    in->history->text = strdup (in->buffer);
    in->history = in->history->next;
    assign_text (in, in->history->text);
}

static struct {
    int key_code;
    void (*fn)(Input *in);
} input_map [] = {
    /* Motion */
    XCTRL('a'),   beginning_of_line,
    XCTRL('e'),   end_of_line,
    KEY_LEFT,     backward_char,
    XCTRL('b'),   backward_char,
    ALT('b'),     backward_word,
    KEY_RIGHT,    forward_char,
    XCTRL('f'),   forward_char,
    ALT('f'),     forward_word,

    /* Editing */
    0177,         backward_delete,
    KEY_BACKSPACE,backward_delete,
    XCTRL('h'),   backward_delete,
    KEY_DC,       delete_char,
    XCTRL('d'),   delete_char,
    ALT('d'),     kill_word,
    ALT(KEY_BACKSPACE), back_kill_word,
    ALT(XCTRL('h')), back_kill_word,
    
    /* Region manipulation */
    0,            set_mark,
    XCTRL('w'),   kill_region,
    ALT('w'),     kill_save,
    XCTRL('y'),   yank,
    XCTRL('k'),   kill_line,
    
    /* History */
    ALT('p'),     hist_prev,
    ALT('n'),     hist_next,
    
    0,            0,
};

/* Inserts text in input line */
void stuff (Input *in, char *text, int insert_extra_space)
{
    while (*text)
	handle_char (in, *text++);
    if (insert_extra_space)
	handle_char (in, ' ');
}

void handle_char (Input *in, int c_code)
{
    int    i;

    if (quote){
	insert_char (in, c_code);
	update_input (in);
	return;
    }

    for (i = 0; input_map [i].fn; i++){
	if (c_code == input_map [i].key_code){
	    (*input_map [i].fn)(in);
	    break;
	}
    }
    if (!input_map [i].fn){
	if (c_code > 127 || c_code < ' ')
	    return;
	if (in->first){
	    *in->buffer = 0;
	    in->point = 0;
	}
	insert_char (in, c_code);
    }
    update_input (in);
}
