/* Pulldown menu code.
   Copyright (C) 1994 Miguel de Icaza.
   
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  */
#include <ncurses.h>
#include <string.h>
#include <stdarg.h>
#include <sys/types.h>
#include <ctype.h>
#include "menu.h"
#include "dialog.h"
#include "global.h"
#include "color.h"
#include "util.h"
#include "main.h"
#include "input.h"
#include "mouse.h"

static char rcsid [] = "$Header: /usr/users/miguel/c/CVS/nc/menu.c,v 1.10 1994/09/01 23:56:27 miguel Exp $";

extern int is_right;
static int event_set;		/* Used to pass information from the */
				/* mouse handler */

#define WAS_HANDLED -1
#define QUIT_MENU   -2
#define BIG_NUMBER  1000

Menu *create_menu (char *name, menu_entry *entries, int count)
{
    Menu *menu;

    menu = (Menu *) xmalloc (sizeof (Menu), "create_menu");
    menu->count = count;
    menu->max_entry_len = 0;
    menu->entries = entries;
    menu->name = name;
    return menu;
}

static void paint_idx (Menu *menu, WINDOW *w, int idx, int sel, int len)
{
    wstandend (w);
    if (sel)
	wattron (w, A_BOLD);
    else
	wattron (w, MENU_ENTRY_COLOR);
    wmove (w, idx+1, 1);
    wprintw (w, " %-*s ", len, menu->entries [idx].text);
}

#ifdef use_mouse
int menu_handler (Gpm_Event *event, Menu *menu)
{
    int pos;
    
    pos = event->y;
    
    if (!(pos >= 0 && pos < menu->count && *menu->entries [pos].text))
	return 0;
    
    if (event->type & (GPM_DOWN|GPM_DRAG)){
	paint_idx (menu, menu->win, menu->selected, 0, menu->max_entry_len);
	menu->selected = pos;
	paint_idx (menu, menu->win, menu->selected, 1, menu->max_entry_len);
	wrefresh (menu->win);
	return MOU_NORMAL;
    }
    if (event->type & GPM_UP){
	if (menu->entries [menu->selected].call_back){
	    (*menu->entries [menu->selected].call_back)(0);
	    event_set = WAS_HANDLED;
	}
	return MOU_ENDLOOP;
    }
    return MOU_NORMAL;
}
#endif

int run_menu (Menu *menu, int y, int x)
{
    WINDOW *w;			/* handle to the window's menu */
    int i;
    int c;
    int max_entry_len = 0;

    /* setup */
    for (i = 0; i < menu->count; i++)
	max_entry_len = max (max_entry_len, strlen (menu->entries [i].text));
    menu->max_entry_len = max_entry_len = max (max_entry_len, 20);
    menu->selected = 0;
    
    /* Draw the nice box */
    menu->win = w = newwin (menu->count+2, max_entry_len+4, y, x);
    leaveok (w, TRUE);
    wstandend (w);
    wattron (w, SELECTED_COLOR);
    werase (w);
    box (w, ACS_VLINE, ACS_HLINE);
    wstandend (w);

    /* Draw the nice contents */
    for (i = 0; i < menu->count; i++)
	paint_idx (menu, w, i, 0, max_entry_len);

    /* Select the default entry */
    paint_idx (menu, w, 0, 1, max_entry_len);
    wrefresh (w);

    /* Now the mouse handler */
    push_event (x+2, y+2, x+max_entry_len+2, y+menu->count+2,
		(mouse_h) menu_handler, menu);

    while ((c = mi_getch ()) != '\e'){
	switch (c){
	case KEY_UP:
	case XCTRL('p'):		/* C-p */
	    paint_idx (menu, w, menu->selected, 0, max_entry_len);
	    menu->selected--;
	    if (menu->selected < 0)
		menu->selected = menu->count-1;
	    if (!menu->entries [menu->selected].call_back)
		menu->selected--;
	    break;

	case KEY_DOWN:
	case XCTRL('n'):		/* C-n */
	    paint_idx (menu, w, menu->selected, 0, max_entry_len);
	    menu->selected++;
	    if (menu->selected == menu->count)
		menu->selected = 0;
	    if (!menu->entries [menu->selected].call_back)
		menu->selected++;
	    break;

	case '\n':
	    if (menu->entries [menu->selected].call_back){
		(*menu->entries [menu->selected].call_back)(0);
	    }
	    pop_event ();
	    return 0;
	    
	case KEY_LEFT:
	case XCTRL('b'):
	    untouch_bar ();
	    pop_event ();
	    return -1;

	case KEY_RIGHT:
	case XCTRL('f'):
	    untouch_bar ();
	    pop_event ();
	    return 1;

	case ' ':
	    break;

	case -1:
	    untouch_bar ();
	    pop_event ();
	    return QUIT_MENU;
	    
	default:
	    for (i = 0; i < menu->count; i++){
		if (tolower (menu->entries [i].hot_key) == tolower (c)){
		    if (menu->entries [i].call_back)
			(*menu->entries [i].call_back)(0);
		    return 0;
		}
	    }
	}
	paint_idx (menu, w, menu->selected, 1, max_entry_len);
	wrefresh (w);
    }
    refresh_screen ();
    pop_event ();
    return 0;
}

/* This is an ugly patch */
int was_enter = 0;

/* Returns -1 on left movement, 1 on right movement, 0 when selecting,
   QUIT_MENU when event_set contains the selected entry from *Menus [] */

int get_motion (Menu *Menus [], int top, int *abort)
{
    int  c, i;
    char *j;			/* Used for scanning the first letter */
    
    was_enter = 0;
    *abort = 0;
    
    while ((c = mi_getch ()) != '\e'){
	switch (c){
	case '\n':
	    return 0;
	    
	case KEY_LEFT:
	case 2:			/* C-b */
	    return -1;

	case KEY_RIGHT:
	case '\t':
	case ' ':
	case 6:
	    return 1;

	case -1:
	    if (was_enter)
		return 0;
	    return QUIT_MENU;

	default:
	    for (i = 0; i < top; i++){
		for (j = Menus [i]->name; *j; j++)
		    if (*j != ' ')
			break;
		if (!*j)
		    break;
		if (tolower (*j) == tolower (c)){
		    event_set = i;
		    return QUIT_MENU;
		}
	    }
	}
    }
    *abort = 1;
    return 0;
}

static void paint_bar (WINDOW *bar, int pos, int color, char *name)
{
    wmove (bar, 0, pos);
    wstandend (bar);
    wattron (bar, color);
    wprintw (bar, name);
}

/* Ugly hack */
int current = 0;

#ifdef use_mouse
int top_menu_handle (Gpm_Event *event, void *x)
{
    if (event->type & (GPM_DOWN|GPM_DRAG)){
	event_set = (int) x;
	if (event_set != current)
	    return MOU_ENDLOOP;
	return MOU_NORMAL;
    }
    return MOU_NORMAL;
}

int send_yes_event (Gpm_Event *event, void *x)
{
    if ((event->type & GPM_DOWN) && (event->buttons & GPM_B_RIGHT)){
	was_enter = 1;
	return MOU_ENDLOOP;
    } else
	return MOU_NORMAL;
}

int quit_all_event (Gpm_Event *event, void *x)
{
    if (event->type & GPM_DOWN){
	event_set = QUIT_MENU;
	return MOU_ENDLOOP;
    }
    return MOU_NORMAL;
}
#endif

/* run_bar.
   WINDOW *parent, the parent window.
   int    x, y,    the starting coordinates for the menu
   int    width    the width of the menu
   int    space    inter-item spacing
   int    items    number of items contained in *Menus []
   int    first    item that is initially highlighted.
   Menu   *Menus[] Array with the menus.
   int    flags    0 or BAR_FILL. If BAR_FILL then it clears the menu before
                   displaying the menu.
   int    color_sel,
          color_uns, Colors used for the selected item and the unselected item

   if Menus [0]->count is 0, it's assumed to be a radio-button like control.
*/
int run_bar (WINDOW *parent, int x, int y, int width, int space, int items,
	     int first, Menu *Menus [], int flags, int color_sel,int color_uns)
{
    int i;
    int abort;
    WINDOW *bar;
    int is_bar;
    int fill  = (flags & BAR_FILL);

    current = 0;
    
    if (first < items && first >= 0)
	current = first;
    bar = derwin (parent, 1, width, y, x);
    leaveok (bar, TRUE);
    wstandend (bar);
    wattron (bar, color_uns);
    if (fill)
	wclr (bar);

    is_bar = Menus [current]->count == 0;
    if (is_bar){
	push_event (1, 1, COLS, LINES, send_yes_event, 0);
    } else {
	push_event (1, 1, COLS, LINES, quit_all_event, 0);
    }
    
    for (i = 0; i < items; i++){
	wmove (bar, 0, i*space+2);
	wprintw (bar, "%s", Menus [i]->name);
	push_event (i*space+2+x, y+1,i*space+2+x+strlen (Menus [i]->name), y+1,
		    top_menu_handle, (void *) i);
    }
    do {
	paint_bar (bar, current*space+2, color_sel, Menus [current]->name);
	touchwin (bar);
	wrefresh (bar);
	is_right = current;
	event_set = current;
	if (is_bar)
	    i = get_motion (Menus, items, &abort);
	else {
	    i = run_menu (Menus [current], 1, current*space);
	}
	paint_bar (bar, current*space+2, color_uns, Menus [current]->name);

	if (i == QUIT_MENU){
	    /* it means that the event was handled by a mouse handler */
	    /* if event_set is == -1 then set i = 0 to exit the loop */
	    if (event_set == WAS_HANDLED)
		i = event_set = 0;
	    else if (event_set == QUIT_MENU)
		break;
	    else {
		/* Else, event_set has the position that was activated */
		current = event_set;
		if (is_bar)
		    break;
	    }
	} else {
	    current += i;
	    if (current < 0) current = items -1;
	    current %= items;
	}
    } while (i);
    refresh_screen ();
    delwin (bar);
    if (abort)
	return -1;
    return current;
}

static Menu *query_arr [MAXQUERY];

int query_dialog (char *header, char *text, int flags, int count, ...)
{
    va_list ap;
    int len, lines;
    int i, result;
    int options_len;
    int color_sel, color_uns;
    
    va_start (ap, count);
    len = max (strlen (header), msglen (text, &lines));
    create_dialog (len, lines+2, header, text, flags & BAR_ERROR);

    if (count > MAXQUERY)
	return -1;		/* XXX Check this */

    for (i = options_len = 0; i < count; i++){
	query_arr [i] = (Menu *) xmalloc (sizeof (Menu), "query_dialog");
	query_arr [i]->name = va_arg (ap, char *);
	query_arr [i]->count = 0;
	options_len += strlen (query_arr [i]->name) + 2;
    }
    va_end (ap);
    
    color_sel = (flags & BAR_ERROR) ? REVERSE_COLOR : Q_SELECTED_COLOR;
    color_uns = (flags & BAR_ERROR) ? ERROR_COLOR : Q_UNSELECTED_COLOR;
    
    result = run_bar (top_window,
	     2 + (len - options_len)/2, /* Start X pos */
	     lines+3,		/* Start Y pos */
	     options_len,	/* Width */
	     7,			/* inner space */
	     count,		/* items */
	     0,                 /* First selected menu entry */
	     query_arr, flags, color_sel, color_uns);
    destroy_dialog ();
    for (i = 0; i < count; i++)
	free (query_arr [i]);
    return result;
}
