/*
 *      Copyright (C) 1993 Bas Laarhoven.

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; see the file COPYING.  If not, write to
the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

 $Source: /usr/src/distr/ftape-0.9.9d/RCS/kernel-interface.c,v $
 $Author: bas $
 *
 $Revision: 1.16 $
 $Date: 1994/01/16 17:20:10 $
 $State: ALPHA $
 *
 *      This file contains the code that interfaces the kernel
 *      for the QIC-40/80 floppy-tape driver for Linux.
 */

static char RCSid[] =
"$Id: kernel-interface.c,v 1.16 1994/01/16 17:20:10 bas ALPHA bas $";


#include <linux/errno.h>
#include <linux/fs.h>
#include <asm/segment.h>
#include <linux/kernel.h>
#include <linux/signal.h>
#include <linux/major.h>
#include <linux/module.h>

#include "ftape-rw.h"
#include "ftape-io.h"
#include "kernel-interface.h"
#include "kernel-version.h"

/*      Global vars.
 */
char *tape_buffer = NULL;

/*      Local vars.
 */
static int busy_flag = 0;
static int old_sigmask;

static int ftape_open( struct inode* ino, struct file* filep);
static void ftape_close( struct inode* ino, struct file* filep);
static int ftape_ioctl( struct inode* ino, struct file* filep, 
                       unsigned int command, unsigned long arg);
static int ftape_read( struct inode* ino, struct file* fp, char* buff, int req_len);
static int ftape_write( struct inode* ino, struct file* fp, char* buff, int req_len);
#if 0
static int ftape_select( void);
static int ftape_mmap( int dev, unsigned off, int prot);
#else
# define ftape_select NULL
# define ftape_mmap NULL
#endif

static struct file_operations ftape_cdev = {
  NULL,				/* lseek */
  ftape_read, 			/* read */
  ftape_write,			/* write */
  NULL,				/* readdir */
  ftape_select, 		/* select */
  ftape_ioctl,			/* ioctl */
  ftape_mmap, 		        /* mmap */
  ftape_open,			/* open */
  ftape_close,			/* release */
  NULL,                         /* fsync */
};

#define SIG_BLOCK_ALL   (0xffffffff)

/*      Called by modules package when installing the driver
 */
int
init_module( void) {

  printk( "ftape v0.9.9d 16/1/94 Bas Laarhoven (bas@vimec.nl)\n"
         " QIC-117 driver for QIC-40 and QIC-80 tape drives\n"
         );
  TRACE( 3, "init_module", "installing QIC-117 ftape driver...");
  if (register_chrdev( QIC117_TAPE_MAJOR, "mt", &ftape_cdev)) {
    TRACE( 1, "init_module", "register_chrdev failed");
    return -EIO;
  }
  /*    Get address of tape buffer, already aligned in kernel.
   */
  tape_buffer = (char*) (((long) ftape_big_buffer +
                          (BUFF_SIZE - 1)) & ~(BUFF_SIZE - 1));
  TRACEx2( 3, "init_module", "tape_buffers (%d) @ %p",
          NR_FTAPE_BUFFERS, tape_buffer);
  busy_flag = 0;
  ftape_unit = -1;
  ftape_failure = 1;            /* inhibit any operation but open */
  udelay_calibrate();           /* must be before fdc_wait_calibrate ! */
  fdc_wait_calibrate();
  ftape_new_cartridge();        /* init some tape related variables */
  return 0;
}

/*      Called by modules package when removing the driver
 */
void
cleanup_module( void) {

  if (unregister_chrdev( QIC117_TAPE_MAJOR, "mt") != 0) {
    TRACE( 3, "cleanup_module", "failed");
  } else {
    TRACE( 3, "cleanup_module", "successfull");
  }
}

/*      Open ftape device
 */
int
ftape_open( struct inode* ino, struct file* filep)
{
  int result;
  MOD_INC_USE_COUNT;            /* lock module in memory */

  TRACEi( 5, "ftape_open", "called for minor", MINOR( ino->i_rdev));
  if (busy_flag) {
    TRACE( 1, "ftape_open", "failed: already busy");
    return -EBUSY;
  }
  ftape_unit = MINOR( ino->i_rdev);
  if ((ftape_unit & ~FTAPE_NO_REWIND) > 1) {
    TRACE( 1, "ftape_open", "failed: illegal unit nr");
    return -ENXIO;
  }
  ftape_failure = 0;            /* allow tape operations */
  old_sigmask = current->blocked;
  current->blocked = SIG_BLOCK_ALL;
  result = _ftape_open();
  if (result < 0) {
    TRACE( 1, "ftape_open", "_ftape_open failed");
    current->blocked = old_sigmask; /* restore mask */
    return result;
  } else {
    busy_flag = 1;
    current->blocked = old_sigmask | _S(SIGPIPE);
    return 0;
  }
}

/*      Close ftape device
 */
void
ftape_close( struct inode* ino, struct file* filep)
{
  int result;

  TRACE( 5, "ftape_close", "called");
  if (!busy_flag || MINOR( ino->i_rdev) != ftape_unit) {
    TRACE( 1, "ftape_close", "failed: not busy or wrong unit");
    return;                     /* keep busy_flag !(?) */
  }
  current->blocked = SIG_BLOCK_ALL;
  result = _ftape_close();
  if (result < 0) {
    TRACE( 1, "ftape_close", "_ftape_close failed");
  }
  ftape_failure = 1;            /* inhibit any operation but open */
  busy_flag = 0;
  current->blocked = old_sigmask; /* restore before open state */
  MOD_DEC_USE_COUNT;            /* unlock module in memory */
}

/*      Ioctl for ftape device
 */
int
ftape_ioctl( struct inode* ino, struct file* filep, 
            unsigned int command, unsigned long arg)
{
  int result = -EIO;
  int old_sigmask;

  TRACElx( 5, "ftape_ioctl", "called with code:", command);
  if (!busy_flag || MINOR( ino->i_rdev) != ftape_unit || ftape_failure) {
    TRACE( 1, "ftape_ioctl", "failed: not busy, failure or wrong unit");
    return -EIO;
  }
  old_sigmask = current->blocked; /* save mask */
  current->blocked = SIG_BLOCK_ALL;
  /* This will work as long as sizeof( void*) == sizeof( long)
   */
  result = _ftape_ioctl( command, (void*) arg);
  current->blocked = old_sigmask; /* restore mask */
  return result;
}

/*      Read from tape device
 */
int
ftape_read( struct inode* ino, struct file* fp, char* buff, int req_len)
{
  int result = -EIO;
  int old_sigmask;

  TRACEi( 5, "ftape_read", "called with count:", req_len);
  if (!busy_flag || MINOR( ino->i_rdev) != ftape_unit || ftape_failure) {
    TRACE( 1, "ftape_read", "failed: not busy, failure or wrong unit");
    return -EIO;
  }
  old_sigmask = current->blocked; /* save mask */
  current->blocked = SIG_BLOCK_ALL;
  result = _ftape_read( buff, req_len);
  TRACEi( 7, "ftape_read", "return with count:", result);
  current->blocked = old_sigmask; /* restore mask */
  return result;
}

/*      Write to tape device
 */
int
ftape_write( struct inode* ino, struct file* fp, char* buff, int req_len)
{
  int result = -EIO;
  int old_sigmask;

  TRACEi( 5, "ftape_write", "called with count:", req_len);
  if (!busy_flag || MINOR( ino->i_rdev) != ftape_unit || ftape_failure) {
    TRACE( 1, "ftape_write", "failed: not busy, failure or wrong unit");
    return -EIO;
  }
  old_sigmask = current->blocked; /* save mask */
  current->blocked = SIG_BLOCK_ALL;
  result = _ftape_write( buff, req_len);
  TRACEi( 7, "ftape_write", "return with count:", result);
  current->blocked = old_sigmask; /* restore mask */
  return result;
}
