/* Pulldown menu code.
   Copyright (C) 1994 Miguel de Icaza.
   
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  */
#include <ncurses.h>
#include <string.h>
#include <stdarg.h>
#include <sys/types.h>
#include <ctype.h>
#include "menu.h"
#include "dialog.h"
#include "global.h"
#include "color.h"
#include "util.h"

static char rcsid [] = "$Header: /usr/users/miguel/c/CVS/nc/menu.c,v 1.1.1.1 1994/05/06 19:14:23 miguel Exp $";

extern int is_right;

Menu *create_menu (char *name, menu_entry *entries, int count)
{
    Menu *menu;

    menu = (Menu *) xmalloc (sizeof (Menu), "create_menu");
    menu->count = count;
    menu->max_entry_len = 0;
    menu->entries = entries;
    menu->name = name;
    return menu;
}

static void paint_idx (Menu *menu, WINDOW *w, int idx, int sel)
{
    wstandend (w);
    if (sel)
	wattron (w, A_BOLD);
    else
	wattron (w, MENU_ENTRY_COLOR);
    wmove (w, idx+1, 1);
    wprintw (w, " %-20s", menu->entries [idx].text);
}

int run_menu (Menu *menu, int x, int y)
{
    WINDOW *w;			/* handle to the window's menu */
    int i;
    int c;
    int selected = 0;
    int max_entry_len = 0;

    for (i = 0; i < menu->count; i++)
	max_entry_len = max (max_entry_len, strlen (menu->entries [i].text));
    max_entry_len = max (max_entry_len, 20);
    w = newwin (menu->count+2, max_entry_len+4, x, y);
    leaveok (w, TRUE);
    wstandend (w);
    wattron (w, SELECTED_COLOR);
    werase (w);
    box (w, ACS_VLINE, ACS_HLINE);
    wstandend (w);
    for (i = 0; i < menu->count; i++){
	wmove (w, i+1, 1);
	wattron (w, MENU_ENTRY_COLOR);
	wprintw (w, " %-20s ", menu->entries [i].text);
    }
    paint_idx (menu, w, 0, 1);
    wrefresh (w);
    while ((c = getch ()) != '\e'){
	switch (c){
	case KEY_UP:
	case 16:		/* C-p */
	    paint_idx (menu, w, selected, 0);
	    selected--;
	    if (selected < 0)
		selected = menu->count-1;
	    if (!menu->entries [selected].call_back)
		selected--;
	    break;

	case KEY_DOWN:
	case 14:		/* C-n */
	    paint_idx (menu, w, selected, 0);
	    selected++;
	    if (selected == menu->count)
		selected = 0;
	    if (!menu->entries [selected].call_back)
		selected++;
	    break;

	case '\n':
	    if (menu->entries [selected].call_back){
		(*menu->entries [selected].call_back)(0);
	    }
	    return 0;
	    
	case KEY_LEFT:
	case 2:			/* C-b */
	    untouch_bar ();
	    return -1;

	case KEY_RIGHT:
	case 6:
	    untouch_bar ();
	    return 1;

	case ' ':
	    break;
	    
	default:
	    for (i = 0; i < menu->count; i++){
		if (tolower (menu->entries [i].hot_key) == tolower (c))
		    if (menu->entries [selected].call_back)
			(*menu->entries [selected].call_back)(0);
		return 0;
	    }
	}
	paint_idx (menu, w, selected, 1);
	wrefresh (w);
    }
    refresh_screen ();
    return 0;
}

int get_motion (int top, int *abort)
{
    int c;
    int selected;

    *abort = 0;
    while ((c = getch ()) != '\e'){
	switch (c){
	case '\n':
	    return 0;
	    
	case KEY_LEFT:
	case 2:			/* C-b */
	    return -1;

	case KEY_RIGHT:
	case '\t':
	case ' ':
	case 6:
	    return 1;
	}
    }
    *abort = 1;
    return 0;
}

paint_bar (WINDOW *bar, int pos, int color, char *name)
{
    wmove (bar, 0, pos);
    wstandend (bar);
    wattron (bar, color);
    wprintw (bar, name);
}

int run_bar (WINDOW *parent, int x, int y, int width, int space, int items,
	     int first, Menu *Menus [], int flags, int color_sel,int color_uns)
{
    int current = 0;
    int i;
    int abort;
    WINDOW *bar;
    /*
    */
    int fill  = (flags & BAR_FILL);

    if (first < items && first >= 0)
	current = first;
    bar = derwin (parent, 1, width, y, x);
    leaveok (bar, TRUE);
    wstandend (bar);
    wattron (bar, color_uns);
    if (fill)
	wclr (bar);
    for (i = 0; i < items; i++){
	wmove (bar, 0, i*space+2);
	wprintw (bar, "%s", Menus [i]->name);
    }
    do {
	paint_bar (bar, current*space+2, color_sel, Menus [current]->name);
	touchwin (bar);
	wrefresh (bar);
	is_right = current;
	if (Menus [current]->count)
	    i = run_menu (Menus [current], 1, current*space);
	else
	    i = get_motion (items, &abort);
	paint_bar (bar, current*space+2, color_uns, Menus [current]->name);
	current += i;
	if (current < 0) current = items -1;
	current %= items;
    } while (i);
    refresh_screen ();
    delwin (bar);
    if (abort)
	return -1;
    return current;
}

static Menu *query_arr [MAXQUERY];

int query_dialog (char *header, char *text, int flags, int count, ...)
{
    va_list ap;
    int len, lines;
    int i, result;
    int options_len;
    int color_sel, color_uns;
    
    va_start (ap, count);
    len = max (strlen (header), msglen (text, &lines));
    create_dialog (len, lines+2, header, text, flags & BAR_ERROR);

    if (count > MAXQUERY)
	return;

    for (i = options_len = 0; i < count; i++){
	query_arr [i] = (Menu *) xmalloc (sizeof (Menu), "query_dialog");
	query_arr [i]->name = va_arg (ap, char *);
	query_arr [i]->count = 0;
	options_len += strlen (query_arr [i]->name) + 2;
    }
    va_end (ap);
    
    color_sel = (flags & BAR_ERROR) ? REVERSE_COLOR : Q_SELECTED_COLOR;
    color_uns = (flags & BAR_ERROR) ? ERROR_COLOR : Q_UNSELECTED_COLOR;
    
    result = run_bar (top_window,
	     2 + (len - options_len)/2, /* Start X pos */
	     lines+3,		/* Start Y pos */
	     options_len,	/* Width */
	     6,			/* inner space */
	     count,		/* items */
	     0,                 /* First selected menu entry */
	     query_arr, flags, color_sel, color_uns);
    done_dialog ();
    for (i = 0; i < count; i++)
	free (query_arr [i]);
    return result;
}
