------------------------------------------------------------------------------
--                                                                          --
--                         GNAT COMPILER COMPONENTS                         --
--                                                                          --
--                             S E M _ U T I L                              --
--                                                                          --
--                                 B o d y                                  --
--                                                                          --
--                            $Revision: 1.87 $                             --
--                                                                          --
--           Copyright (c) 1992,1993,1994 NYU, All Rights Reserved          --
--                                                                          --
-- GNAT is free software;  you can  redistribute it  and/or modify it under --
-- terms of the  GNU General Public License as published  by the Free Soft- --
-- ware  Foundation;  either version 2,  or (at your option) any later ver- --
-- sion.  GNAT is distributed in the hope that it will be useful, but WITH- --
-- OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
-- or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License --
-- for  more details.  You should have  received  a copy of the GNU General --
-- Public License  distributed with GNAT;  see file COPYING.  If not, write --
-- to the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA. --
--                                                                          --
------------------------------------------------------------------------------

with Atree;    use Atree;
with Debug;    use Debug;
with Errout;   use Errout;
with Itypes;   use Itypes;
with Namet;    use Namet;
with Nlists;   use Nlists;
with Nmake;    use Nmake;
with Output;   use Output;
with Opt;      use Opt;
with Scans;    use Scans;
with Scn;      use Scn;
with Sem;      use Sem;
with Sem_Eval; use Sem_Eval;
with Sem_Res;  use Sem_Res;
with Sem_Type; use Sem_Type;
with Sinfo;    use Sinfo;
with Sinput;   use Sinput;
with Stand;    use Stand;
with Style;
with Tbuild;   use Tbuild;
with Uintp;    use Uintp;

package body Sem_Util is

   ------------------------------
   -- Access_Checks_Suppressed --
   ------------------------------

   function Access_Checks_Suppressed (E : Entity_Id) return Boolean is
   begin
      return Scope_Suppress.Access_Checks
        or else Suppress_Access_Checks (E);
   end Access_Checks_Suppressed;

   -------------------------------------
   -- Accessibility_Checks_Suppressed --
   -------------------------------------

   function Accessibility_Checks_Suppressed (E : Entity_Id) return Boolean is
   begin
      return Scope_Suppress.Accessibility_Checks
        or else Suppress_Accessibility_Checks (E);
   end Accessibility_Checks_Suppressed;

   ------------------------
   -- Apply_Access_Check --
   ------------------------

   procedure Apply_Access_Check (N : Node_Id; Typ : Entity_Id) is
   begin
      if not Access_Checks_Suppressed (Typ) then
         Set_Do_Access_Check (N, True);
      end if;
   end Apply_Access_Check;

   ------------------------------
   -- Apply_Discriminant_Check --
   ------------------------------

   procedure Apply_Discriminant_Check (N : Node_Id; Typ : Entity_Id) is
   begin
      if not Discriminant_Checks_Suppressed (Typ) then
         Set_Do_Discriminant_Check (N, True);
      end if;
   end Apply_Discriminant_Check;

   -----------------------
   -- Apply_Range_Check --
   -----------------------

   --  A range constraint may be applied in some of the following contexts:
   --  object declaration, subtype declaration, derived declaration
   --  assignment, function/procedure/entry call, type conversion

   procedure Apply_Range_Check
     (N           : Node_Id;
      Source_Type : Entity_Id;
      Target_Type : Entity_Id) is

   begin
      if Range_Checks_Suppressed (Target_Type)
        or else Index_Checks_Suppressed (Target_Type)
        or else Source_Type = Any_Type
      then
         return;

      --  Confine the range checks currently to only signed integer types and
      --  enumeration types since support for floating point and fixed point
      --  types is too limited to do useful checks at this time and checks
      --  for modular types need to be better understood by us.

      elsif not Is_Discrete_Type (Source_Type)
        or else Is_Modular_Integer_Type (Source_Type)
      then
         return;

      --  ???
      --  Currently the Etype of literals are given the subtype of LHS
      --  rather than the base type. When this is corrected this test can
      --  be removed.

      elsif Nkind (N) = N_Integer_Literal then
         if Is_Static_Subtype (Target_Type)
           and then UI_Le (Expr_Value (Type_Low_Bound (Target_Type)),
                           Intval (N))
           and then UI_Ge (Expr_Value (Type_High_Bound (Target_Type)),
                           Intval (N)) then
            return;
         else
            null;
         end if;

      --  There is no need to have a range check at run-time if the value
      --  considered is guaranteed to be in the range of the target type
      --  even if it might be a dynamic value.

      elsif In_Subrange_Of (Source_Type, Target_Type) then
         return;
      end if;

      Set_Do_Range_Check (N, True);
   end Apply_Range_Check;

   --------------------------
   -- Check_Fully_Declared --
   --------------------------

   procedure Check_Fully_Declared (T : Entity_Id; N : Node_Id) is
   begin
      if Ekind (T) = E_Incomplete_Type
        or else Has_Private_Component (T)
      then
         Error_Msg_NE
           ("premature usage of private or incomplete type &", N, T);
      end if;
   end Check_Fully_Declared;

   --------------------
   -- Current_Entity --
   --------------------

   --  The currently visible definition for a given identifier is the
   --  one most chained at the start of the visibility chain, i.e. the
   --  one that is referenced by the Node_Id value of the name of the
   --  given identifier.

   function Current_Entity (N : Node_Id) return Entity_Id is
   begin
      return Get_Name_Entity_Id (Chars (N));
   end Current_Entity;

   -------------------
   -- Current_Scope --
   -------------------

   function Current_Scope return Entity_Id is
      C : constant Entity_Id := Scope_Stack.Table (Scope_Stack.last).Entity;

   begin
      if Present (C) then
         return C;
      else
         return Standard_Standard;
      end if;
   end Current_Scope;

   -------------------------------
   -- Defining_Unit_Simple_Name --
   -------------------------------

   function Defining_Unit_Simple_Name (N : Node_Id) return Entity_Id is
      Nam : Node_Id := Defining_Unit_Name (N);
   begin
      if Nkind (Nam) in N_Entity then
         return Nam;
      else
         return Defining_Identifier (Nam);
      end if;
   end Defining_Unit_Simple_Name;

   ------------------------------------
   -- Discriminant_Checks_Suppressed --
   ------------------------------------

   function Discriminant_Checks_Suppressed (E : Entity_Id) return Boolean is
   begin
      return Scope_Suppress.Discriminant_Checks
        or else Suppress_Discriminant_Checks (E);
   end Discriminant_Checks_Suppressed;

   --------------------------------
   -- Division_Checks_Suppressed --
   --------------------------------

   function Division_Checks_Suppressed (E : Entity_Id) return Boolean is
   begin
      return Scope_Suppress.Division_Checks
        or else Suppress_Division_Checks (E);
   end Division_Checks_Suppressed;

   -----------------------------------
   -- Elaboration_Checks_Suppressed --
   -----------------------------------

   function Elaboration_Checks_Suppressed (E : Entity_Id) return Boolean is
   begin
      return Scope_Suppress.Elaboration_Checks
        or else Suppress_Elaboration_Checks (E);
   end Elaboration_Checks_Suppressed;

   ----------------
   -- Enter_Name --
   ----------------

   procedure Enter_Name (Def_Id : Node_Id) is
      E : constant Entity_Id := Current_Entity (Def_Id);
      S : constant Entity_Id := Current_Scope;

   begin
      --  Add new name to current scope declarations. First we check
      --  for the case of a duplicate declaration

      if Present (E) and then Scope (E) = S then

         --  Case of previous entity entered because of a missing declaration
         --  or else a bad subtype indication. Best is to use the new entity,
         --  and make the previous one invisible.

         if Etype (E) = Any_Type then
            Set_Is_Directly_Visible (E, False);

         --  Case of genuine duplicate declaration. Keep previous declaration
         --  visible, but give some usable attributes to new one.

         else
            Set_Ekind (Def_Id, E_Variable);
            Set_Etype (Def_Id, Any_Type);
            Set_Scope (Def_Id,  S);
            Error_Msg_Sloc_1 := Sloc (E);
            Error_Msg_N
              ("declaration of& conflicts with line #", Def_Id);
            return;
         end if;
      end if;

      Set_Ekind (Def_Id, E_Void);
      Set_Etype (Def_Id, Any_Type);

      --  The kind E_Void insures that premature uses of the entity will be
      --  detected. Any_Type insures that no cascaded errors will occur.

      Set_Is_Directly_Visible (Def_Id);
      Set_Current_Entity (Def_Id);
      Set_Homonym (Def_Id, E);
      Append_Entity (Def_Id, S);
      Set_Public_Status (Def_Id);

   end Enter_Name;

   ------------------
   -- First_Actual --
   ------------------

   function First_Actual (Node : Node_Id) return Node_Id is
      N : Node_Id;

   begin
      if No (Parameter_Associations (Node)) then
         return Empty;
      end if;

      N := First (Parameter_Associations (Node));

      if Nkind (N) = N_Parameter_Association then
         return First_Named_Actual (Node);
      else
         return N;
      end if;
   end First_Actual;

   --------------------------
   -- Get_Declaration_Node --
   --------------------------

   function Get_Declaration_Node (Unit_Id : Entity_Id) return Node_Id is
      N : Node_Id := Parent (Unit_Id);

   begin
      --  Predefined operators do not have a full function declaration.

      if Ekind (Unit_Id) = E_Operator then
         return N;
      end if;

      while Nkind (N) /= N_Generic_Package_Declaration
        and then Nkind (N) /= N_Generic_Subprogram_Declaration
        and then Nkind (N) /= N_Package_Declaration
        and then Nkind (N) /= N_Package_Body
        and then Nkind (N) /= N_Package_Renaming_Declaration
        and then Nkind (N) /= N_Subprogram_Declaration
        and then Nkind (N) /= N_Subprogram_Body
        and then Nkind (N) /= N_Subprogram_Body_Stub
        and then Nkind (N) /= N_Subprogram_Renaming_Declaration
        and then Nkind (N) not in N_Generic_Renaming_Declaration
      loop
         N := Parent (N);
         pragma Assert (Present (N));
      end loop;

      return N;
   end Get_Declaration_Node;

   ----------------------
   -- Get_Index_Bounds --
   ----------------------

   procedure Get_Index_Bounds (I : Node_Id; L, H : out Node_Id) is
      Kind : Node_Kind := Nkind (I);

   begin
      pragma Assert
        (Kind = N_Range
          or else Kind = N_Subtype_Indication
          or else (Kind in N_Entity_Name
                     and then Is_Type (Entity (I))));

      if Kind = N_Range then
         L := Low_Bound (I);
         H := High_Bound (I);

      elsif Kind = N_Subtype_Indication then
         L := Low_Bound (Range_Expression (Constraint (I)));
         H := High_Bound (Range_Expression (Constraint (I)));

      else -- Kind in N_Entity_Name and then Is_Type (Entity (I)) then
         L := Low_Bound (Scalar_Range (Entity (I)));
         H := High_Bound (Scalar_Range (Entity (I)));

      end if;
   end Get_Index_Bounds;

   ------------------------
   -- Get_Name_Entity_Id --
   ------------------------

   function Get_Name_Entity_Id (Id : Name_Id) return Entity_Id is
   begin
      return Entity_Id (Get_Name_Table_Info (Id));
   end Get_Name_Entity_Id;

   ---------------------------
   -- Has_Private_Component --
   ---------------------------

   function Has_Private_Component (Type_Id : Entity_Id) return Boolean is
      T : Entity_Id := Base_Type (Type_Id);
      Component : Entity_Id;

   begin
      if Ekind (T) in Private_Kind then
         return No (Full_Declaration (T)) and not Is_Generic_Type (T);

      elsif Ekind (T) in Array_Kind then
         return Has_Private_Component (Component_Type (T));

      elsif Ekind (T) in Record_Kind then
         Component := First_Entity (T);
         while Present (Component) loop
            if Has_Private_Component (Etype (Component)) then
               return True;
            end if;
            Component := Next_Entity (Component);
         end loop;

         return False;

      else
         return False;
      end if;
   end Has_Private_Component;

   --------------------------
   -- Has_Tagged_Component --
   --------------------------

   function Has_Tagged_Component (Typ : Entity_Id) return Boolean is
      Comp : Entity_Id;

   begin
      if Ekind (Typ) in Private_Kind
        and then Present (Full_Declaration (Typ))
      then
         return Has_Tagged_Component (Full_Declaration (Typ));

      elsif Is_Array_Type (Typ) then
         return Is_Tagged_Type (Component_Type (Typ));

      elsif Is_Tagged_Type (Typ) then
         return True;

      elsif Is_Record_Type (Typ) then
         Comp := First_Component (Typ);

         while Present (Comp) loop
            if Has_Tagged_Component (Etype (Comp)) then
               return True;
            end if;

            Comp := Next_Component (Typ);
         end loop;

         return False;

      else
         return False;
      end if;
   end Has_Tagged_Component;

   --------------------
   -- In_Subrange_Of --
   --------------------

   function In_Subrange_Of (T1 : Entity_Id; T2 : Entity_Id) return Boolean is
   begin
      if T1 = T2 or else Is_Subtype_Of (T1, T2) then
         return True;

      elsif not (Is_Static_Subtype (T1) and then Is_Static_Subtype (T2)) then
         return False;

      elsif Is_Discrete_Type (T1) then
         return UI_Le (Expr_Value (Type_Low_Bound (T2)),
                       Expr_Value (Type_Low_Bound (T1)))
           and then
                UI_Ge (Expr_Value (Type_High_Bound (T2)),
                       Expr_Value (Type_High_Bound (T1)));
      else
         return False;
      end if;
   end In_Subrange_Of;

   -----------------------------
   -- Index_Checks_Suppressed --
   -----------------------------

   function Index_Checks_Suppressed (E : Entity_Id) return Boolean is
   begin
      return Scope_Suppress.Index_Checks
        or else Suppress_Index_Checks (E);
   end Index_Checks_Suppressed;

   --------------------
   -- Is_Entity_Name --
   --------------------

   function Is_Entity_Name (N : Node_Id) return Boolean is
   begin
      return Nkind (N) in N_Entity_Name;
   end Is_Entity_Name;

   ------------------------------
   -- Length_Checks_Suppressed --
   ------------------------------

   function Length_Checks_Suppressed (E : Entity_Id) return Boolean is
   begin
      return Scope_Suppress.Length_Checks
        or else Suppress_Length_Checks (E);
   end Length_Checks_Suppressed;

   -------------------------
   -- New_External_Entity --
   -------------------------

   function New_External_Entity
     (Kind         : Entity_Kind;
      Scope_Id     : Entity_Id;
      Sloc_Value   : Source_Ptr;
      Related_Id   : Entity_Id;
      Suffix       : Character;
      Suffix_Index : Nat := 0;
      Prefix       : Character := ' ')
      return         Entity_Id
   is
      N : constant Entity_Id :=
            Make_Defining_Identifier (Sloc_Value,
              New_External_Name
                (Chars (Related_Id), Suffix, Suffix_Index, Prefix));

   begin
      Set_Ekind (N, Kind);
      Set_Is_Internal (N);
      Append_Entity (N, Scope_Id);
      Set_Current_Entity (N);
      return N;
   end New_External_Entity;

   -------------------------
   -- New_Internal_Entity --
   -------------------------

   function New_Internal_Entity
     (Kind       : Entity_Kind;
      Scope_Id   : Entity_Id;
      Sloc_Value : Source_Ptr;
      Id_Char    : Character)
      return       Entity_Id
   is
      N : constant Entity_Id :=
            Make_Defining_Identifier (Sloc_Value, New_Internal_Name (Id_Char));

   begin
      Set_Ekind (N, Kind);
      Set_Is_Internal (N);
      Append_Entity (N, Scope_Id);
      Set_Current_Entity (N);
      return N;
   end New_Internal_Entity;

   -----------------
   -- Next_Actual --
   -----------------

   function Next_Actual (Actual_Id : Node_Id) return Node_Id is
      N  : Node_Id;

   begin
      --  If we are pointing at a positional parameter, it is a member of
      --  a node list (the list of parameters), and the next parameter
      --  is the next node on the list, unless we hit a parameter
      --  association, in which case we shift to using the chain whose
      --  head is the First_Named_Actual in the parent, and then is
      --  threaded using the Next_Named_Actual of the Parameter_Association.
      --  All this fiddling is because the original node list is in the
      --  textual call order, and what we need is the declaration order.

      if Is_List_Member (Actual_Id) then
         N := Next (Actual_Id);

         if Nkind (N) = N_Parameter_Association then
            return First_Named_Actual (Parent (Actual_Id));
         else
            return N;
         end if;

      else
         return Next_Named_Actual (Parent (Actual_Id));
      end if;
   end Next_Actual;

   -----------------------
   -- Normalize_Actuals --
   -----------------------

   --  Chain actuals according to formals of subprogram. If there are
   --  no named associations, the chain is simply the list of Parameter
   --  Associations, since the order is the same as the declaration order.
   --  If there are named associations, then the First_Named_Actual field
   --  in the N_Procedure_Call_Statement node or N_Function_Call node
   --  points to the Parameter_Association node for the parameter that
   --  comes first in declaration order. The remaining named parameters
   --  are then chained in declaration order using Next_Named_Actual.

   --  This routine also verifies that the number of actuals is compatible
   --  with the number and default values of formals, but performs no type
   --  checking (type checking is done by the caller).

   --  If the matching succeeds, the function returns True, and the caller
   --  proceeds with type-checking. If the match is unsuccessful, then the
   --  function returns False, and the caller attempts a different inter-
   --  pretation, if there is one.

   --  If the flag Report is on, the call is not overloaded, and a failure
   --  to match can be reported here, rather than in the caller.

   function Normalize_Actuals
     (N      : Node_Id;
      S      : Entity_Id;
      Report : Boolean)
      return   Boolean
   is
      Actuals     : constant List_Id := Parameter_Associations (N);
      Actual      : Node_Id   := Empty;
      Formal      : Entity_Id;
      Last        : Entity_Id := Empty;
      First_Named : Entity_Id := Empty;
      Found       : Boolean;

      Formals_To_Match : Integer := 0;
      Actuals_To_Match : Integer := 0;

      procedure Chain (A : Node_Id);
      --  Need somd documentation on this spec ???

      procedure Chain (A : Node_Id) is
      begin
         if No (Last) then

            --  Call node points to first actual in list.

            Set_First_Named_Actual (N, Actual_Parameter (A));

         else
            Set_Next_Named_Actual (Last, Actual_Parameter (A));
         end if;

         Last := A;
      end Chain;

   --  Start of processing for Normalize_Actuals

   begin
      if Is_Access_Type (S) then

         --  The name in the call is a function call that returns an access
         --  to subprogram. The designated type has the list of formals.

         Formal := First_Formal (Designated_Type (S));
      else
         Formal := First_Formal (S);
      end if;

      while Present (Formal) loop
         Formals_To_Match := Formals_To_Match + 1;
         Formal := Next_Formal (Formal);
      end loop;

      --  Find if there is a named association, and verify that no positional
      --  associations appear after named ones.

      if Present (Actuals) then
         Actual := First (Actuals);
      end if;

      while Present (Actual)
        and then Nkind (Actual) /= N_Parameter_Association
      loop
         Actuals_To_Match := Actuals_To_Match + 1;
         Actual := Next (Actual);
      end loop;

      if No (Actual) and Actuals_To_Match = Formals_To_Match then

         --  Most common case: positional notation, no defaults

         return True;

      elsif Actuals_To_Match > Formals_To_Match then

         --  Too many actuals: will not work.

         if Report then
            Error_Msg_N ("too many arguments in call", N);
         end if;

         return False;
      end if;

      First_Named := Actual;

      while Present (Actual) loop
         if Nkind (Actual) /= N_Parameter_Association then
            Error_Msg_N
              ("positional parameters not allowed after named ones", Actual);
            return False;

         else
            Actuals_To_Match := Actuals_To_Match + 1;
         end if;

         Actual := Next (Actual);
      end loop;

      if Present (Actuals) then
         Actual := First (Actuals);
      end if;

      Formal := First_Formal (S);

      while Present (Formal) loop

         --  Match the formals in order. If the corresponding actual
         --  is positional,  nothing to do. Else scan the list of named
         --  actuals to find the one with the right name.

         if Present (Actual)
           and then Nkind (Actual) /= N_Parameter_Association
         then
            Actual := Next (Actual);
            Actuals_To_Match := Actuals_To_Match - 1;
            Formals_To_Match := Formals_To_Match - 1;

         else
            --  For named parameters, search the list of actuals to find
            --  one that matches the next formal name.

            Actual := First_Named;
            Found  := False;

            while Present (Actual) loop
               if Chars (Selector_Name (Actual)) = Chars (Formal) then
                  Found := True;
                  Chain (Actual);
                  Actuals_To_Match := Actuals_To_Match - 1;
                  Formals_To_Match := Formals_To_Match - 1;
                  exit;
               end if;

               Actual := Next (Actual);
            end loop;

            if not Found then
               if Ekind (Formal) /= E_In_Parameter
                 or else No (Default_Value (Formal))
               then
                  if Report then
                     Error_Msg_NE ("missing argument in call:&", N, Formal);
                  end if;

                  return False;

               else
                  Formals_To_Match := Formals_To_Match - 1;
                  null; -- Chain_Default_Node;
               end if;
            end if;
         end if;

         Formal := Next_Formal (Formal);
      end loop;

      if  Formals_To_Match = 0 and then Actuals_To_Match = 0 then
         return True;
      else
         if Report then
            Error_Msg_N ("too many arguments in call", N);
         end if;

         return False;
      end if;
   end Normalize_Actuals;

   --------------------------------
   -- Overflow_Checks_Suppressed --
   --------------------------------

   function Overflow_Checks_Suppressed (E : Entity_Id) return Boolean is
   begin
      return Scope_Suppress.Overflow_Checks
        or else Suppress_Overflow_Checks (E);
   end Overflow_Checks_Suppressed;

   -----------------------------
   -- Range_Checks_Suppressed --
   -----------------------------

   function Range_Checks_Suppressed (E : Entity_Id) return Boolean is
   begin
      return Scope_Suppress.Range_Checks
        or else Suppress_Range_Checks (E);
   end Range_Checks_Suppressed;

   ------------------
   -- Real_Convert --
   ------------------

   --  We do the conversion to get the value of the real string by using
   --  the scanner, see Sinput for details on use of the internal source
   --  buffer for scanning internal strings.

   function Real_Convert (S : String) return Node_Id is
      Negative : Boolean;

   begin
      Source := Internal_Source_Ptr;
      Scan_Ptr := 1;

      for J in S'range loop
         Source (Source_Ptr (J)) := S (J);
      end loop;

      Source (S'Length + 1) := EOF;

      if Source (Scan_Ptr) = '-' then
         Negative := True;
         Scan_Ptr := Scan_Ptr + 1;
      else
         Negative := False;
      end if;

      Scan;

      if Negative then
         Set_Numerator (Token_Node, UI_Negate (Numerator (Token_Node)));
      end if;

      --  We used the scanner to construct the node, so Comes_From_Source
      --  got set True, but this literal doesn't really come from the source

      Set_Comes_From_Source (Token_Node, False);
      return Token_Node;
   end Real_Convert;

   ---------------
   -- Same_Name --
   ---------------

   function Same_Name (N1, N2 : Node_Id) return Boolean is
      K1 : constant Node_Kind := Nkind (N1);
      K2 : constant Node_Kind := Nkind (N2);

   begin
      if (K1 = N_Identifier or else K1 = N_Defining_Identifier)
        and then (K2 = N_Identifier or else K2 = N_Defining_Identifier)
      then
         return Chars (N1) = Chars (N2);

      elsif (K1 = N_Selected_Component or else K1 = N_Expanded_Name)
        and then (K2 = N_Selected_Component or else K2 = N_Expanded_Name)
      then
         return Same_Name (Selector_Name (N1), Selector_Name (N2))
           and then Same_Name (Prefix (N1), Prefix (N2));

      else
         return False;
      end if;
   end Same_Name;

   ------------------------
   -- Set_Current_Entity --
   ------------------------

   --  The given entity is to be set as the currently visible definition
   --  of its associated name (i.e. the Node_Id associated with its name).
   --  All we have to do is to get the name from the identifier, and
   --  then set the associated Node_Id to point to the given entity.

   procedure Set_Current_Entity (E : Entity_Id) is
   begin
      Set_Name_Entity_Id (Chars (E), E);
   end Set_Current_Entity;

   ---------------------------------
   -- Set_Entity_With_Style_Check --
   ---------------------------------

   procedure Set_Entity_With_Style_Check (N : Node_Id; Val : Entity_Id) is
      Val_Actual : Entity_Id;

   begin
      if Style_Check and then Nkind (N) = N_Identifier then
         Val_Actual := Val;

         --  A special situation arises for derived operations, where we want
         --  to do the check against the parent (since the Sloc of the derived
         --  operation points to the derived type declaration itself).

         while not Comes_From_Source (Val_Actual)
           and then Nkind (Val_Actual) in N_Entity
           and then (Ekind (Val_Actual) = E_Enumeration_Literal
                      or else Ekind (Val_Actual) = E_Function
                      or else Ekind (Val_Actual) = E_Generic_Function
                      or else Ekind (Val_Actual) = E_Procedure
                      or else Ekind (Val_Actual) = E_Generic_Procedure)
           and then Present (Alias (Val_Actual))
         loop
            Val_Actual := Alias (Val_Actual);
         end loop;

         Style.Check_Identifier (N, Val_Actual);
      end if;

      Set_Entity (N, Val);
   end Set_Entity_With_Style_Check;

   ------------------------
   -- Set_Name_Entity_Id --
   ------------------------

   procedure Set_Name_Entity_Id (Id : Name_Id; Val : Entity_Id) is
   begin
      Set_Name_Table_Info (Id, Int (Val));
   end Set_Name_Entity_Id;

   ---------------------
   -- Set_Next_Actual --
   ---------------------

   procedure Set_Next_Actual (Ass1_Id : Node_Id; Ass2_Id : Node_Id) is
   begin
      if Nkind (Parent (Ass1_Id)) = N_Parameter_Association then
         Set_First_Named_Actual (Parent (Ass1_Id), Ass2_Id);
      end if;
   end Set_Next_Actual;

   -----------------------
   -- Set_Public_Status --
   -----------------------

   procedure Set_Public_Status (Id : Entity_Id) is
      S : constant Entity_Id := Current_Scope;

   begin
      if S = Standard_Standard
        or else (Is_Public (S)
                  and then (Ekind (S) = E_Package
                             or else Is_Record_Type (S)
                             or else Ekind (S) = E_Void))
      then
         Set_Is_Public (Id);
      end if;
   end Set_Public_Status;

   --------------------
   -- Static_Integer --
   --------------------

   function Static_Integer (N : Node_Id) return Uint is
   begin
      Analyze (N);
      Resolve (N, Any_Integer);

      if Is_Static (N) then
         return Expr_Value (N);

      else
         Error_Msg_N ("static expression required here", N);
         return No_Uint;
      end if;
   end Static_Integer;

   -------------------------------
   -- Storage_Checks_Suppressed --
   -------------------------------

   function Storage_Checks_Suppressed (E : Entity_Id) return Boolean is
   begin
      return Scope_Suppress.Storage_Checks
        or else Suppress_Storage_Checks (E);
   end Storage_Checks_Suppressed;

   ---------------------------
   -- Tag_Checks_Suppressed --
   ---------------------------

   function Tag_Checks_Suppressed (E : Entity_Id) return Boolean is
   begin
      return Scope_Suppress.Tag_Checks
        or else Suppress_Tag_Checks (E);
   end Tag_Checks_Suppressed;

   -----------------
   -- Trace_Scope --
   -----------------

   procedure Trace_Scope (N : Node_Id; E : Entity_Id; Msg : String) is
   begin
      if Debug_Flag_W then
         for J in 0 .. Scope_Stack.Last loop
            Write_Str ("  ");
         end loop;

         Write_Str (Msg);
         Write_Name (Chars (E));
         Write_Str ("   line ");
         Write_Int (Int (Get_Line_Number (Sloc (N))));
         Write_Eol;
      end if;
   end Trace_Scope;

   -------------------
   -- Unimplemented --
   -------------------

   procedure Unimplemented (N : Node_Id; Feature : String) is
   begin
      Error_Msg_N (Feature & " not implemented yet", N);
   end Unimplemented;

end Sem_Util;
