/*
 *  isis_flood.c,v 1.11 1993/01/07 22:39:17 jch Exp
 */

/* Gated Release 3.0 */
/* Copyright (c) 1990, 1991, 1992 by Cornell University. All rights reserved. */
/* Refer to Particulars and other Copyright notices at the end of this file. */


#include "include.h"
#include "isis_includes.h"

/*
 *	Hash table used to locate LSPs quickly
 *	Hash is based upon the sourceID and chains LSPs entries
 *	off of each entry in the table.
 */

static LSEntry *L1Hash[L1HashTableSize];
static LSEntry *L2Hash[L2HashTableSize];

static u_short 
IDHash(sourceID)
LSPID	sourceID;
{
	u_short	hash = 0;
	int i;
	u_char	*cp = (u_char *)&sourceID;

	for (i=0; i<7; i++)
		hash ^= *cp++;
	
	return(hash);
}


/* insert LSP into hash table based upon sourceID address */
void
insertLSPByID(lsp, sourceID)
LSEntry	*lsp;
LSPID		sourceID;
{
	if (lsp->signature.level == 1) {
		u_short indx = IDHash(sourceID) % L1HashTableSize;
		if (L1Hash[indx]) {
			lsp->hashNext = L1Hash[indx];
		}
		L1Hash[indx] = lsp;
	} else {
		u_short indx = IDHash(sourceID) % L2HashTableSize;
		assert(lsp->signature.level == 2);
		if (L2Hash[indx]) {
			lsp->hashNext = L2Hash[indx];
		}
		L2Hash[indx] = lsp;
	}
}

/*
 *	Remove 'ent' from hash table
 */
void
removeLSP(ent)
LSEntry	*ent;
{
	LSEntry	*scan, *lastScan;
	u_short	indx;

	if (ent->signature.level == 1) {
		indx = IDHash(ent->signature.id) % L1HashTableSize;
		if (L1Hash[indx] == ent) {
			L1Hash[indx] = ent->hashNext;
			return;
		} else
			scan = L1Hash[indx];
	} else {
		assert(ent->signature.level == 2);
		indx = IDHash(ent->signature.id) % L2HashTableSize;
		if (L2Hash[indx] == ent) {
			L2Hash[indx] = ent->hashNext;
			return;
		} else
			scan = L2Hash[indx];
	}
	for ( ; scan; lastScan=scan, scan=scan->hashNext) {
		if (scan == ent) {
			lastScan->hashNext = ent->hashNext;
			return;
		}
	}
	trace(TR_ISIS, LOG_ERR, "removeLSP: can't find entry");
}

/*
 *	Sorted LSP list. Two sorted lists of LSEntries are maintained - a
 *	level 1 and a level 2 list. The lists are sorted in ascending order
 *	based upon LSPID. These lists are used to build and process
 *	sequence number pdus. The lists are singly linked.
 */
static LSEntry *sortL1 = NULL, *sortL2 = NULL;

/*
 *	Return the head of the sort list.
 */
LSEntry *
firstLSP(level)
int	level;
{
	return(level == 1 ? sortL1 : sortL2);
}

/*
 *	Add 'ent' to proper sort list. This is basically an insertion sort
 */
void
addToSortList(ent)
LSEntry	*ent;
{
	LSEntry	**listPtr, *scan, *lastScan = NULL;

	listPtr = (ent->signature.level == 1) ? &sortL1 : &sortL2;

	if (*listPtr == NULL) {
		*listPtr = ent;
	} else {
		for (scan = *listPtr; scan; lastScan = scan, scan = scan->sortNext) {
			if (LSPIDcmp(&scan->signature.id, &ent->signature.id) > 0) {
				if (scan == *listPtr) {
					ent->sortNext = *listPtr;
					*listPtr = ent;
				} else {
					ent->sortNext = lastScan->sortNext;
					lastScan->sortNext = ent;
				}
				return;
			}
		}
		lastScan->sortNext = ent;
	}
}

/*
 *	Remove 'ent' from the sort list.
 */
void
rmFromSortList(ent)
LSEntry	*ent;
{
	LSEntry	**listPtr, *scan, *lastScan = NULL;
	extern LSEntry *lastL1Ent, *lastL2Ent;

	/*
	 *	Check if 'ent' is pointed to by lastLnEnt (maintained for 
	 *	transmission of PSNPs). If so, set lastLnEnt to NULL
	 */
	if (ent == lastL1Ent)
		lastL1Ent = NULL;
	else if (ent == lastL2Ent)
		lastL2Ent = NULL;

	listPtr = (ent->signature.level == 1) ? &sortL1 : &sortL2;
	if (*listPtr) {
		for (scan = *listPtr; scan; lastScan = scan, scan = scan->sortNext) {
			if (scan == ent) {
				if (scan == *listPtr) {
					*listPtr = ent->sortNext;
				} else {
					lastScan->sortNext = ent->sortNext;
				}
				return;
			}
		}
		lastScan->sortNext = NULL;
	}
}

/* Store LSEntry in hash table and sorted list */
void
storeLSEntry(ent)
LSEntry *ent;
{
	insertLSPByID(ent, ent->signature.id);
	addToSortList(ent);
}


/*
 *	Allocate an LSP Entry & initialize
 *	Return null or pointer to LSP entry
 */
LSEntry *
newLSEntry(buf, len, sig)
u_char		*buf;
int			len;
LSSignature	*sig;
{
	LSEntry	*ent = (LSEntry *)justMalloc(sizeof(LSEntry));

	if (ent == NULL)
		return(NULL);
	
	/* justMalloc seems to return a funny value if passed zero */
	if (len > 0) {
		ent->buf = (u_char *)justMalloc(len);
		if (ent->buf == NULL)
			return(NULL);
	} else
		ent->buf = NULL;

	ent->signature = *sig;
	ent->len = len;
	ent->arrivalTimestamp = timeStamp;
	ent->originalLifetime = ent->remainingTime = sig->lifetime;
	bcopy((caddr_t) buf, (caddr_t) ent->buf, len);
	storeLSEntry(ent);
	ageListInsert(normalLSPAge, (u_char *) ent);
	signalDecisionProcess(ent->signature.level);

	return(ent);
}

/*
 *	Update LSEntry 'ent' with new contents. This may fail if we need
 *	to allocate more memory to store the new buf. Return False if we
 *	can't malloc, otherwise true.
 */
Boolean
updateLSEntry(ent, buf, len, sig)
LSEntry		*ent;
u_char		*buf;
int			len;
LSSignature	*sig;
{
	Boolean	whatMeWorry = False;

	assert(ent->lspDesc == NULL);		/*should not call this on my own LSP */

	/*
	 *	Determine if this new LSP is sufficently different that we
	 *	ought to signal the decision process
	 */
	if ((ent->len != len) || (ent->signature.checksum != sig->checksum) ||
		(!bcmp((caddr_t) ent->buf, (caddr_t) buf, ent->len)))
			whatMeWorry = True;

	if (ent->len < len) {
		/* need to enlarge buf */
		free(ent->buf);
		ent->buf = (u_char *)justMalloc(len);
		if (ent->buf == NULL)
			return(False);
	}
	ent->signature = *sig;
	ent->arrivalTimestamp = timeStamp;
	ent->len = len;
	bcopy((caddr_t) buf, (caddr_t) ent->buf, len);

	/*
	 *	This entry could be on the deleted list. If so, remove it from
	 *	that and add to the normal list. 
	 */
	if (ent->onDeletedList) {
		removeFromDeletedList(ent);
	} else {
		ageListRemove(normalLSPAge, (u_char *) ent);
	}
	ent->originalLifetime = ent->remainingTime = sig->lifetime;
	ageListInsert(normalLSPAge, (u_char *) ent);

	if (whatMeWorry)
		signalDecisionProcess(ent->signature.level);

	return(True);
}

void
removeFromDeletedList(ent)
LSEntry	*ent;
{
	assert(ent->onDeletedList == True);
	ageListRemove(deletedLSPAge, (u_char *) ent);
	ent->onDeletedList = False;
}

/* 
 *	return a pointer to (or NULL) the first matching LSPs 
 *	entry in hashtable 
 */
LSEntry *
locateLSPByID(level, id)
int		level;
LSPID	id;
{
	LSEntry	*ent;

	if (level == 1) {
		ent = L1Hash[IDHash(id) % L1HashTableSize];
	} else {
		assert(level == 2);
		ent = L2Hash[IDHash(id) % L2HashTableSize];
	}
	while (ent) {
		if (equalID(&id, &ent->signature.id, 8))
			return(ent);
		ent = ent->hashNext;
	} 
	return(NULL);
}

/* 
 *	return a pointer to (or NULL) the LSP with the lowest LSP number 
 *	in the hash table
 */
LSEntry *
locateLowestLSP(int level, 
		LSPID id)
{
	LSEntry	*ent, *poss = NULL;
	u_char	lowest = 255;

	if (level == 1) {
		ent = L1Hash[IDHash(id) % L1HashTableSize];
	} else {
		assert(level == 2);
		ent = L2Hash[IDHash(id) % L2HashTableSize];
	}
	while (ent) {
		if (equalID(&id, &ent->signature.id, 7)) {
			if (ent->signature.id.LSPNumber < lowest) {
				lowest = ent->signature.id.LSPNumber;
				poss = ent;
			}
		}
		ent = ent->hashNext;
	} 
	return(poss);
}

/*
 *	Insure that flag bits are only set if there is a circuit
 *	enabled at that bit position
 */
void
setFlags(flags, mode, c)
Boolean		*flags;		/* array MaximumCircuits of flags */
int			mode;		/* which flags to set */
int			c;			/* optional circuit identifier */
{
	int i;

	if (mode == AllCircuits) {
		for (i=0; i<MaximumCircuits; i++) {
			flags[i] = False;
			if (circuitList[i])
				flags[i] = True;
		}
	} else {
		assert(mode == SingleCircuit);
		flags[c] = True;
	}
}


void
clrFlags(flags, mode, c)
Boolean		*flags;		/* array MaximumCircuits of flags */
int			mode;		/* which flags to clear */
int			c;			/* optional circuit identifier */
{
	int i;

	if (mode == AllCircuits) {
		for (i=0; i<MaximumCircuits; i++)
			flags[i] = False;
	} else {
		assert(mode == SingleCircuit);
		flags[c] = False;
	}
}

/*
 *	Return true if is is ok to flood an lsp of the level specified on the
 *	circuit.
 *
 *	If level is 1, there must be at least 1 L1 adj or at least one L2
 *		adj not marked Level 2 only
 *
 *	If level is 2, there must be at least one L2 adj.
 */
int
okToFlood(c, level)
CircuitEntry	*c;
int				level;
{
	AdjacencyEntry  *scan;

	IterateDLListForw(scan, &c->isAdjs->links, AdjacencyEntry *) {
		if (scan->state == AdjUp) {
			if (level == 2 && scan->adjacencyType == L2IS)
				return(True);
			if (level == 1) {
				if ((scan->adjacencyType == L1IS) || 
					((scan->adjacencyType == L2IS) && (!scan->level2Only)))
					return(True);
			}
		}
	}
	return(False);
}


/* 
 * Print LSP IS Neighbors in human-readable form.
 */
void
printLSPISN __PF4(fd, FILE *,
		  lsp, struct is_lsp_header *,
		  cp, char *,
		  len, u_char)
{
	int metric;
	int idLen;
	int vFlag;
	char *EndOfField;
	char *isn;
	char pseudo;

	EndOfField = cp + len;
	if (lsp->idLen == 0) 
		idLen = 6;
	else if (lsp->idLen == 255)
		idLen = 0;
	else idLen = lsp->idLen;
	vFlag = *cp++;
#define	PRINTISN \
	metric = *cp; \
	cp += 4; \
	isn = cp; \
	cp += idLen + 1; \
	pseudo = (isn[idLen] == 0) ? 'N' : 'P'; \
	if (fd) fprintf(fd, "%s[%c%x] (%d)", IDToStr((u_char *) isn, idLen), pseudo, isn[idLen], metric); \
	else tracef("%s[%c%x] (%d)", IDToStr((u_char *) isn, idLen), pseudo, isn[idLen], metric); 
	while (cp < EndOfField) {
		if (fd) 
			if (vFlag) fprintf(fd, "  vISNs: ");
			else fprintf(fd, "  ISNs: ");
		else 
			if (vFlag) tracef("  vISNs: ");
			else tracef("  ISNs: ");
		PRINTISN;
		if (cp >= EndOfField) {
			if (fd) fprintf(fd, "\n");
			else trace(TR_ISIS, LOG_INFO, NULL);
			break;
		}
		if (fd) fprintf(fd, ", ");
		else tracef(", ");
		PRINTISN;
		if (fd) fprintf(fd, "\n");
		else trace(TR_ISIS, LOG_INFO, NULL);
	}
}     

/* 
 * Print LSP ES Neighbors in human-readable form.
 */
void
printLSPESN __PF4(fd, FILE *,
		  lsp, struct is_lsp_header *,
		  cp, char *,
		  len, u_char)
{
	int metric;
	int idLen;
	char *EndOfField;
	char *esn;

	EndOfField = cp + len;
	if (lsp->idLen == 0) 
		idLen = 6;
	else if (lsp->idLen == 255)
		idLen = 0;
	else idLen = lsp->idLen;
#define	PRINTESN \
	esn = cp; \
	cp += idLen; \
	if (fd) fprintf(fd, "%s (%d)", IDToStr((u_char *) esn, idLen), metric); \
	else tracef("%s (%d)", IDToStr((u_char *) esn, idLen), metric); 
	metric = *cp; 
	cp += 4; 
	while (cp < EndOfField) {
		if (fd) fprintf(fd, "  ESNs: ");
		else tracef("  ESNs: ");
		PRINTESN;
		if (cp >= EndOfField) {
			if (fd) fprintf(fd, "\n");
			else trace(TR_ISIS, LOG_INFO, NULL);
			break;
		}
		if (fd) fprintf(fd, ", ");
		else tracef(",  ");
		PRINTESN;
		if (fd) fprintf(fd, "\n");
		else trace(TR_ISIS, LOG_INFO, NULL);
	}
}     

/* 
 * Print LSP Area Addresses in human-readable form.
 */
void
printLSPArea __PF3(fd, FILE *,
		   cp, char *,
		   len, u_char)
{
	char *EndOfField;
	int alen;
	char *area;
	sockaddr_un *sa; 

	EndOfField = cp + len;
#define	PRINTAREA \
	alen = *cp++; \
	area = cp; \
	cp += alen; \
	sa = sockbuild_iso((byte *) area, alen); \
	if (fd) fprintf(fd, "%A", sa); \
	else tracef("%A", sa); \

	while (cp < EndOfField) {
		if (fd) fprintf(fd, "  AA: ");
		else tracef("  AA: ");
		PRINTAREA;
		if (fd) fprintf(fd, "\n");
		else trace(TR_ISIS, LOG_INFO, NULL);
	}
}     

/* 
 * Print LSP Reachable Address Prefix Neighbors in human-readable form.
 */
void
printLSPPref __PF3(fd, FILE *,
		   cp, char *,
		   len, u_char)
{
	int metric;
	int prefLen;
	char *EndOfField;
	char *prefix;
	char metricType;
	sockaddr_un *sa;

	EndOfField = cp + len;
#define	PRINTPREFIX \
	prefLen = (*cp++ + 1) >> 1; \
	prefix = cp; \
	cp += prefLen; \
	sa = sockbuild_iso((byte *) prefix, prefLen); \
	if (fd) fprintf(fd, "%A", sa); \
	else tracef("%A", sa); \
	if (fd) fprintf(fd, " (%c.%d)", metricType, metric & 0x3f); \
	else tracef(" (%c.%d)", metricType, metric & 0x3f); 
	metric = *cp;
	metricType = (metric & 0x40) ? 'E' : 'I'; 
	cp += 4;
	while (cp < EndOfField) {
		if (fd) fprintf (fd, "  PREF: ");
		else tracef("  PREF: ");
		PRINTPREFIX;
		if (fd) fprintf(fd, "\n");
		else trace(TR_ISIS, LOG_INFO, NULL);
	}
}     

/* 
 * Print protocols supported field in human-readable form.
 */
void
printProtoSupported __PF3(fd, FILE *,
			  cp, char *,
		  	  len, u_char)
{
	char *EndOfField;
	int nlpid;

	EndOfField = cp + len;
#define	PRINTProto \
	nlpid = *cp++; \
	if (nlpid == OSI_NLPID) if (fd) fprintf(fd, "CLNP"); else tracef("CLNP"); \
	else if (nlpid == IP_NLPID) if (fd) fprintf(fd, "IP"); else tracef("IP"); \
	else if (fd) fprintf(fd, "0x%02x", nlpid); else tracef("0x%02x", nlpid); 
	if (fd) fprintf(fd, "  Protocols Supported: ");
	else tracef("  Protocols Supported: ");
	PRINTProto;
	while (cp < EndOfField) {
		if (fd) fprintf(fd, ", ");
		else tracef(", ");
		PRINTProto;
	}
	if (fd) fprintf(fd, "\n");
	else trace(TR_ISIS, LOG_INFO, NULL);
}     

/* 
 * Print ISO authentication information.
 */
void
printISOAuth __PF3(fd, FILE *,
		   cp, char *,
		   len, u_char)
{
	char *authinfo;
	int authType;

	authType = *cp++; 
	authinfo = cp; 
	if (fd) fprintf(fd, "  ISO Authentication: ");
	else tracef("  ISO Authentication: ");
	if (authType == 1) if (fd) fprintf(fd, "cleartext password\n"); 
				else trace(TR_ISIS, LOG_INFO, "cleartext password"); 
	else if (fd) fprintf(fd, "private\n"); else trace(TR_ISIS, LOG_INFO, "private");
}     

/* 
 * Print partition designated L2 IS in human-readable form.
 */
void
printPDL2IS __PF4(fd, FILE *,
		  lsp, struct is_lsp_header *,
		  cp, char *,
		  len, u_char)
{
	int idLen;

	if (lsp->idLen == 0) 
		idLen = 6;
	else if (lsp->idLen == 255)
		idLen = 0;
	else idLen = lsp->idLen;
	if (fd) fprintf(fd, "  Partition Designated L2 IS: %s\n", IDToStr((u_char *) cp, idLen));
	else trace(TR_ISIS, LOG_INFO, "  Partition Designated L2 IS: %s", IDToStr((u_char *) cp, idLen));
}     

/* 
 * Print IP interface addresses in human-readable form.
 */
void
printLSPipIfAddr __PF3( fd, FILE *,
			cp, char *,
			len, u_char)
{
	char *EndOfField;
	char *addr;

	EndOfField = cp + len;
#define	PRINTipIfAddr \
	addr = cp; \
	cp += 4; \
	if (fd) fprintf(fd, "%d.%d.%d.%d", addr[0], addr[1], addr[2], addr[3]); \
	else tracef("%d.%d.%d.%d", addr[0], addr[1], addr[2], addr[3]);
	while (cp < EndOfField) {
		if (fd) fprintf(fd, "  IP interfaces: ");
		else tracef("  IP interfaces: ");
		PRINTipIfAddr;
		if (cp >= EndOfField) {
			if (fd) fprintf(fd, "\n"); 
			else trace(TR_ISIS, LOG_INFO, NULL);
			break;
		}
		if (fd) fprintf(fd, ",  ");
		else tracef(",  ");
		PRINTipIfAddr;
		if (fd) fprintf(fd, "\n");
		else trace(TR_ISIS, LOG_INFO, NULL);
	}
}     

/* 
 * Print LSP IP internal reachability info in human-readable form.
 */
void
printLSPipReach  __PF4(fd, FILE *,
		       cp, char *,
		       len, u_char,
		       internal, int)
{
	int metric;
	char *EndOfField;
	char *net;
	char *mask;
	char metricType;
	char buf[300];

	EndOfField = cp + len;
#define	PRINTipInt \
	metric = *cp; \
	net = &cp[4]; \
	mask = &cp[8]; \
	cp += 12; \
	sprintf(buf, "%d.%d.%d.%d\t%02x.%02x.%02x.%02x ", net[0],net[1],net[2],net[3],mask[0],mask[1],mask[2],mask[3]); \
	if (fd) fprintf(fd, "%s", buf); \
	else tracef("%s", buf); \
	if (internal) if (fd) fprintf(fd, "(%d)", metric); else tracef("(%d)", metric); \
	else { \
		metricType = (metric & 0x40) ? 'E' : 'I'; \
		if (fd) fprintf(fd, "(%c.%d)", metricType, metric & 0x3f); \
		else tracef("(%c.%d)", metricType, metric & 0x3f); \
	}
	while (cp < EndOfField) {
		if (fd) fprintf(fd, "  IP: ");
		else tracef("  IP: ");
		PRINTipInt;
		if (fd) fprintf(fd, "\n");
		else trace(TR_ISIS, LOG_INFO, NULL);
	}
}     

/*
 * Print LSP contents (variable length fields) in human-readable
 * form.
 */
void
printLSPContents __PF3(fd, FILE *,
		       pdu, char *,
		       len, int)
{
	struct is_lsp_header *lsp;
	char *cp;
	u_char vcode;			/* variable length field code */
	u_char vlen;			/* variable length field length */

	lsp = (struct is_lsp_header *) pdu;
	cp = pdu + sizeof(struct is_lsp_header);	/* get to the option part */

	while (cp < (pdu + len)) {
		vcode = *cp++;
		vlen = *cp++;
		switch (vcode) {
		case LSPESNeighborCode:
			printLSPESN(fd, lsp, cp, vlen);
			break;
		case LSPISNeighborCode:
			printLSPISN(fd, lsp, cp, vlen);
			break;
		case LSPPrefixNeighborCode:
			printLSPPref(fd, cp, vlen); 
			break;
		case AreaAddressCode:
			printLSPArea(fd, cp, vlen);
			break;
		case AuthenticationCode:
			printISOAuth(fd, cp, vlen);
			break;
		case LSPPDL2ISCode:
			printPDL2IS(fd, lsp, cp, vlen);
			break;
		case ProtoSupportedCode:
			printProtoSupported(fd, cp, vlen);
			break;
		case IPExtReachCode:
			printLSPipReach(fd, cp, vlen, 0); 
			break;
		case IPIntReachCode:
			printLSPipReach(fd, cp, vlen, 1); 
			break;
		case IPIfAddrCode:
			printLSPipIfAddr(fd, cp, vlen);
			break;
		default:
			if (fd) fprintf(fd, "  Unsupported Field (code %d)\n", vcode);
			else trace(TR_ISIS, LOG_INFO, "  Unsupported Field (code %d)", vcode);
		}
		cp += vlen;
	}
}

/*
 *	Transmit lsp in 'ent' on circuit 'c'
 */
void
transmitLSP(c, ent)
CircuitEntry	*c;
LSEntry	*ent;
{
	updateLifetime(ent);
	IFTRACE(T_FLOODING)
		trace(TR_ISIS, LOG_INFO, "Flooding: %s: (%d bytes) %s", c->name, ent->len,
			LSSigToStr(&ent->signature));
	ENDTRACE

	IFTRACE(T_DUMPLSP)
                trace(TR_ISIS, LOG_INFO, "LSP %s (%d bytes) - xmit on %s", 
			LSSigToStr(&ent->signature), ent->len, c->name);
		IFTRACE(T_LSPCONTENT)
			printLSPContents(0, (char *) ent->buf, ent->len);
		ENDTRACE
	ENDTRACE
			
	transmitPacket(c, ent->signature.level == 1 ? AllL1ISmac : AllL2ISmac,
		ent->buf, ent->len);
}

/*
 *	Scan the list of lsps. For each LSEntry, transmit the LSP on
 *	any circuit where the SRM flag is set and the circuit is a
 *	pt-2-pt. Do not clear the SRM flag after transmission (this is
 *	down when an acknowledgement - via the seq number packet - is 
 *	received).
 */
void
floodPt2Pt()
{
	int						level;

	for (level = 1; level < 3; level++) {
		LSEntry				*ent;
		for (ent = firstLSP(level); ent; ent = ent->sortNext) {
			int				i;
			CircuitEntry	*c;
			for (i=0; i<MaximumCircuits; i++) {
				if ((c = circuitList[i]) && 
					(c->circuitType == Pt2Pt) && 
					(ent->SRM[i])) {
					if (okToFlood(c, ent->signature.level)) {
						transmitLSP(c, ent);
					} else {
						/* take off of flood queue */
						clrFlags(ent->SRM, SingleCircuit, i);
					}
				}
			}
		}
	}

	setTimer(minLSPXmitInterval, floodPt2Pt, NULL);
}

/*
 *	For every Broadcast circuit, transmit up to BLSPThrottle LSPs queued for 
 *	flooding on that circuit. Clear the SRM flag after transmission.
 *	TODO: should start scan where we left off last, not at start
 */
void
floodBroadcast()
{
	int						level;

	for (level = 1; level < 3; level++) {
		int						i;
		int						count[MaximumCircuits];
		LSEntry				*ent;

		/* count number of packets sent on each circuit here */
		for (i=0; i<MaximumCircuits; i++)
			count[i] = BLSPThrottle;

		for (ent = firstLSP(level); ent; ent = ent->sortNext) {
			CircuitEntry	*c;

			for (i=0; i<MaximumCircuits; i++) {
				if ((c = circuitList[i]) && (c->circuitType == Broadcast) &&
					(ent->SRM[i]) && (count[i])) {
					if (okToFlood(c, ent->signature.level)) {
						transmitLSP(c, ent);
						count[i]--;
					}
					clrFlags(ent->SRM, SingleCircuit, i);
				}
			}
		}
	}

	setTimer(minBLSPXmitInterval, floodBroadcast, NULL);
}


void
initFloodList()
{
	setTimer(minLSPXmitInterval, floodPt2Pt, NULL);
	setTimer(minBLSPXmitInterval, floodBroadcast, NULL);
}

/*
 *	Map the pointer to CircuitEntry back to an index into the
 *	circuitList. It would be, like, a total bummer for this to
 *	fail.
 */
int
CEToIndex(c)
CircuitEntry	*c;
{
	int	i;
	for (i=0; i<MaximumCircuits; i++) {
		if (circuitList[i] == c)
			return(i);
	}
	assert(False);
	return(0);
}

void
ackLSP(c, ent)
CircuitEntry	*c;
LSEntry			*ent;
{
	clrFlags(ent->SRM, SingleCircuit, CEToIndex(c));
	if (c->circuitType == Pt2Pt) {
		setFlags(ent->SSN, SingleCircuit, CEToIndex(c));
	}
}

void
sendLSP(c, ent)
CircuitEntry	*c;
LSEntry			*ent;
{
	setFlags(ent->SRM, SingleCircuit, CEToIndex(c));
	clrFlags(ent->SSN, SingleCircuit, CEToIndex(c));
}

/*
 *	Increment the sequence number and flood the LSP on all
 *	links
 */
void
updateSeqAndFlood(ent, seqNum)
LSEntry	*ent;
long	seqNum;
{
	LSPDesc		*desc = ent->lspDesc;

	assert(desc != NULL);
	desc->seqNumber = seqNum+1;

	IFTRACE(T_BUILDLSP)
		dumpLSPDesc(desc, "LSP changed (Seq Update) ");
	ENDTRACE

	lspChanged(desc);
}

/*
 *	acknowledge sender,
 *	and flood to all other neighbors
 */
void
ackAndFlood(c, ent)
CircuitEntry	*c;
LSEntry			*ent;
{
	int	i;
	ackLSP(c, ent);
	for (i=0; i<MaximumCircuits; i++) {
		if ((circuitList[i]) && (circuitList[i] != c))
			setFlags(ent->SRM, SingleCircuit, CEToIndex(circuitList[i]));
	}
}

/* make this LSEntry disappear in 'delay' seconds */
void
markForDeletion(ent, delay)
LSEntry	*ent;
int		delay;
{
	if (ent->onDeletedList) {
		trace(TR_ISIS, LOG_INFO, "IS-IS markForDeletion found entry already on deleted list - ignoring");
	} else {
		ent->remainingTime = delay;
		ageListInsert(deletedLSPAge, (u_char *) ent);
		ent->onDeletedList = True;
	}
}

/*
 */
void
purgeLSP(ent)
LSEntry	*ent;
{
	assert(ent->lspDesc == NULL); 	/* shouldn't do this to my own LSP */
									/* it isn't on 'normalLSPAge' */
	ageListRemove(normalLSPAge, (u_char *) ent);

	/*
	 *	If the sequence number is zero, then this is actually a dummy
	 *	LSP entry created during sequence number processing. Don't
	 *	keep it around.
	 */
	if (ent->signature.seqNum == 0) {
		deletedLSPExpired(ent);
	} else {
		normalLSPExpired(ent);
		signalDecisionProcess(ent->signature.level);
	}
}

/*
 *	Return the correct value for the lifetime value of the PDU based upon
 *	the original lifetime field (stored in the LSEntry) and the
 *	amount of time the PDU has spent in storage. If the signature lifetime
 *	has been set to zero, then return zero as the corrected lifetime.
 *	Also, special case for my own LSPs - they never age before I send them
 */
u_short
correctedLifetime(ent)
LSEntry	*ent;
{
	u_long	clt;	/* corrected lifetime */

	if (ent->signature.lifetime && (!(locallyGenerated(ent)))) {
		/* determine how long the pdu has been hanging around */
		/* calculate this in longs so as to avoid overflow */

		if ((timeStamp - ent->arrivalTimestamp) > ent->originalLifetime)
			clt = 0;
		else
			clt = ent->originalLifetime - (timeStamp - ent->arrivalTimestamp);

		ent->signature.lifetime = clt;
	}
	return(ent->signature.lifetime);
}

/*
 *	Update the lifetime field in the PDU to be consistent with the
 *	corrected Lifetime
 */
void
updateLifetime(ent)
LSEntry	*ent;
{
	/* check for empty header here */
	if (ent->len && ent->buf) {
		LSPHeader	*hdr = (LSPHeader *)ent->buf;
		hostToU16(correctedLifetime(ent), hdr->remainingLifetime);
	}
}

/*
 *	Called when an LSP's ttl expires
 *	Mark LSP to be deleted
 *	Retain only the header, and flood on all circuits.
 */
int
normalLSPExpired(ent)
LSEntry	*ent;
{
	LSPHeader	*hdr = (LSPHeader *)ent->buf;

	setFlags(ent->SRM, AllCircuits, 0);

	IFTRACE(T_FLOODING)
		trace(TR_ISIS, LOG_INFO, "LSP: %s: %s", ent->remainingTime > 0 ? "Purged" : "Expired",
			LSSigToStr(&ent->signature));
			
	ENDTRACE

	if (ent->remainingTime)
		markForDeletion(ent, maximumAge);
	else
		markForDeletion(ent, zeroAgeLifetime);

	ent->signature.lifetime = 0;
	ent->len = sizeof(LSPHeader);
	/* DTS 12-5-90 check for null hdr here; buf pointer  is sometimes null */
	/* if so, zero the length; this should be ok to indicate a dummy entry */
	/* but should check with rob on this. */
	if (hdr) {
		hostToU16(sizeof(LSPHeader), hdr->pduLength); 
	} else
		ent->len = 0;

	return(0);
}


int
deletedLSPExpired(ent)
LSEntry	*ent;
{
	IFTRACE(T_FLOODING)
		trace(TR_ISIS, LOG_INFO, "LSP Deleted: %s", LSSigToStr(&ent->signature));
	ENDTRACE

	removeLSP(ent);				/* remove from hash table */
	rmFromSortList(ent);		/* remove from sorted list */

	/*
	 *	Delete an LSEntry. Special case for our own LSPs because the
	 *	buf is part of the LSPDesc
	 */
	if (locallyGenerated(ent)) {
		assert(ent == ent->lspDesc->lspEntry);
		if (ent->lspDesc->schedRegenerate) {
			/* clear LSP timer if it's set */
			clearTimer(ent->lspDesc->timer);
		}
		ent->lspDesc->lspEntry = NULL;
	} else {
		free(ent->buf);
	}
	free(ent);
	return(0);
}


void
dumpLSPList(al)
AgeList	*al;
{
	LSEntry	*scan;

	IterateDLListForw(scan, &al->links, LSEntry *) {
		printf("LSP: rt %d, %s x%x\n", scan->remainingTime, LSSigToStr(&scan->signature), scan);
	}
}


/*
 *  Cause decision to run
 */
void
signalDecisionProcess(level)
int level;    /* level 1 or level 2 */
{
	if (systemType == L2IS)
		updateAreaSummary();
	isis_spf(level);
}

/*
 *	dump the contents of the LSP database at level specified
 */
void
dumpLSPDB __PF2(fd, FILE *,
		level, int)
{
	LSEntry	*ent = firstLSP(level);
	char scratch[40];

	if (fd) fprintf(fd, "L%d LSP Database\n", level);
	else trace(TR_ISIS, LOG_INFO, "******* L%d LSP Database", level);
	while (ent) {
		(void) correctedLifetime(ent);	/* correct the lifetime for display */
		/* STEVE - if I only knew how strcat worked ... */
		if (!locallyGenerated(ent) || ent->onDeletedList) {
			if (locallyGenerated(ent)) sprintf(scratch, "local|will delete|rt:%d", ent->remainingTime);
			else if (ent->onDeletedList) sprintf(scratch, "\twill delete|rt:%d", ent->remainingTime);
			else sprintf(scratch, "rt:%d", ent->remainingTime);
		} else sprintf(scratch, "local");
		if (fd) {
			fprintf(fd, "LSP %s  \t%s\n", LSSigToStr(&ent->signature), scratch);
			printLSPContents(fd, (char *) ent->buf, ent->len);
		} else {
			trace(TR_ISIS, LOG_INFO, "LSP %s  \t%s", LSSigToStr(&ent->signature), scratch);
			IFTRACE(T_LSPCONTENT)
				printLSPContents(0, (char *) ent->buf, ent->len);
			ENDTRACE
		}
		ent = ent->sortNext;
	}
	if (fd) fprintf(fd, "\n");
	else trace(TR_ISIS, LOG_INFO, NULL);
}


void
dumpDB()
{
	IFTRACE(T_DUMPDB)
		dumpLSPDB(0, 1);
		if (systemType == L2IS)
			dumpLSPDB(0, 2);
	ENDTRACE

	setTimer(dumpDBinterval, dumpDB);
}

/* flood all lsps on a particular circuit */
void
floodAllLSPs(c, level)
CircuitEntry	*c;
int				level;
{
	LSEntry	*ent = firstLSP(level);

	while (ent) {
		setFlags(ent->SRM, SingleCircuit, CEToIndex(c));
		ent = ent->sortNext;
	}
}


/*
 * ------------------------------------------------------------------------
 * 
 * 	GateD, Release 3
 * 
 * 	Copyright (c) 1990,1991,1992,1993 by Cornell University
 * 	    All rights reserved.
 * 
 * 	THIS SOFTWARE IS PROVIDED "AS IS" AND WITHOUT ANY
 * 	EXPRESS OR IMPLIED WARRANTIES, INCLUDING, WITHOUT
 * 	LIMITATION, THE IMPLIED WARRANTIES OF MERCHANTABILITY
 * 	AND FITNESS FOR A PARTICULAR PURPOSE.
 * 
 * 	Royalty-free licenses to redistribute GateD Release
 * 	3 in whole or in part may be obtained by writing to:
 * 
 * 	    GateDaemon Project
 * 	    Information Technologies/Network Resources
 * 	    200 CCC
 * 	    Cornell University
 * 	    Ithaca, NY  14853-2601  USA
 * 
 * 	GateD is based on Kirton's EGP, UC Berkeley's routing
 * 	daemon	 (routed), and DCN's HELLO routing Protocol.
 * 	Development of GateD has been supported in part by the
 * 	National Science Foundation.
 * 
 * 	Please forward bug fixes, enhancements and questions to the
 * 	gated mailing list: gated-people@gated.cornell.edu.
 * 
 * 	Authors:
 * 
 * 		Jeffrey C Honig <jch@gated.cornell.edu>
 * 		Scott W Brim <swb@gated.cornell.edu>
 * 
 * ------------------------------------------------------------------------
 * 
 *       Portions of this software may fall under the following
 *       copyrights:
 * 
 * 	Copyright (c) 1988 Regents of the University of California.
 * 	All rights reserved.
 * 
 * 	Redistribution and use in source and binary forms are
 * 	permitted provided that the above copyright notice and
 * 	this paragraph are duplicated in all such forms and that
 * 	any documentation, advertising materials, and other
 * 	materials related to such distribution and use
 * 	acknowledge that the software was developed by the
 * 	University of California, Berkeley.  The name of the
 * 	University may not be used to endorse or promote
 * 	products derived from this software without specific
 * 	prior written permission.  THIS SOFTWARE IS PROVIDED
 * 	``AS IS'' AND WITHOUT ANY EXPRESS OR IMPLIED WARRANTIES,
 * 	INCLUDING, WITHOUT LIMITATION, THE IMPLIED WARRANTIES OF
 * 	MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 * ------------------------------------------------------------------------
 * 
 * 	Copyright 1991 D.L.S. Associates
 * 
 * 	Permission to use, copy, modify, distribute, and sell this software
 * 	and its documentation for any purpose is hereby granted without
 * 	fee, provided that the above copyright notice appear in all copies
 * 	and that both that copyright notice and this permission notice
 * 	appear in supporting documentation, and that the name of D.L.S. not
 * 	be used in advertising or publicity pertaining to distribution of
 * 	the software without specific, written prior permission.  D.L.S.
 * 	makes no representations about the suitability of this software for
 * 	any purpose.  It is provided "as is" without express or implied
 * 	warranty.
 * 
 * 	D.L.S. DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * 	INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN
 * 	NO EVENT SHALL D.L.S.  BE LIABLE FOR ANY SPECIAL, INDIRECT OR
 * 	CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS
 * 	OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
 * 	NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN
 * 	CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 * 
 * 	Authors:  Robert Hagens and Dan Schuh
 * 
 * 
 */
