;; Modified by Thomas Gauweiler <gauweil@ira.uka.de> on 27-Jan-94 (Modula 2*)
;; Modified by Mitchell Tasman <tasman@cs.wisc.edu> on 24-Feb-93 (Fix 5)
;; From: yoshio@cs.ucla.edu (Yoshio Turner)
;;
;; Article 5469 of comp.emacs:
;; Path: ucla-cs!rutgers!mit-eddie!PIE8.PIE.CS.CMU.EDU!John.Myers
;; From: John.Myers@PIE8.PIE.CS.CMU.EDU
;; Newsgroups: comp.emacs
;; Subject: m2-mode.el
;; Message-ID: <8901062058.AA02965@EDDIE.MIT.EDU>
;; Date: 6 Jan 89 19:31:33 GMT
;; Sender: uucp@eddie.MIT.EDU
;; Lines: 552

;; Electric Modula-2* mode, version 1.0

;; Copyright (C) 1988 Free Software Foundation, Inc.

;; GNU Emacs is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY.  No author or distributor
;; accepts responsibility to anyone for the consequences of using it
;; or for whether it serves any particular purpose or works at all,
;; unless he says so in writing.  Refer to the GNU Emacs General Public
;; License for full details.

;; Everyone is granted permission to copy, modify and redistribute
;; GNU Emacs, but only under the conditions described in the
;; GNU Emacs General Public License.   A copy of this license is
;; supposed to have been given to you along with GNU Emacs so you
;; can know your rights and responsibilities.  It should be in a
;; file named COPYING.  Among other things, the copyright notice
;; and this notice must be preserved on all copies.

;; Written by Tom Lord (lord+@andrew.cmu.edu)
;; Modified by Yoshio Turner (yoshio@cs.ucla.edu) to do case-insensitive search
;; with C-s and C-r when in m2s-mode.

;; To enable, place the following lines in .emacs:
;;
;; (autoload 'm2s-mode "m2s-mode" "Modula-2* mode" t)
;; (setq auto-mode-alist
;;       (append '(("\\.msd$" . m2s-mode) ("\\.msi$" . m2s-mode) ("\\.msm$" . m2s-mode))
;;	       auto-mode-alist))

;; Modula-2* is a filthy blotch upon the earth, but that doesn't save some
;; of you FROM having TO use it.  IF you are one OF those unfortunates, AND 
;; IF you'd like to try out a new gnu-emacs modula-2* mode, THEN now's your
;; big chance. (END)
;; 
;; Features OF m2s-mode:
;; 
;;   Autocapitalization OF keywords (but NOT within comments)
;;   Autoindentation
;;   moderately ok comment support
;; 
;; Autocapitalization works this way:  just TYPE all OF your keywords IN lower
;; CASE, AND as you go along, emacs will automatically capitalize them FOR you.
;; 
;; Autoindentation: never use the RETURN key IN an m2s-mode buffer.
;; Always hit control-j at the END OF a line.  Control-j will NOT only
;; open up a NEW line, but will indent on that line as much as you
;; probably want.  BUT! IF the indentation isn't right, AND IF you're NOT
;; typing a multi-line simple-statement, just keep typing!  Typing
;; certain keywords NOT only triggers the autocapitalization function,
;; but also triggers the autoindentation function.  So, FOR example:
;; 
;; Before C-j
;;      IF foo THEN
;;         bar^
;; 
;; After C-j
;;      IF foo THEN
;;         bar
;;             ^
;; (the indentation is expecting the statement ``bar'' TO be continued).
;; 
;; After typing end;
;; 
;;     IF foo THEN
;;        bar
;;     END;
;; 
;; (after typing the semi-colon, the editor capitalized END AND corrected
;; the indentation FOR the line).
;; 
;; Finally, comment magic:
;; Typing M-; inserts an empty comment on the current line, OR ELSE puts
;; the point inside OF an existing comment on the current line.  M-j 
;; continues a comment on the following line.

(provide 'm2s-mode)

(defvar m2s-mode-map nil
  "Modula2* mode keymap.")

(if m2s-mode-map
    ()
  (setq m2s-mode-map (make-sparse-keymap))
  (define-key m2s-mode-map "\C-j" 'm2s-newline-and-indent)
  (define-key m2s-mode-map "\C-s" 'case-isens-isearch-forward) ; yoshio
  (define-key m2s-mode-map "\C-r" 'case-isens-isearch-backward) ; yoshio
  (define-key m2s-mode-map "\C-i" 'm2s-tab-command)
  (define-key m2s-mode-map "\M-;" 'm2s-indent-for-comment)
)

;yoshio
(defun case-isens-isearch-forward ()
  "Like ISEARCH-FORWARD, only always case-insensitive"
  (interactive)
  (let ((tmp case-fold-search))
    (unwind-protect
	(progn
	  (setq case-fold-search t)
	  (isearch-forward))
      (setq case-fold-search tmp))))

;yoshio
(defun case-isens-isearch-backward ()
  "Like ISEARCH-BACKWARD, only always case-insensitive"
  (interactive)
  (let ((tmp case-fold-search))
    (unwind-protect
	(progn
	  (setq case-fold-search t)
	  (isearch-backward))
      (setq case-fold-search tmp))))


(defun m2s-indent-for-comment ()
  "Like INDENT-FOR-COMMENT, but expands abbrevs first."
  (interactive)
  (expand-abbrev)
  (indent-for-comment))

;; Create the modula2* syntax table 
;; The default syntax table is taken from the existing m2s.el
(defvar m2s-mode-syntax-table ()
  ()
  "Modula2*'s syntax table")

;(if m2s-mode-syntax-table
;    ()
  (setq m2s-mode-syntax-table (make-syntax-table))
  (modify-syntax-entry ?\\ "\\" m2s-mode-syntax-table)
  (modify-syntax-entry ?\( ". 1" m2s-mode-syntax-table)
  (modify-syntax-entry ?\) ". 4" m2s-mode-syntax-table)
  (modify-syntax-entry ?* ". 23" m2s-mode-syntax-table)
  (modify-syntax-entry ?+ "." m2s-mode-syntax-table)
  (modify-syntax-entry ?- "." m2s-mode-syntax-table)
  (modify-syntax-entry ?= "." m2s-mode-syntax-table)
  (modify-syntax-entry ?% "." m2s-mode-syntax-table)
  (modify-syntax-entry ?< "." m2s-mode-syntax-table)
  (modify-syntax-entry ?> "." m2s-mode-syntax-table)
  (modify-syntax-entry ?\' "\"" m2s-mode-syntax-table)
  (modify-syntax-entry ?_ "_" m2s-mode-syntax-table)
;)

;; Modula-2* has silly capitalization conventions.  To smooth these over
;; a bit, gnu-emacs' abbrev mode is used.  Users type identifiers in lower 
;; case, and the editor corrects capitalization on the fly.
;; Abbrevs are also used in the indentation process.

(defvar m2s-auto-capitalize t)
(defvar m2s-voltage-on t)

(defvar m2s-keywords-for-capitalization
  '("AND" "ARRAY" "BEGIN" "BY" 
    "CASE" "CONST" "DEFINITION" "DIV" "DO" "ELSE" "ELSIF" "END" "EXIT"
    "EXPORT" "FOR" "FROM" "IF" "IMPLEMENTATION" "IMPORT" "IN" "INTERFACE"
    "LOOP" "MOD" "MODULE" "NOT" "OF" "OR" "POINTER" "PROCEDURE"
    "QUALIFIED" "RECORD" "REPEAT" "RETURN" "SET" "THEN" "TO" "TYPE"
    "UNTIL" "VAR" "WHILE" "WITH" "BITSET" "BOOLEAN" "CARDINAL" "CHAR"
    "FALSE" "INTEGER" "LONGCARD" "LONGINT" "LONGREAL" "MAX" "MIN" "NIL" "PROC" "REAL"
    "TRUE" "MaxCard" "MaxInt" "MinInt" "ABS" "CAP" "CHR" "DEC" "DISPOSE"
    "EXCL" "FLOAT" "HALT" "HIGH" "INC" "INCL" "LENGTHEN" "LONGFLOAT" "MAX"
    "MIN" "NEW" "ODD" "ORD" "SHORTEN" "TRUNC" "VAL" "SYSTEM" "ALLOCATE"
    "DEALLOCATE" "WORD" "ADDRESS" "PROCESS" "BYTE" "ADR" "SIZE" "TSIZE"
    "NEWPROCESS" "TRANSFER" "WRITEF" "READF" "FWRITEF" "FREADF"
    "FORALL" "SYNC" "PARALLEL")
  "The default set of words to capitalization correct")

(defvar m2s-keywords-for-reindentation
  '("MODULE" "PROCEDURE" "BEGIN" "END" "VAR" "IF" "THEN" "ELSE" "ELSIF"
    "LOOP" "UNTIL" "DO" "IMPORT" "EXPORT"))

(defvar m2s-abbrev-table ()
  "The abbrev table used in modula-2* mode for capitalization and indentation 
dispatching")
(define-abbrev-table 'm2s-abbrev-table ())

(defvar m2s-capitalizations ()
  "The abbrev table used to handle modula-2*'s capitalizations")
(define-abbrev-table 'm2s-capitalizations ())

(defvar m2s-electrification ())
(define-abbrev-table 'm2s-electrification ())

;;(defun m2s-guarantee-abbrev-correctness ()
;;  (save-excursion
;;    (if (m2s-inside-comment-p (point))
;;	(unexpand-abbrev))
;;    (m2s-reindent-line)))

(defun m2s-abbrev-dispatcher ()
  (if (or (m2s-inside-comment-p (point))
	  (m2s-inside-string-p (point)))
      (progn (unexpand-abbrev) nil)
    (if m2s-auto-capitalize
	(let ((local-abbrev-table m2s-capitalizations)
	      saved-word)
	  (save-excursion
; Mitchell Tasman added left and right bounds recording -- otherwise might
; accidentally downcase text immediately following point.
	    (setq right (point))
	    (forward-word -1)
	    (setq left (point))
	    (setq saved-word
		  (buffer-substring (point)
				    (save-excursion (forward-word 1) (point))))
; Mitchell Tasman changed from downcase-word to downcase-region
	    (downcase-region left right))
	  (if (expand-abbrev)
	      nil
	    (forward-word -1)
	    (delete-char (length saved-word))
	    (insert saved-word))))
    (if m2s-voltage-on
	(let ((local-abbrev-table m2s-electrification))
	  (expand-abbrev)))))
    

(defun m2s-electric-word (word)
  (define-abbrev m2s-abbrev-table (downcase word) (downcase word) 'm2s-abbrev-dispatcher))

(defun m2s-learn-to-capitalize (word)
  "Learn to expand lowercased WORD into WORD, preserving any abbrev hooks"
  (interactive "sWord? ")
  (define-abbrev m2s-capitalizations (downcase word) word nil)
  (m2s-electric-word word))

(defun inside-indentation ()
  "True if the point is inside of indentation."
  (save-excursion
    (skip-chars-backward " \t")
    (= (point) (bol-point))))

(defun m2s-correct-indentation (&optional dontmove)
  "Correct the indentation of the current line.  If the point winds up in
whitespace, leave it at the start of the text, otherwise don't move the point.
Optional parameter, if non nil, also prevents cursor motion."
  (interactive)
  (if (and (not dontmove) (inside-indentation))
      (m2s-reindent-line)
    (save-excursion
      (m2s-reindent-line))))

(defun m2s-learn-to-indent (word)
  "Learn to reindent after keyword WORD using m2s-correct-indentation, preserving any
existing expantion."
  (define-abbrev m2s-electrification (downcase word) (downcase word)
    (function (lambda () (m2s-correct-indentation t))))
  (m2s-electric-word word))

  
(defun m2s-tab-command (prefix)
  "Normal TAB if inside a comment, otherwise, reindent line."
  (interactive "p")
  (cond ((m2s-inside-comment-p (point))
	 (if (inside-indentation)
	     (indent-relative)
	   (self-insert-command prefix)))
	((m2s-inside-string-p (point)) (self-insert-command prefix))
	(t (indent-for-tab-command))))

(defun m2s-newline-and-indent ()
  "Just like newline-and-indent, but causes a preceding abbrev to be expanded"
  (interactive)
  (expand-abbrev)
  (newline-and-indent))

;; set up the default abbrevs:
(mapcar 'm2s-learn-to-capitalize m2s-keywords-for-capitalization)
(mapcar 'm2s-learn-to-indent m2s-keywords-for-reindentation)

(defun m2s-mode ()
  "Major mode for editing Modula-2* code.
Tab indents for Modula-2* code.
Abbrev mode is used to correctly capitalize and indent Modula-2* keywords.
This auto-capitalization is not done inside comments.
\\{m2s-mode-map}
Variables controlling indentation style:
 m2s-indent-level
    The number of spaces to indent each statement block.
 m2s-case-label-outdent
    The number of spaces to outdent case labels.

Turning on Modula-2* mode calls the value of the variable m2s-mode-hook with
no args, if that value is non-nil."
  (interactive)
  (kill-all-local-variables)
  (setq major-mode 'm2s-mode)
  (setq mode-name "Modula2")
  (setq case-fold-search nil)
  (use-local-map m2s-mode-map)
  (run-hooks 'm2s-mode-hook)
  (abbrev-mode 7)
  (make-local-variable 'indent-line-function)
  (setq indent-line-function 'm2s-correct-indentation)
  (make-local-variable 'comment-start)
  (setq comment-start "(* ")
  (make-local-variable 'comment-end)
  (setq comment-end " *)")
  (make-local-variable 'comment-start-skip)
  (setq comment-start-skip "(\\*+[ \t]*")
  (make-local-variable 'comment-indent-hook)
  (setq comment-indent-hook 'm2s-comment-indent-hook)
  (make-local-variable 'local-abbrev-table)
  (setq local-abbrev-table m2s-abbrev-table))

;; indentation code for modula-2*
(defun m2s-inside-comment-p (pos)
  "Return true if POS is inside of a modula-2* comment"
  (save-excursion
    (goto-char pos)
    (and (search-backward "(*" nil t)
	 (not (search-forward "*)" pos t)))))


(defun m2s-comment-start-line-p (pos)
  "True if pos is on a line containing only the beginning of a comment"
  (save-excursion
    (beginning-of-line)
    (skip-chars-forward " \t")
    (looking-at "(*")))

(defun m2s-beginning-of-comment ()
  "Move to the beginning of the containing comment"
  (search-backward "(*"))

(defun m2s-end-of-comment ()
  "Move to the end of the containing comment"
  (search-forward "*)" nil 'leaveatend))

;; All of the other syntax predicates presume that pos is NOT inside of a 
;; comment.

(defun m2s-skip-chars-forward (chars)
  "Skip forward over chars in CHARS, and over comments"
  (skip-chars-forward chars)
  (while (and (< (point) (point-max))
	      (looking-at "(\\*"))
    (m2s-end-of-comment)
    (skip-chars-forward chars)))


(defun m2s-skip-chars-backward (chars)
  "Skip backward over chars in CHARS, and over comments"
  (skip-chars-backward chars)
  (while (and (> (point) (point-min))
	      (m2s-inside-comment-p (- (point) 1)))
    (m2s-beginning-of-comment)
    (skip-chars-backward chars)))

(defun m2s-following-bob-p (pos)
  "True if pos is not preceded by code."
  (save-excursion
    (m2s-skip-chars-backward " \t\n")
    (= (point) 1)))

;; This assumes that strings must begin and end on the same line:
(defun m2s-inside-string-p (pos)
  "Test whether POS falls inside of a modula-2* string"
  (save-excursion
    (goto-char pos)
    (beginning-of-line)
    (let ((looking-for-delimeter nil))
      (while (< (point) pos)
	(let ((char (char-after (point))))
	  (if (eq looking-for-delimeter char)
	      (setq looking-for-delimeter nil)
	    (if (and (null looking-for-delimeter)
		     (or (eq ?\' char) (eq ?\" char)))
		(setq looking-for-delimeter char)))
	  (forward-char 1)))
      looking-for-delimeter)))
					  

(defun m2s-looking-at (regexp)
  "Like looking-at, but skips intervening whitespace and modula-2* comments"
  (save-excursion
    (m2s-skip-chars-forward " \t")
    (looking-at regexp)))

(defun m2s-looking-back-at (regexp)
  "True if a match for regexp followed by whitespace and modula-2* comments
precededs the point.  False if the previous match falls inside a string."
  (save-excursion
    (let ((old-pos (point)))
	  (and (re-search-backward regexp nil t)
	       (save-excursion
		 (goto-char (match-end 0))
		 (m2s-skip-chars-forward "\n\t ")
		 (>= (point) old-pos))
	       (not (m2s-inside-string-p (point)))))))


(defconst m2s-terminated-statement ";")
(defconst m2s-END "END")
(defconst m2s-block-openers
  "\\(DO\\|THEN\\|ELSE\\|LOOP\\|ELSIF\\|REPEAT\\|BEGIN\\|RECORD\\|SYNC\\|PARALLEL\\)")
(defconst m2s-block-enders  "\\(END\\|ELSE\\|ELSIF\\|UNTIL\\)")
(defconst m2s-begin "\\(BEGIN\\)")

; Mitchell Tasman redefined m2s-scope-header, in order to more accurately
; describe a Modula-2* PROCEDURE definition.
(defconst m2s-scope-header 
"\\(MODULE\\([ \t\n]+[a-zA-Z0-9].*;\\)?\\|PROCEDURE\\([^(;]*;\\|[^)]*).*;\\)\\)")

; Earlier attempts to RegExp match a PROCEDURE definition...
;PROCEDURE\\(\\([^)]\\|\n\\)*)\\)?\\([^;]\\|\n\\)*;\\)
;PROCEDURE\\(\\(\\([^)]\\|\n\\)*)\\)?[^;]*;\\)
(defconst m2s-scope-section
  "\\(VAR.*\\|TYPE.*\\|CONST.*\\|FROM\\([^;]*;\\)?\\|EXPORT\\([^;]*;\\)?\\|END\\([ \t]*[a-zA-Z].*;\\)?\\|IMPORT\\([^;]*;\\)?\\)"
)
(defconst m2s-case-label-context "OF\\||")

(defvar m2s-indent-level 4
  "The number of spaces to indent by for modula 2*")
(defvar m2s-case-label-outdent 2
  "The number of spaces case lables should stick out.")

(defun m2s-deindent (level)
  "Return Max(0, level - m2s-indent-level)"
  (max 0 (- level m2s-indent-level)))

(defun m2s-indent (level)
  "Add m2s-indent-level to level"
  (+ m2s-indent-level level))

(defun preceeding-lines-indentation ()
  "Return preceeding lines indent-level, or zero if there is not preceeding 
line"
  (save-excursion
    (beginning-of-line)
    (if (= (point) (point-min))
	0
      (forward-char -1)
      (current-indentation))))

(defun m2s-context-indent (pos)
  "Return the indentation level of the line preceding POS.  Will search
backward to determine if POS is a continued statement."
  (save-excursion
    (goto-char pos)
    (beginning-of-line)
    (if (= (point) (point-min))
	0;
      (m2s-skip-chars-backward " \n\t")
      (m2s-statement-begin-indent))))

(defun m2s-statement-begin-indent ()
  "Report the indentation level of the beginning of a statement."
  (m2s-statement-begin-line)
  (current-indentation))

(defun m2s-statement-begin-line ()
  "Move to the beginning of the statement containing the point."
  (beginning-of-line)
  (and (not (eq (point) (point-min)))
       (not (m2s-looking-back-at m2s-block-openers))
       (not (m2s-looking-back-at m2s-terminated-statement))
       (not (m2s-looking-back-at m2s-scope-header))
       (not (m2s-looking-back-at m2s-scope-section))
       (not (m2s-looking-at m2s-block-enders))
       (not (m2s-looking-back-at m2s-case-label-context))
       (progn
	 (forward-char -1)
;; Mitchell Tasman changed the following from "(m2s-statement-begin-indent)".
 	 (m2s-statement-begin-line))))

(defun m2s-case-labled-line ()
  (save-excursion
    (beginning-of-line)
    (m2s-looking-back-at m2s-case-label-context)))

;; Mitchell Tasman changed the following function so that it will
;; skip whitespace preceding the current line.
(defun m2s-prev-statement-case-lable ()
  (save-excursion
    (beginning-of-line)
    (if (= (point) (point-min))
	nil;
      (m2s-skip-chars-backward " \n\t")
      (m2s-statement-begin-line)
      (m2s-case-labled-line))))

(defun m2s-indentation-level (pos)
  "Return the number of spaces the line at POS should be indented.  Assumes
that the line preceeding POS (if any) is correctly indented.  Returns -1
if the current line shouldn't be adjusted."
  (save-excursion
    (goto-char pos)
    (beginning-of-line)
    (let* ((line-start (bol-point))
	   (context-indent (m2s-context-indent pos))
	   (follows-statement (m2s-looking-back-at m2s-terminated-statement))
	   (follows-END (m2s-looking-back-at m2s-END))
	   (follows-block-opener (m2s-looking-back-at m2s-block-openers))
	   (follows-scope-header (m2s-looking-back-at m2s-scope-header))
	   (follows-scope-section (m2s-looking-back-at m2s-scope-section))
	   (is-begin (m2s-looking-at m2s-begin))
	   (is-scope-header (m2s-looking-at m2s-scope-header))
	   (is-scope-section (m2s-looking-at m2s-scope-section))
	   (is-block-ender (m2s-looking-at m2s-block-enders)))
      (cond ((m2s-inside-comment-p line-start)
	     (if (m2s-comment-start-line-p pos)
		 context-indent
	       -1))
	    ((m2s-following-bob-p line-start) 0)
	    (is-begin
	     (cond (follows-scope-header context-indent)
		   (follows-scope-section (m2s-deindent context-indent))
		   ((m2s-looking-back-at "END[ \t]*[a-zA-Z].*;")
		    (m2s-deindent context-indent))
		   (t (m2s-deindent (m2s-deindent context-indent)))))
	    (is-scope-header
	     (cond (follows-scope-header (m2s-indent context-indent))
		   (follows-scope-section context-indent)
		   (t (m2s-deindent context-indent))))
	    (is-scope-section
	     (cond (follows-scope-header (m2s-indent context-indent))
		   (follows-END (m2s-deindent context-indent))
		   (follows-scope-section context-indent)
		   (follows-block-opener context-indent)
;;; Mitchell Tasman changed the following to outdent by
;;; (- m2s-indent-level m2s-case-label-outdent), if we haven't
;;; yet seen a complete statement following the last case label.
		   (t (if (m2s-prev-statement-case-lable)
			  (- context-indent 
			     (- m2s-indent-level m2s-case-label-outdent))
			(m2s-deindent context-indent)))))
	    (is-block-ender
	     (cond (follows-block-opener context-indent)
		   ((m2s-prev-statement-case-lable)
		    (- context-indent
		       (- m2s-indent-level m2s-case-label-outdent)))
		   (t (m2s-deindent context-indent))))
	    (follows-END (m2s-deindent context-indent))
	    (follows-scope-section (m2s-indent context-indent))
;;; Mitchell Tasman changed the following to indent by m2s-case-label-outdent,
;;; if we haven't yet seen a complete statement following the last case label.
	    (follows-block-opener
		    (if (m2s-prev-statement-case-lable)
			(m2s-indent (+ context-indent m2s-case-label-outdent))
			(m2s-indent context-indent)))
	    ((m2s-looking-back-at "OF")
	     (+ context-indent (- m2s-indent-level m2s-case-label-outdent)))
	    ((m2s-looking-back-at "|")
;; Mitchell Tasman notes that this rather strange test appears to be
;; necessary.
	     (if (m2s-prev-statement-case-lable)
		 context-indent
	       (- context-indent m2s-case-label-outdent)))
;; Mitchell Tasman reworked the following logic for determining if we need
;; to correct for a previous outdent.  Note that m2s-prev-statement-case-lable
;; [sic] is true for the SECOND as well as the first statement after a case
;; label!  This is because m2s-statement-begin-indent decides, for better or
;; worse, that that the previous statement (i.e., the first after a case
;; label) is actually a continuation of the case label itself!
	    ((m2s-prev-statement-case-lable)
	     (+ context-indent m2s-case-label-outdent))
	    (follows-statement
	     context-indent)
	    (t ; continuing a statement? let the user work it out
	     (let ((current (current-indentation)))
	       (if (> current 0)
		   current
		 (m2s-indent context-indent))))))))

(defun m2s-reindent-line ()
  "Indent the current line according to m2s-indentation-level.
Leaves point at the end of the indentation."
  (beginning-of-line)
  (let ((level (m2s-indentation-level (point))))
    (if (= level -1)
	nil
      (delete-horizontal-space)
      (indent-to-column level)
      (move-to-column level))))

(defun m2s-comment-indent-hook ()
  (if (inside-indentation)
      (m2s-indentation-level (point))
    comment-column))

;; these are functions that should go back into the distribution in other
;; places

;; from replace.el
;;
(defun number-matches (regexp &optional start end)
  "Return the number of matches for REGEXP that ocurr between START and END."
  (let ((count 0) opoint)
    (or start (setq start (point)))
    (or end (setq end (point-max)))
    (save-excursion
      (goto-char start)
      (while (and (not (eobp))
		  (not (> (point) end))
		  (progn (setq opoint (point))
			 (re-search-forward regexp end t)))
	;; This is as it was in replace.el, but it looks pretty silly to me.
	;; -T
	(if (= opoint (point))
	    (forward-char 1)
	  (setq count (1+ count))))
      count)))

(defun how-many (regexp)
    "Print number of matches for REGEXP following point."
    (interactive "sHow many matches for (regexp): ")
    (message "%d occurrences" (number-matches regexp)))

;; needed:
(defun bol-point ()
  "Return the point at the beginning of the line"
  (save-excursion
    (beginning-of-line)
    (point)))

(defun eol-point ()
  "Return the point at the end of the line"
  (save-excursion
    (end-of-line)
    (point))) 


