------------------------------------------------------------------------------
--                                                                          --
--                         GNAT COMPILER COMPONENTS                         --
--                                                                          --
--                             W I D E C H A R                              --
--                                                                          --
--                                 B o d y                                  --
--                                                                          --
--                            $Revision: 1.3 $                              --
--                                                                          --
--           Copyright (c) 1992,1993,1994 NYU, All Rights Reserved          --
--                                                                          --
-- GNAT is free software;  you can  redistribute it  and/or modify it under --
-- terms of the  GNU General Public License as published  by the Free Soft- --
-- ware  Foundation;  either version 2,  or (at your option) any later ver- --
-- sion.  GNAT is distributed in the hope that it will be useful, but WITH- --
-- OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
-- or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License --
-- for  more details.  You should have  received  a copy of the GNU General --
-- Public License  distributed with GNAT;  see file COPYING.  If not, write --
-- to the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA. --
--                                                                          --
------------------------------------------------------------------------------

--  This package encapsulates the method used to represent wide characters
--  using ESC sequences. The assumption is that each wide character is always
--  represented with a separate ESC sequences, starting with an ESC character.
--  It is also assumed that the ESC sequence cannot contain an embedded
--  format effector character (CR, LF, FF, VT, HT).

--  Other methods of representing wide characters (Unicode, shift in/out etc)
--  will require more extensive changes to the scanner.

--  The representation used in this version is as follows:

--   If the character code is in the range 16#0000# .. 16#00FF# then it is
--   never represented using an ESC sequence, instead the second byte
--   appears as a Latin-1 character in the usual manner.

--   If both the high order and low order bytes of the 16-bit value are in
--   the range 16#20# .. 16#FF#, then the ESC sequence is simply:

--      ESC   high-order-byte   low-order-byte

--   If either the high order or low order bytes of the 16-bit value are in
--   the range 16#00# .. 16#1F#, then the ESC sequence is:

--      ESC  16#00#  flag-byte  high-order-byte  low-order-byte

--   where flag-byte has one of the following settings:

--      1 = subtract 16#20# from high order byte value
--      2 = subtract 16#20# from low order byte value
--      3 = subtract 16#20# from both low and high order byte value

--   This allows the encoding of these cases without control characters
--   appearing in the ESC sequence as required by the rules.

package body Widechar is

   ---------------
   -- Scan_Wide --
   ---------------

   procedure Scan_Wide
     (S : Source_Buffer_Ptr;
      P : in out Source_Ptr;
      C : out Char_Code;
      E : out Boolean)
   is
      Code : Natural;

      subtype Flag_Range is Byte range 0 .. 3;
      --  Valid values for Flag_Byte including internal value of zero used
      --  to indicate that neither byte of the sequence needs adjusting

      Flag_Byte : Flag_Range;

   begin
      pragma Assert (S (P) = ESC);
      P := P + 1;

      if S (P) = NUL then
         Flag_Byte := Character'Pos (S (P));

         if Flag_Byte not in 1 .. 3 then
            E := True;
            return;
         else
            P := P + 1;
         end if;

      else
         Flag_Byte := 0;
      end if;

      if S (P) < ' ' then
         E := True;
         return;
      end if;

      Code := 256 * Character'Pos (S (P));
      P := P + 1;

      if S (P) < ' ' then
         E := True;
         return;
      end if;

      Code := Code + Character'Pos (S (P));
      P := P + 1;

      case Flag_Range (Flag_Byte) is

         when 0 =>
            null;

         when 1 =>
            Code := Code - 256 * 16#20#;

         when 2 =>
            Code := Code - 16#20#;

         when 3 =>
            Code := Code - (256 * 16#20# + 16#20#);

      end case;

      C := Char_Code (Code);
      E := False;
   end Scan_Wide;

   ----------------
   -- Set_Escape --
   ----------------

   procedure Set_Escape
     (C : Char_Code;
      S : in out String;
      P : in out Natural)
   is
      C1   : Natural := Natural (C) / 256;
      C2   : Natural := Natural (C) mod 256;
      Code : Natural;

   begin
      pragma Assert (C in 16#0100# .. 16#FFFF#);
      P := P + 1;
      S (P) := ESC;

      if C1 >= 16#20# and then C2 >= 16#20# then
         S (P + 1) := Character'Val (C1);
         S (P + 2) := Character'Val (C2);
         P := P + 2;
         return;

      else
         Code := 0;

         if C1 < 16#20# then
            C1 := C1 + 16#20#;
            Code := Code + 1;
         end if;

         if C2 < 16#20# then
            C2 := C2 + 16#20#;
            Code := Code + 2;
         end if;

         S (P + 1) := Character'Val (16#00#);
         S (P + 2) := Character'Val (Code);
         S (P + 3) := Character'Val (C1);
         S (P + 4) := Character'Val (C2);

         P := P + 4;
      end if;
   end Set_Escape;

   -----------------
   -- Skip_Escape --
   -----------------

   procedure Skip_Escape (S : String; P : in out Natural) is
   begin
      pragma Assert (S (P) = ESC);

      if S (P + 1) /= NUL then
         P := P + 3;
      else
         P := P + 5;
      end if;
   end Skip_Escape;

end Widechar;
