------------------------------------------------------------------------------
--                                                                          --
--                         GNAT COMPILER COMPONENTS                         --
--                                                                          --
--                              S T R I N G T                               --
--                                                                          --
--                                 S p e c                                  --
--                                                                          --
--                            $Revision: 1.20 $                             --
--                                                                          --
--           Copyright (c) 1992,1993,1994 NYU, All Rights Reserved          --
--                                                                          --
-- GNAT is free software;  you can  redistribute it  and/or modify it under --
-- terms of the  GNU General Public License as published  by the Free Soft- --
-- ware  Foundation;  either version 2,  or (at your option) any later ver- --
-- sion.  GNAT is distributed in the hope that it will be useful, but WITH- --
-- OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
-- or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License --
-- for  more details.  You should have  received  a copy of the GNU General --
-- Public License  distributed with GNAT;  see file COPYING.  If not, write --
-- to the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA. --
--                                                                          --
------------------------------------------------------------------------------

with System; use System;
with Types;  use Types;
with Table;

package Stringt is

--  This package contains routines for handling the strings table which is
--  used to store string constants encountered in the source, and also those
--  additional string constants generated by compile time concatenation and
--  other similar processing.

--  A string constant in this table consists of a series of Char_Code values,
--  so that 16-bit character codes can be properly handled if this feature
--  is implemented in the scanner.

--  There is no guarantee that hashing is used in the implementation, although
--  it maybe. This means that the caller cannot count on having the same Id
--  value for two identical strings stored separately and also cannot count on
--  the two Id values being different.

   --------------------------------------
   -- String Table Access Subprograms --
   --------------------------------------

   function End_String return String_Id;
   pragma Inline (End_String);
   --  Terminates current string and returns its Id

   function Get_String_Char (Id : String_Id; Index : Int) return Char_Code;
   --  Obtains the specified character from a stored string. The lower bound
   --  of stored strings is always 1, so the range is 1 .. String_Length (Id).

   procedure Initialize;
   --  Initializes the strings table for a new compilation

   procedure Start_String;
   --  Sets up for storing a new string in the table. To store a string, a
   --  call is first made to Start_String, then successive calls are
   --  made to Store_String_Character to store the characters of the string.
   --  Finally, a call to End_String terminates the entry and returns it Id.

   procedure Start_String (S : String_Id);
   --  Like Start_String with no parameter, except that the contents of the
   --  new string is initialized to be a copy of the given string. A test is
   --  made to see if S is the last created string, and if so it is shared,
   --  rather than copied, this can be particularly helpful for the case of
   --  a continued concatenaion of string constants.

   procedure Store_String_Char (C : Char_Code);
   --  Store next character of string, see description above for Start_String

   function String_Chars_Address return System.Address;
   --  Return address of String_Chars table (used by Back_End call to Gigi)

   function String_Length (Id : String_Id) return Nat;
   pragma Inline (String_Length);
   --  Returns length of previously stored string

   function Strings_Address return System.Address;
   --  Return address of Strings table (used by Back_End call to Gigi)

   procedure Unstore_String_Char;
   --  Undoes effect of previous Store_String_Char call, used in some error
   --  situations of unterminated string constants.

   procedure Write_Char_Code (Code : Char_Code);
   --  Procedure to write a character code value, used for debugging purposes
   --  for writing character codes. If the character code is in the range
   --  16#20# .. 16#7A# or 16#7C# .. 16#7F# then the single graphic character
   --  corresponding to the code is output. For any other codes in the range
   --  16#00# .. 16#FF#, the code is output as {hh} where hh is the two
   --  digit hex value for the code. Codes greater than 16#FF# are output as
   --  {hhhh} where hhhh is the four digit hex representation of the code
   --  value (high order byte first). Note: the reason that we output the
   --  code 7B (left brace) in hex is to avoid escape sequence confusion.

   procedure Write_String_Table_Entry (Id : String_Id);
   --  Writes a string value with enclosing quotes to the current file using
   --  routines in package Output. Does not write an end of line character.
   --  This procedure is used for debugging purposes. 7-bit ASCII graphics
   --  (except for double quote and left brace) are output literally. The
   --  double quote appears as two successive double quotes. All other codes,
   --  are output as desribed for Write_Char_Code. For example, the string
   --  created by folding "A" & Ascii.LF & "Hello" will print as "A{0A}Hello".

end Stringt;
