------------------------------------------------------------------------------
--                                                                          --
--                         GNAT COMPILER COMPONENTS                         --
--                                                                          --
--                              S E M . C H 7                               --
--                                                                          --
--                                 B o d y                                  --
--                                                                          --
--                            $Revision: 1.112 $                            --
--                                                                          --
--           Copyright (c) 1992,1993,1994 NYU, All Rights Reserved          --
--                                                                          --
-- GNAT is free software;  you can  redistribute it  and/or modify it under --
-- terms of the  GNU General Public License as published  by the Free Soft- --
-- ware  Foundation;  either version 2,  or (at your option) any later ver- --
-- sion.  GNAT is distributed in the hope that it will be useful, but WITH- --
-- OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
-- or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License --
-- for  more details.  You should have  received  a copy of the GNU General --
-- Public License  distributed with GNAT;  see file COPYING.  If not, write --
-- to the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA. --
--                                                                          --
------------------------------------------------------------------------------

--  This package contains the routines to process package specifications and
--  bodies. The most important semantic aspects of package processing are the
--  handling of private and full declarations, and the construction of
--  dispatch tables for tagged types.

with Atree;    use Atree;
with Debug;    use Debug;
with Einfo;    use Einfo;
with Elists;   use Elists;
with Errout;   use Errout;
with Exp_Util; use Exp_Util;
with Itypes;   use Itypes;
with Lib;      use Lib;
with Namet;    use Namet;
with Nlists;   use Nlists;
with Opt;      use Opt;
with Output;   use Output;
with Sem;      use Sem;
with Sem_Ch3;  use Sem_Ch3;
with Sem_Ch8;  use Sem_Ch8;
with Sem_Ch12; use Sem_Ch12;
with Sem_Ch13; use Sem_Ch13;
with Sem_Util; use Sem_Util;
with Stand;    use Stand;
with Sinfo;    use Sinfo;
with Sinput;   use Sinput;

package body Sem_Ch7 is

   -----------------------------------
   -- Handling private declarations --
   -----------------------------------

   --  The principle that each entity has a single defining occurrence clashes
   --  with the presence of two separate definitions for private types: the
   --  first is the private type declaration, and the second is the full type
   --  declaration. It is important that all references to the type point to
   --  the same defining occurence, namely the first one. To enforce the two
   --  separate views of the entity, the corresponding information is swapped
   --  between the two declarations. Outside of the package, the defining
   --  occurence only contains the private declaration information, while in
   --  the private part and the body of the package the defining occurrence
   --  contains the full declaration. To simplify the swap, the defining
   --  occurrence that currently holds the private declaration points to the
   --  full declaration. During semantic processing the defining occurence also
   --  points to a list of private dependents, that is to say access types or
   --  composite types whose designated types or component types are subtypes
   --  or derived types of the private type in question. After the full decla-
   --  ration has been seen, the private dependents are updated to indicate
   --  that they have full definitions.

   ---------------------------------
   -- Analyze_Package_Declaration --
   ---------------------------------

   procedure Analyze_Package_Declaration (N : Node_Id) is
      Id : constant Node_Id := Defining_Unit_Simple_Name (Specification (N));

   begin
      Enter_Name (Id);
      Set_Ekind (Id, E_Package);
      Set_Etype (Id, Standard_Void_Type);
      New_Scope (Id);

      if Debug_Flag_C then
         Write_Str ("====  Compiling package spec ");
         Write_Name (Chars (Id));
         Write_Str (" from ");
         Write_Location (Sloc (N));
         Write_Eol;
      end if;

      Analyze (Specification (N));
      End_Package_Scope (Id);

      --  For a compilation unit, indicate whether it needs a body.

      if Nkind (Parent (N)) = N_Compilation_Unit then
         Set_Body_Required (Parent (N), Unit_Requires_Body (Id));
      end if;
   end Analyze_Package_Declaration;

   -----------------------------------
   -- Analyze_Package_Specification --
   -----------------------------------

   procedure Analyze_Package_Specification (N : Node_Id) is
      Id         : constant Entity_Id := Defining_Unit_Simple_Name (N);
      Vis_Decls  : constant List_Id := Visible_Declarations (N);
      Priv_Decls : constant List_Id := Private_Declarations (N);
      E          : Entity_Id;
      L          : Entity_Id;

   begin
      if Present (Vis_Decls) then
         Analyze_Declarations (Vis_Decls);
      end if;

      --  Verify that incomplete types have received full declarations.

      E := First_Entity (Id);

      while Present (E) loop
         if Ekind (E) = E_Incomplete_Type
           and then No (Full_Declaration (E))
         then
            Error_Msg_NE
              ("incomplete type& has no full declaration in visible part",
                     E, E);
         end if;

         E := Next_Entity (E);
      end loop;

      --  If package is a public child unit, then make the private
      --  declarations of the parent visible.

      if Present (Parent_Spec (Parent (N)))
         and then not Is_Private_Descendant (Id)
      then
         declare
            Par : Entity_Id := Scope (Id);
         begin
            while Par /= Standard_Standard loop
               Install_Private_Declarations (Par);
               Par := Scope (Par);
            end loop;
         end;
      end if;

      --  Analyze private part if present

      if Present (Priv_Decls) then
         L := Last_Entity (Id);
         Set_In_Private_Part (Id);
         Analyze_Declarations (Priv_Decls);

         --  The first private entity is the immediate follower of the last
         --  visible entity, if there was one.

         if Present (L) then
            Set_First_Private_Entity (Id, Next_Entity (L));
         else
            Set_First_Private_Entity (Id, First_Entity (Id));
         end if;

         Set_In_Private_Part (Id, False);
      end if;

      End_Use (Vis_Decls);
      End_Use (Priv_Decls);

      if Present (Priv_Decls) then
         Append_List (Freeze_All, Priv_Decls);
      elsif Present (Vis_Decls) then
         Append_List (Freeze_All,  Vis_Decls);
      end if;
   end Analyze_Package_Specification;

   -----------------------
   -- End_Package_Scope --
   -----------------------

   procedure End_Package_Scope (P : Entity_Id) is
      Id   : Entity_Id;
      Full : Entity_Id;

   begin
      Id := First_Entity (P);

      while Present (Id) and then Id /= First_Private_Entity (P) loop
         if Debug_Flag_E then
            Write_Str ("unlinking visible entity ");
            Write_Int (Int (Id));
            Write_Eol;
         end if;

         Set_Is_Use_Visible (Id, In_Use (P));
         Set_Is_Directly_Visible (Id, False);

         if (Ekind (Id) = E_Private_Type
               or else Ekind (Id) = E_Limited_Private_Type)
           and then No (Full_Declaration (Id))
           and then not Is_Generic_Type (Id)
           and then not Is_Derived_Type (Id)
         then
            Error_Msg_N ("missing full declaration for private type", Id);

         elsif Ekind (Id) = E_Constant
           and then No (Constant_Value (Id))
           and then No (Full_Declaration (Id))
         then
            Error_Msg_N ("missing full declaration for deferred constant", Id);
         end if;

         Id := Next_Entity (Id);
      end loop;

      --  Make private entities invisible and exchange full and private
      --  declarations for private types.

      while Present (Id) loop
         if Debug_Flag_E then
            Write_Str ("unlinking private entity ");
            Write_Int (Int (Id));
            Write_Eol;
         end if;

         Set_Is_Directly_Visible (Id, False);

         if Ekind (Id) in Private_Kind
           and then Present (Full_Declaration (Id))
         then
            Full := Full_Declaration (Id);

            --  The entry in the private part points to the full declaration,
            --  which is currently visible. Exchange them so only the private
            --  type declaration remains accessible, and link private and
            --  full declaration in the opposite direction. Before the actual
            --  exchange, we make sure that the Size is copied back into the
            --  private declaration. This is the one attribute of the full
            --  type that must be available for the private type too.

            --  Are there other attributes that should be treated the same,
            --  for example Eaddress ???

            --  For Taft amendment types,  there are no attributes to transfer
            --  because the second view is incomplete as well.

            if Ekind (Full) /= E_Incomplete_Type then
               Set_Has_Tasks (Id, Has_Tasks (Full));
               Set_Esize (Id, Esize (Full));
               Set_Is_Controlled (Id, Is_Controlled (Full));
               Set_Has_Controlled (Id, Has_Controlled (Full));
            else
               --  For now, indicate that these types are not supported.
               Unimplemented
                     (Id, "incomplete types completed in package body");
            end if;

            Set_Is_Use_Visible (Id, In_Use (P));
            Exchange_Declarations (Id);

            if Ekind (Id) = E_Incomplete_Type then
               --  the incomplete type will have a full declaration if the
               --  private type does.
               Set_Direct_Full_Declaration (Id, Full);
            end if;

         else
            Set_Is_Private (Id);
            Set_Is_Use_Visible (Id, False);
         end if;

         Id  := Next_Entity (Id);
      end loop;

      Pop_Scope;
   end End_Package_Scope;

   ---------------------------
   -- Exchange_Declarations --
   ---------------------------

   procedure Exchange_Declarations (Id : Entity_Id) is
      Full_Id : constant Entity_Id := Full_Declaration (Id);
      H1      : constant Entity_Id := Homonym (Id);
      Next1   : constant Entity_Id := Next_Entity (Id);
      H2      : Entity_Id;
      Next2   : Entity_Id;

   begin
      --  If missing full declaration for type, nothing to exchange

      if No (Full_Id) then
         return;
      end if;

      --  Otherwise complete the exchange, and preserve semantic links

      Next2 := Next_Entity (Full_Id);
      H2    := Homonym (Full_Id);

      --  Reset full declaration pointer to reflect the switched entities
      --  and readjust the next entity chains.

      Exchange_Entities (Id, Full_Id);
      Set_Direct_Full_Declaration (Full_Id, Id);
      Set_Next_Entity (Id, Next1);
      Set_Next_Entity (Full_Id, Next2);
      Set_Homonym (Id, H1);
      Set_Homonym (Full_Id, H2);
   end Exchange_Declarations;

   --------------------------------------
   -- Analyze_Private_Type_Declaration --
   --------------------------------------

   procedure Analyze_Private_Type_Declaration (N : Node_Id) is
      Id : Node_Id := Defining_Identifier (N);

   begin
      Enter_Name (Id);

      if Limited_Present (N) then
         Set_Ekind (Id, E_Limited_Private_Type);
         Set_Is_Limited_Type (Id);
      else
         Set_Ekind (Id, E_Private_Type);
      end if;

      Set_Is_Tagged_Type (Id, Tagged_Present (N));
      if Tagged_Present (N) then
         Unimplemented (N, "tagged private types");
      end if;

      Set_Etype (Id, Id);

      Set_Is_Private_Type (Id);
      Set_Is_Delayed (Id);

      if Ekind (Current_Scope) /= E_Package
        and then Ekind (Current_Scope) /= E_Generic_Package
      then
         Error_Msg_N ("invalid context for private declaration", N);
      end if;

      New_Scope (Id);

      if Present (Discriminant_Specifications (N)) then
         Process_Discriminants (N);
      end if;

      End_Scope;

      --  Initialize dispatch_table in tagged case

      if Tagged_Present (N) then
         Make_Class_Wide_Type (Id);
         Set_Primitive_Operations (Id, New_Elmt_List);
         Set_Is_Abstract (Id,  Abstract_Present (N));

      elsif Abstract_Present (N) then
         Error_Msg_N ("only a tagged type can be abstract", N);
      end if;
   end Analyze_Private_Type_Declaration;

   --------------------------
   -- Analyze_Package_Body --
   --------------------------

   procedure Analyze_Package_Body (N : Node_Id) is
      Body_Id          : Entity_Id := Defining_Unit_Simple_Name (N);
      Spec_Id          : Entity_Id;
      Last_Spec_Entity : Entity_Id;
      New_N            : Node_Id;
      Pack_Decl        : Node_Id;

   begin
      --  Find corresponding package specification, and establish the
      --  current scope. The visible defining entity for the package is the
      --  defining occurrence in the spec. On exit from the package body, all
      --  body declarations are attached to the defining entity for the body,
      --  but the later is never used for name resolution. In this fashion
      --  there is only one visible entity that denotes the package.

      if Debug_Flag_C then
         Write_Str ("====  Compiling package body ");
         Write_Name (Chars (Body_Id));
         Write_Str (" from ");
         Write_Location (Sloc (N));
         Write_Eol;
      end if;

      if Present (Corresponding_Spec (N)) then

         --  Body is body of package instantiation. Corresponding spec
         --  has already been set.

         Spec_Id := Corresponding_Spec (N);
         Pack_Decl := Get_Declaration_Node (Spec_Id);

      else
         Spec_Id := Current_Entity (Defining_Unit_Simple_Name (N));
         while Present (Spec_Id) loop
            exit when Scope (Spec_Id) = Current_Scope
              and then (Ekind (Spec_Id) = E_Package
                or else Ekind (Spec_Id) = E_Generic_Package);
            Spec_Id := Homonym (Spec_Id);
         end loop;

         if No (Spec_Id) then
            Error_Msg_N ("missing specification for package body", Body_Id);
            return;

         else
            Pack_Decl := Get_Declaration_Node (Spec_Id);

            if Present (Corresponding_Body (Pack_Decl)) then
               Error_Msg_N ("redefinition of package body", N);
               return;
            end if;
         end if;

         if Ekind (Spec_Id) = E_Package
           and then Current_Scope = Standard_Standard
           and then Parent (N) = Cunit (Main_Unit)
           and then not Unit_Requires_Body (Spec_Id)
         then
            if Ada_83 then
               Error_Msg_N
                   ("optional package body (not allowed in Ada9X)?", N);
            else
               Error_Msg_N
                   ("spec of this package does not allow a body", N);
            end if;
         end if;
      end if;

      --  If the package is generic,  disable expansion,  and perform
      --  analysis on copy. Unannotated body is used in  instantiations.

      if Ekind (Spec_Id) = E_Generic_Package then
         New_N := Copy_Generic_Node (N, Empty, Instantiating => False);
         Rewrite_Substitute_Tree (N, New_N);
         Expander_Mode_Save_And_Set (False);
      end if;

      Body_Id := Defining_Unit_Simple_Name (N);
      Set_Ekind (Body_Id, E_Package_Body);

      --  Defining name for the package body is not a visible entity: Only
      --  the defining name for the declaration is visible.

      Set_Ekind (Body_Id, Ekind (Spec_Id));
      Set_Etype (Body_Id, Standard_Void_Type);
      Set_Scope (Body_Id, Current_Scope);
      Set_Corresponding_Spec (N, Spec_Id);
      Set_Corresponding_Body (Pack_Decl, Body_Id);

      --  Indicate that we are currently compiling the body of the package.

      Set_Is_Package_Body (Spec_Id);
      Set_Has_Completion (Spec_Id);
      Last_Spec_Entity := Last_Entity (Spec_Id);

      New_Scope (Spec_Id);
      Install_Visible_Declarations (Spec_Id);
      Install_Private_Declarations (Spec_Id);
      Set_Use (Visible_Declarations (Specification (Pack_Decl)));
      Set_Use (Private_Declarations (Specification (Pack_Decl)));

      if Present (Declarations (N)) then
         Analyze_Declarations (Declarations (N));
      end if;

      if Present (Handled_Statement_Sequence (N)) then
         Analyze (Handled_Statement_Sequence (N));
      end if;

      End_Use (Declarations (N));
      End_Use (Private_Declarations (Specification (Pack_Decl)));
      End_Use (Visible_Declarations (Specification (Pack_Decl)));
      Check_Completion (Body_Id);

      if Ekind (Spec_Id) = E_Package then
         End_Package_Scope (Spec_Id);

      else
         --  For a generic package, collect global references and mark
         --  them on the original body so that they are not resolved
         --  again at the point of instantiation.

         Save_Global_References (Original_Node (N));
         Expander_Mode_Restore;

         if Nkind (Parent (N)) = N_Compilation_Unit then
            End_Package_Scope (Spec_Id);
         else

            --  Local declarations of local packages are not subsequently
            --  visible. Declarations of child units, on the other hand,
            --  should not be unlinked because they may be withed later.

            End_Scope;
         end if;

      end if;

      --  Chain the body declarations to the defining occurrence in the package

      if Present (Last_Spec_Entity) then
         Set_First_Entity (Body_Id, Next_Entity (Last_Spec_Entity));
         Set_Next_Entity (Last_Spec_Entity, Empty);

      else
         Set_First_Entity (Body_Id, First_Entity (Spec_Id));
         Set_First_Entity (Spec_Id, Empty);
      end if;

      --  All entities declared in body are not visible.

      declare
         E : Entity_Id := First_Entity (Body_Id);

      begin
         while Present (E) loop
            Set_Is_Private (E);
            E := Next_Entity (E);
         end loop;
      end;

      Set_Is_Package_Body (Spec_Id, False);
   end Analyze_Package_Body;

   ----------------------------------
   -- Install_Visible_Declarations --
   ----------------------------------

   procedure Install_Visible_Declarations (P : Entity_Id) is
      Id : Entity_Id;

   begin
      Id := First_Entity (P);

      while Present (Id) and then Id /= First_Private_Entity (P) loop
         Install_Package_Entity (Id);
         Id := Next_Entity (Id);
      end loop;
   end Install_Visible_Declarations;

   ----------------------------------
   -- Install_Private_Declarations --
   ----------------------------------

   procedure Install_Private_Declarations (P : Entity_Id) is
      Id : Entity_Id;

   begin
      --  First exchange declarations for private types, so that the
      --  full declaration is visible. If the full declaration is implicit,
      --  it was created to complete a subtype of an external private type,
      --  and nothing should be exchanged. If the full declaration is in
      --  another scope, then the entity is derived from a private type,
      --  and stays private as well. Finally,  if this is a Taft amendment
      --  type,  the incomplete declaration is irrelevant, and we want to
      --  link the eventual full declaration with the original private one
      --  so we also skip the exchange.

      Id := First_Entity (P);
      while Present (Id) and then Id /= First_Private_Entity (P) loop
         if Ekind (Id) in Private_Kind
           and then Present (Full_Declaration (Id))
           and then not Is_Itype (Full_Declaration (Id))
           and then Scope (Full_Declaration (Id)) = Scope (Id)
           and then Ekind (Full_Declaration (Id)) /= E_Incomplete_Type
         then
            Exchange_Declarations (Id);
            Set_Is_Directly_Visible (Id);
         end if;
         Id := Next_Entity (Id);
      end loop;

      --  Next make other declarations in the private part visible as well.

      Id := First_Private_Entity (P);
      while Present (Id) loop
         Install_Package_Entity (Id);
         Id := Next_Entity (Id);
      end loop;
   end Install_Private_Declarations;

   -----------------------------
   --  Install_Package_Entity --
   -----------------------------

   procedure Install_Package_Entity (Id : Entity_Id) is
   begin
      if not Is_Internal (Id) then
         if Debug_Flag_E then
            Write_Str ("Install: ");
            Write_Name (Chars (Id));
            Write_Eol;
         end if;
         Set_Is_Directly_Visible (Id);
      end if;
   end Install_Package_Entity;

   ------------------------
   -- Unit_Requires_Body --
   ------------------------

   function Unit_Requires_Body (P : Entity_Id) return Boolean is
      E : Entity_Id;

   begin
      --  Body required if library package with pragma Elaborate_Body

      if Ekind (P) = E_Package
        and then Scope (P) = Standard_Standard
        and then Elaborate_Body_Present
          (Cunit (Get_Sloc_Unit_Number (Sloc (P))))
      then
         return True;
      end if;

      --  Otherwise search entity chain for entity requiring completion

      E := First_Entity (P);
      while Present (E) loop
         if (Is_Overloadable (E)
               and then Ekind (E) /= E_Enumeration_Literal
               and then Ekind (E) /= E_Operator
               and then not Has_Completion (E))
           or else
             (Ekind (E) = E_Package and then E /= P
               and then not Has_Completion (E)
               and then Unit_Requires_Body (E))
           or else
             (Ekind (E) = E_Incomplete_Type
               and then No (Full_Declaration (E)))
           or else
             ((Ekind (E) = E_Task_Type or else Ekind (E) = E_Protected_Type)
               and then not Has_Completion (E))

           or else
             (Ekind (E) = E_Generic_Package and then E /= P
               and then not Has_Completion (E)
               and then Unit_Requires_Body (E))

           or else (Ekind (E) = E_Generic_Function
               and then not Has_Completion (E))

           or else (Ekind (E) = E_Generic_Procedure
               and then not Has_Completion (E))
         then
            return True;
         else
            null;
         end if;

         E := Next_Entity (E);
      end loop;

      return False;
   end Unit_Requires_Body;

   ----------------------
   -- Is_Fully_Visible --
   ----------------------

   --  The full declaration of a private type is visible in the private
   --  part of the package declaration, and in the package body, at which
   --  point the full declaration must have been given.

   function Is_Fully_Visible (Type_Id : Entity_Id) return Boolean is
      S : constant Entity_Id := Scope (Type_Id);

   begin
      if Is_Generic_Type (Type_Id) then
         return False;

      elsif In_Private_Part (S) then
         return Present (Full_Declaration (Type_Id));

      else
         return Is_Package_Body (S);
      end if;
   end Is_Fully_Visible;

end Sem_Ch7;
