------------------------------------------------------------------------------
--                                                                          --
--                 GNU ADA RUNTIME LIBRARY (GNARL) COMPONENTS               --
--                                                                          --
--      S Y S T E M . T A S K I N G . P R O T E C T E D _ O B J E C T S     --
--                                                                          --
--                                  B o d y                                 --
--                                                                          --
--                             $Revision: 1.9 $                             --
--                                                                          --
--           Copyright (c) 1991,1992,1993, FSU, All Rights Reserved         --
--                                                                          --
--  GNARL is free software; you can redistribute it and/or modify it  under --
--  terms  of  the  GNU  Library General Public License as published by the --
--  Free Software Foundation; either version 2,  or (at  your  option)  any --
--  later  version.   GNARL is distributed in the hope that it will be use- --
--  ful, but but WITHOUT ANY WARRANTY; without even the implied warranty of --
--  MERCHANTABILITY  or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Gen- --
--  eral Library Public License for more details.  You should have received --
--  a  copy of the GNU Library General Public License along with GNARL; see --
--  file COPYING. If not, write to the Free Software Foundation,  675  Mass --
--  Ave, Cambridge, MA 02139, USA.                                          --
--                                                                          --
------------------------------------------------------------------------------

with System.Compiler_Exceptions;
--  Used for, Compiler_Exceptions."="
--            Compiler_Exceptions.Raise_Exceptions

with System.Error_Reporting;
--  Used for, System.Error_Reporting.Assert

with System.Tasking.Abortion;
--  Used for, Abortion.Defer_Abortion,
--            Abortion.Undefer_Abortion
--            Abortion.Abort_To_Level

with System.Task_Primitives; use System.Task_Primitives;

with System.Tasking.Runtime_Types;
--  Used for, Runtime_Types.ATCB_Ptr,
--            Runtime_Types.ATCB_To_ID,
--            Runtime_Types.ID_To_ATCB

package body System.Tasking.Protected_Objects is

   procedure Assert (B : Boolean; M : String)
     renames Error_Reporting.Assert;

   function ID_To_ATCB (ID : Task_ID) return Runtime_Types.ATCB_Ptr
     renames Tasking.Runtime_Types.ID_To_ATCB;

   function ATCB_To_ID (Ptr : Runtime_Types.ATCB_Ptr) return Task_ID
     renames Runtime_Types.ATCB_To_ID;

   procedure Defer_Abortion
     renames Abortion.Defer_Abortion;

   procedure Undefer_Abortion
     renames Abortion.Undefer_Abortion;

--   function "=" (L, R : System.Address) return Boolean renames System."=";
--   why is this commented out ???

   function "=" (L, R : Runtime_Types.ATCB_Ptr) return Boolean
     renames Runtime_Types."=";

--  This is temporarily commented out. Gnat produces internal error ???
--  function "=" (L, R : Task_ID) return Boolean
--         renames "=";

   function "=" (L, R : Exception_ID) return Boolean
     renames Compiler_Exceptions."=";

   -----------------------------
   -- Raise_Pending_Exception --
   -----------------------------

   procedure Raise_Pending_Exception (Block : Communication_Block) is
      T  : Runtime_Types.ATCB_Ptr := ID_To_ATCB (Block.Self);
      Ex : Exception_ID := T.Exception_To_Raise;
   begin

      T.Exception_To_Raise := Null_Exception;
      Compiler_Exceptions.Raise_Exception (Ex);
   end Raise_Pending_Exception;

   ---------------------
   -- Check_Exception --
   ---------------------

   procedure Check_Exception is
      T  : Runtime_Types.ATCB_Ptr := ID_To_ATCB (Self);
      Ex : Exception_ID := T.Exception_To_Raise;

   begin
      T.Exception_To_Raise := Null_Exception;
      Compiler_Exceptions.Raise_Exception (Ex);
   end Check_Exception;

   ---------------------------
   -- Initialize_Protection --
   ---------------------------

   procedure Initialize_Protection
     (Object           : Protection_Access;
      Ceiling_Priority : Integer)
   is
      Init_Priority : Integer := Ceiling_Priority;

   begin
      if Init_Priority = Unspecified_Priority then
         Init_Priority := System.Default_Priority;
      end if;

      Initialize_Lock (Init_Priority, Object.L);
      Object.Pending_Call := null;
      Object.Call_In_Progress := null;

      for E in Object.Entry_Queues'range loop
         Object.Entry_Queues (E).Head := null;
         Object.Entry_Queues (E).Tail := null;
      end loop;
   end Initialize_Protection;

   -------------------------
   -- Finalize_Protection --
   -------------------------

   procedure Finalize_Protection (Object : Protection_Access) is
   begin
      --  Need to purge entry queues and pending entry call here. ???

      Finalize_Lock (Object.L);
   end Finalize_Protection;

   ----------
   -- Lock --
   ----------

   procedure Lock (Object : Protection_Access) is
   begin
      Write_Lock (Object.L);
   end Lock;

   --------------------
   -- Lock_Read_Only --
   --------------------

   procedure Lock_Read_Only (Object : Protection_Access) is
   begin
      Read_Lock (Object.L);
   end Lock_Read_Only;

   ------------
   -- Unlock --
   ------------

   procedure Unlock (Object : Protection_Access) is
   begin
      Unlock (Object.L);
   end Unlock;

   --------------------------
   -- Protected_Entry_Call --
   --------------------------

   procedure Protected_Entry_Call
     (Object    : Protection_Access;
      E         : Protected_Entry_Index;
      Uninterpreted_Data : System.Address;
      Mode      : Call_Modes;
      Block     : out Communication_Block)
   is
      Level : ATC_Level;
      Caller : Runtime_Types.ATCB_Ptr := ID_To_ATCB (Self);

   begin
      Block.Self := ATCB_To_ID (Caller);
      Caller.ATC_Nesting_Level := Caller.ATC_Nesting_Level + 1;
      Level := Caller.ATC_Nesting_Level;

      Object.Pending_Call := Caller.Entry_Calls (Level)'access;

      --   I don't think that we need the calling task's lock here.
      --   Only the calling task will get to access this record until
      --   it is queued, since the calling task
      --   will call Next_Entry_Call before releasing the PO lock,
      --   and since Next_Entry_Call always removes Pending_Call. ???

      Object.Pending_Call.Next := null;
      Object.Pending_Call.Call_Claimed := False;
      Object.Pending_Call.Mode := Mode;
      Object.Pending_Call.Abortable := True;
      Object.Pending_Call.Done := False;
      Object.Pending_Call.E := Entry_Index (E);
      Object.Pending_Call.Prio := Caller.Current_Priority;
      Object.Pending_Call.Uninterpreted_Data := Uninterpreted_Data;
      Object.Pending_Call.Called_PO := Protection_Access (Object);

      Object.Pending_Call.Called_Task := Null_Task;
      Object.Pending_Call.Exception_To_Raise := Null_Exception;

   end Protected_Entry_Call;

   --------------------------------------------
   -- Vulnerable_Cancel_Protected_Entry_Call --
   --------------------------------------------

   procedure Vulnerable_Cancel_Protected_Entry_Call
     (Caller         : Runtime_Types.ATCB_Ptr;
      Call           : Entry_Call_Link;
      PO             : Protection_Access;
      Call_Cancelled : out Boolean)
   is
      TAS_Result : Boolean;

   begin
      Test_And_Set (Call.Call_Claimed'Address, TAS_Result);

      if TAS_Result then
         Lock (PO);
         Dequeue (PO.Entry_Queues (Protected_Entry_Index (Call.E)), Call);

      else
         Write_Lock (Caller.L);

         while not Call.Done loop
            Cond_Wait (Caller.Rend_Cond, Caller.L);
         end loop;

         Unlock (Caller.L);
      end if;

      Caller.ATC_Nesting_Level := Caller.ATC_Nesting_Level - 1;

      Write_Lock (Caller.L);

      if Caller.Pending_ATC_Level = Caller.ATC_Nesting_Level then
         Caller.Pending_ATC_Level := ATC_Level_Infinity;
         Caller.Aborting := False;
      end if;

      Unlock (Caller.L);

      --   If we have reached the desired ATC nesting level, reset the
      --   requested level to effective infinity, to allow further calls.

      Caller.Exception_To_Raise := Call.Exception_To_Raise;
      Call_Cancelled := TAS_Result;

   end Vulnerable_Cancel_Protected_Entry_Call;

   -------------------------
   -- Wait_For_Completion --
   -------------------------

   --  Control flow procedure.
   --  Abortion must be deferred before calling this procedure.

   procedure Wait_For_Completion
     (Call_Cancelled : out Boolean;
      Block          : in out Communication_Block)
   is
      Caller     : Runtime_Types.ATCB_Ptr := ID_To_ATCB (Block.Self);
      Call       : Entry_Call_Link;
      PO         : Protection_Access;
      Cancelled  : Boolean;

   begin

      Assert (Caller.ATC_Nesting_Level > ATC_Level_Base'First,
        "Attempt to wait on a nonexistant protected entry call.");

      Call := Caller.Entry_Calls (Caller.ATC_Nesting_Level)'access;

      Assert (Call.Mode = Simple_Call,
        "Attempt to wait on a on a conditional or asynchronous call");

      PO := Call.Called_PO;

      Write_Lock (Caller.L);

      if Call.Abortable then
         Caller.Suspended_Abortably := True;

         while not Call.Done and then
            Caller.Pending_ATC_Level >= Caller.ATC_Nesting_Level loop
            Cond_Wait (Caller.Cond, Caller.L);
         end loop;

         Caller.Suspended_Abortably := False;

      else
         while not Call.Done loop
            Cond_Wait (Caller.Cond, Caller.L);
         end loop;
      end if;

      Unlock (Caller.L);

      Vulnerable_Cancel_Protected_Entry_Call (Caller, Call, PO, Cancelled);

   end Wait_For_Completion;

   ---------------------------------
   -- Cancel_Protected_Entry_Call --
   ---------------------------------

   procedure Cancel_Protected_Entry_Call
     (Call_Cancelled : out Boolean;
      Block          : in out Communication_Block)
   is
      Caller     : Runtime_Types.ATCB_Ptr := ID_To_ATCB (Block.Self);
      Call       : Entry_Call_Link;
      PO         : Protection_Access;
      TAS_Result : Boolean;
      Cancelled  : Boolean;

   begin
      Defer_Abortion;

      Assert (Caller.ATC_Nesting_Level > ATC_Level_Base'First,
        "Attempt to cancel a nonexistant task entry call.");

      Call := Caller.Entry_Calls (Caller.ATC_Nesting_Level)'access;

      Assert (Call.Mode = Asynchronous_Call,
        "Attempt to cancel a conditional or simple call");

      Assert (Call.Called_Task = Null_Task,
        "Attempt to use Cancel_Protected_Entry_Call on task entry call.");

      PO := Call.Called_PO;
      Vulnerable_Cancel_Protected_Entry_Call (Caller, Call, PO, Cancelled);
      Undefer_Abortion;

      Call_Cancelled := Cancelled;
   end Cancel_Protected_Entry_Call;

   --------------------------
   -- Wait_Until_Abortable --
   --------------------------

   procedure Wait_Until_Abortable (Block : in out Communication_Block) is
      Caller     : Runtime_Types.ATCB_Ptr := ID_To_ATCB (Block.Self);
      Call       : Entry_Call_Link;
      PO         : Protection_Access;
   begin
      Defer_Abortion;
      Assert (Caller.ATC_Nesting_Level > ATC_Level_Base'First,
        "Attempt to wait for a nonexistant call to be abortable.");
      Call := Caller.Entry_Calls (Caller.ATC_Nesting_Level)'access;
      Assert (Call.Mode = Asynchronous_Call,
        "Attempt to wait for a non-asynchronous call to be abortable");
      PO := Call.Called_PO;

      Write_Lock (Caller.L);
      while not Call.Abortable loop
         Cond_Wait (Caller.Cond, Caller.L);
      end loop;
      Unlock (Caller.L);

      Undefer_Abortion;
   end Wait_Until_Abortable;

   ---------------------
   -- Next_Entry_Call --
   ---------------------

   --   This procedure assumes that a task will have to enter the eggshell to
   --   cancel a call, so there is no need to check for cancellation here.
   --   This seems to obviate the need to lock the task at this point, since
   --   the task will be forced to wait before doing the cancellation, meaning
   --   that it will not take place.

   procedure Next_Entry_Call
     (Object    : Protection_Access;
      Barriers  : Barrier_Vector;
      Uninterpreted_Data : out System.Address;
      E         : out Protected_Entry_Index)
   is
      TAS_Result        : Boolean;
      Selected_Entry    : Protected_Entry_Index;
      Selected_Priority : System.Any_Priority;
      Next_Priority     : System.Any_Priority;

   begin
      Object.Call_In_Progress := null;
      if Object.Pending_Call /= null then

         Assert (Self = Object.Pending_Call.Self,
           "Pending call handled by a task that did not pend it.");

         --   Note that the main cost of the above assertion is likely
         --   to be the call to Self.  If this is not optimized away,
         --   nulling out Assert will not be of much value.

         if Barriers (Protected_Entry_Index (Object.Pending_Call.E)) then
            Test_And_Set
              (Object.Pending_Call.Call_Claimed'Address, TAS_Result);

            if TAS_Result then
               declare
                  Caller : Runtime_Types.ATCB_Ptr :=
                            ID_To_ATCB (Object.Pending_Call.Self);

                  --   Note that Object.Pending_Call.Self has to be Self;
                  --   otherwise, this would be illegal.
                  --   The task that pends the call must keep the object locked
                  --   until it calls Next_Entry_Call, and it will not be
                  --   pending on exit from Next_Entry_Call.

               begin
                  Object.Call_In_Progress := Object.Pending_Call;
               end;

            else
               Object.Pending_Call := null;
            end if;

         else
            Enqueue (
              Object.Entry_Queues (
              Protected_Entry_Index (Object.Pending_Call.E)),
              Object.Pending_Call);
         end if;

         Object.Pending_Call := null;
      end if;

      if Object.Call_In_Progress = null then

      --   The following loop attempts to claim a call on an open barrier.

         loop
            Selected_Entry := Null_Protected_Entry;
            Selected_Priority := System.Priority'First;

            --   The following loop finds the caller waiting on an open barrier
            --   with the highest base priority.  Active priority is not used,
            --   since it should be the same as base priority.  The only way
            --   that the active priority could be higher than the base
            --   priority is if the call had been made from within an eggshell.
            --   As an entry call is a potentially blocking operation, it is
            --   illegal to make one from within an eggshell.

            for B in Barriers'range loop
               if Barriers (B) and then
                  Head (Object.Entry_Queues (B)) /= null then
                  Next_Priority := Head (Object.Entry_Queues (B)).Prio;

                  if (Selected_Entry = Null_Protected_Entry or else
                     Next_Priority >= Selected_Priority) then
                     Selected_Entry := B;
                     Selected_Priority := Next_Priority;
                  end if;
               end if;
            end loop;

            exit when Selected_Entry = Null_Protected_Entry;

            Dequeue_Head
              (Object.Entry_Queues (Selected_Entry), Object.Call_In_Progress);

            if Object.Call_In_Progress.Abortable then
               Test_And_Set
                 (Object.Call_In_Progress.Call_Claimed'Address, TAS_Result);
               exit when TAS_Result;
               Object.Call_In_Progress := null;

            --   If call is not abortable, it has already been claimed for us

            else
               exit;
            end if;
         end loop;

      end if;

      if Object.Call_In_Progress /= null then
         E := Protected_Entry_Index (Object.Call_In_Progress.E);
         Uninterpreted_Data := Object.Call_In_Progress.Uninterpreted_Data;

      else
         E := Null_Protected_Entry;
      end if;

   end Next_Entry_Call;

   -------------------------
   -- Complete_Entry_Body --
   -------------------------

   procedure Complete_Entry_Body
     (Object           : Protection_Access;
      Pending_Serviced : out Boolean)
   is
   begin
      Exceptional_Complete_Entry_Body
        (Object, Pending_Serviced, Null_Exception);

   end Complete_Entry_Body;

   -------------------------------------
   -- Exceptional_Complete_Entry_Body --
   -------------------------------------

   procedure Exceptional_Complete_Entry_Body
     (Object           : Protection_Access;
      Pending_Serviced : out Boolean;
      Ex               : Exception_ID)
   is
      Caller : Runtime_Types.ATCB_Ptr :=
                    ID_To_ATCB (Object.Call_In_Progress.Self);

   begin
      Pending_Serviced := False;
      Object.Call_In_Progress.Exception_To_Raise := Ex;

      if Object.Pending_Call /= null then
         Assert (Object.Pending_Call = Object.Call_In_Progress,
           "Serviced a protected entry call when another was pending");

         Pending_Serviced := True;
         Caller.ATC_Nesting_Level := Caller.ATC_Nesting_Level - 1;
         Object.Pending_Call := null;
      end if;

      --   If we have completed a pending entry call, pop it and set the
      --   Pending_Serviced flag to indicate that it is complete.

      Write_Lock (Caller.L);
      Object.Call_In_Progress.Done := True;
      Unlock (Caller.L);

      if Object.Call_In_Progress.Mode = Asynchronous_Call then
         Abortion.Abort_To_Level
           (ATCB_To_ID (Caller), Object.Call_In_Progress.Level - 1);

      elsif Object.Call_In_Progress.Mode = Simple_Call then
         Cond_Signal (Caller.Cond);
      end if;

   end Exceptional_Complete_Entry_Body;

   -----------------------------
   -- Requeue_Protected_Entry --
   -----------------------------

   procedure Requeue_Protected_Entry
     (Object     : Protection_Access;
      New_Object : Protection_Access;
      E          : Protected_Entry_Index;
      With_Abort : Boolean)
   is
   begin
      Object.Call_In_Progress.Abortable := With_Abort;
      Object.Call_In_Progress.E := Entry_Index (E);

      if With_Abort then
         Object.Call_In_Progress.Call_Claimed := False;
      end if;

      if Object = New_Object then
         Enqueue (New_Object.Entry_Queues (E), Object.Call_In_Progress);
      else
         New_Object.Pending_Call := Object.Call_In_Progress;
      end if;

   end Requeue_Protected_Entry;

   -------------------------------------
   -- Requeue_Task_To_Protected_Entry --
   -------------------------------------

   procedure Requeue_Task_To_Protected_Entry
     (New_Object : Protection_Access;
      E          : Protected_Entry_Index;
      With_Abort : Boolean)
   is
      Old_Acceptor : Runtime_Types.ATCB_Ptr := ID_To_ATCB (Self);
      Entry_Call : Entry_Call_Link;

   begin
      Write_Lock (Old_Acceptor.L);
      Entry_Call := Old_Acceptor.Call;
      Old_Acceptor.Call := null;
      Unlock (Old_Acceptor.L);
      Entry_Call.Abortable := With_Abort;
      Entry_Call.E := Entry_Index (E);
      Entry_Call.Called_PO := Protection_Access (New_Object);

      if With_Abort then
         Entry_Call.Call_Claimed := False;
      end if;

      New_Object.Pending_Call := Entry_Call;
   end Requeue_Task_To_Protected_Entry;

   ---------------------
   -- Protected_Count --
   ---------------------

   function Protected_Count
     (Object : Protection;
      E      : Protected_Entry_Index)
      return   Natural
   is
   begin
      return Count_Waiting (Object.Entry_Queues (E));
   end Protected_Count;

   -----------------------------
   -- Broadcast_Program_Error --
   -----------------------------

   procedure Broadcast_Program_Error
     (Object        : Protection_Access) is
      Entry_Call    : Entry_Call_Link;
      Current_Task  : Runtime_Types.ATCB_Ptr;
      Raise_In_Self : Boolean := True;

   begin
      for E in Object.Entry_Queues'range loop
         Dequeue (Object.Entry_Queues (E), Entry_Call);

         while Entry_Call /= null loop
            Current_Task := ID_To_ATCB (Entry_Call.Self);
            Entry_Call.Exception_To_Raise  :=
            Program_Error_ID;
            Write_Lock (Current_Task.L);
            Entry_Call.Done := True;
            Unlock (Current_Task.L);

            case Entry_Call.Mode is

               when Simple_Call =>
                  Abortion.Abort_To_Level
                    (ATCB_To_ID (Current_Task), Entry_Call.Level - 1);

               when Conditional_Call =>
                  Assert (False, "Conditional call found on entry queue.");

               when Asynchronous_Call =>
                  Abortion.Abort_To_Level
                    (ATCB_To_ID (Current_Task), Entry_Call.Level - 1);

            end case;

            Dequeue (Object.Entry_Queues (E), Entry_Call);
         end loop;
      end loop;
   end Broadcast_Program_Error;

end System.Tasking.Protected_Objects;
