------------------------------------------------------------------------------
--                                                                          --
--                         GNAT COMPILER COMPONENTS                         --
--                                                                          --
--                                  R A T                                   --
--                                                                          --
--                                 B o d y                                  --
--                                                                          --
--                            $Revision: 1.20 $                             --
--                                                                          --
--           Copyright (c) 1992,1993,1994 NYU, All Rights Reserved          --
--                                                                          --
-- GNAT is free software;  you can  redistribute it  and/or modify it under --
-- terms of the  GNU General Public License as published  by the Free Soft- --
-- ware  Foundation;  either version 2,  or (at your option) any later ver- --
-- sion.  GNAT is distributed in the hope that it will be useful, but WITH- --
-- OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
-- or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License --
-- for  more details.  You should have  received  a copy of the GNU General --
-- Public License  distributed with GNAT;  see file COPYING.  If not, write --
-- to the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA. --
--                                                                          --
------------------------------------------------------------------------------

with Atree; use Atree;
with Einfo; use Einfo;
with Nmake; use Nmake;
with Sinfo; use Sinfo;
with Uintp; use Uintp;

package body Rat is

   --  Rational numbers are represented as pairs of Universal Integers
   --  (Uints) with the denominator always positive, so the sign is
   --  carried in the numerator. These values are not necessarily
   --  normalized (i.e. they may not be expressed in lowest terms).

   type Rational is record
      Num : Uint;
      Den : Uint;
   end record;

   Loc : Source_Ptr;
   --  Source pointer used to construct result node

   R_Half : Node_Id := Empty;
   R_1    : Node_Id := Empty;
   R_2    : Node_Id := Empty;
   R_10   : Node_Id := Empty;
   --  These are the constant Nodes which will be returned by the functions
   --  Rlit_Half, Rlit_1, Rlit_2 and Rlit_10. They are initialized by the
   --  procedure Initialize.

   -----------------------
   -- Local Subprograms --
   -----------------------

   function Normalize (Rat : Rational) return Rational;
   --  Normalizes the rational by reducing it to lowest terms

   function Rational_To_Real
     (Loc  : Source_Ptr;
      R    : Rational)
      return Real_Literal;
   --  Build real literal from rational value, normalizes the value. The
   --  resulting literal uses the given source location (taken from one of
   --  the input arguments)

   function Real_To_Rational (Real : Real_Literal) return Rational;
   --  Gets rational value from real, normalizing the result

   function Val (Real : Real_Literal) return Real_Literal;
   --  Returns new real literal which is a copy of the value of Real

   function Rat_Sign (Real : Real_Literal) return Uint;
   --  Returns -1 if real negative and 1 otherwise.

   ----------------
   -- Initialize --
   ----------------

   procedure Initialize is
   begin
      R_Half := Make_Real_Literal (No_Location, Uint_1, Uint_2);
      R_1    := Make_Real_Literal (No_Location, Uint_1, Uint_1);
      R_2    := Make_Real_Literal (No_Location, Uint_2, Uint_1);
      R_10   := Make_Real_Literal (No_Location, Uint_10, Uint_1);
   end Initialize;

   ------------------
   -- Int_From_Rat --
   ------------------

   function Int_From_Rat (Real : Real_Literal) return Integer_Literal is
      Rat : Rational := Real_To_Rational (Real);

   begin
      --  Round value away from zero, so truncate gives right result

      if UI_Is_Negative (Rat.Num) then
         Rat.Num := UI_Difference (Rat.Num, UI_Quotient (Rat.Den, Uint_2));
      else
         Rat.Num := UI_Sum (Rat.Num, UI_Quotient (Rat.Den, Uint_2));
      end if;

      return
        Make_Integer_Literal (Sloc (Real), UI_Quotient (Rat.Num, Rat.Den));
   end Int_From_Rat;

   -----------------
   -- Normal_Form --
   -----------------

   procedure Normal_Form
     (Real         : Real_Literal;
      Normal_Value : out Real_Literal;
      Scale_Factor : out Uint)
   is
   begin
      Scale_Factor := Uint_0;
      Normal_Value := Rat_Abs (Real);

      while Rat_Lt (Normal_Value, Rlit_Half) loop
         Normal_Value := Rat_Product (Normal_Value, Rlit_2);
         Scale_Factor := UI_Difference (Scale_Factor, Uint_1);
      end loop;

      while Rat_Ge (Normal_Value, Rlit_1) loop
         Normal_Value := Rat_Quotient (Normal_Value, Rlit_2);
         Scale_Factor := UI_Sum (Scale_Factor, Uint_1);
      end loop;

   end Normal_Form;

   ---------------
   -- Normalize --
   ---------------

   function Normalize (Rat : Rational) return Rational is
      J   : Uint;
      K   : Uint;
      Tmp : Uint;

   begin
      --  Start by setting J to the greatest of the absolute values of the
      --  two numbers, and K to the lesser of the two absolute values. The
      --  gcd of Num and Den is the gcd of J and K.

      J := UI_Abs (Rat.Num);
      K := UI_Abs (Rat.Den);

      if UI_Gt (K, J) then
         Tmp := J;
         J := K;
         K := Tmp;
      end if;

      --  Now apply Euclid's algorithm to find the gcd, which is left in J.

      while not UI_Is_Zero (K) loop
         Tmp := UI_Mod (J, K);
         J := K;
         K := Tmp;
      end loop;

      --  Divide numerator and denominator by gcd and return result

      return (Num => UI_Quotient (Rat.Num, J),
              Den => UI_Quotient (Rat.Den, J));
   end Normalize;

   -------------
   -- Rat_Abs --
   -------------

   function Rat_Abs (Right : Real_Literal) return Real_Literal is
      Rrat : constant Rational := Real_To_Rational (Right);

   begin
      return Rational_To_Real (Sloc (Right), (UI_Abs (Rrat.Num), Rrat.Den));
   end Rat_Abs;

   --------------------
   -- Rat_Difference --
   --------------------

   function Rat_Difference (Left, Right : Real_Literal) return Real_Literal is
      Lrat : constant Rational := Real_To_Rational (Left);
      Rrat : constant Rational := Real_To_Rational (Right);

   begin
      return Rational_To_Real (Sloc (Left), (
        UI_Difference (UI_Product (Lrat.Num, Rrat.Den),
                      UI_Product (Rrat.Num, Lrat.Den)),
        UI_Product (Rrat.Den, Lrat.Den)));
   end Rat_Difference;

   ------------
   -- Rat_Eq --
   ------------

   function Rat_Eq (Left, Right : Real_Literal) return Boolean is
      Lrat : constant Rational := Real_To_Rational (Left);
      Rrat : constant Rational := Real_To_Rational (Right);

   begin
      return     UI_Eq (Rrat.Num, Lrat.Num)
        and then UI_Eq (Rrat.Den, Lrat.Den);
   end Rat_Eq;

   ----------------------
   -- Rat_Exponentiate --
   ----------------------

   function Rat_Exponentiate
     (Left  : Real_Literal;
      Right : Uint)
      return  Real_Literal
   is
      Lrat : constant Rational := Real_To_Rational (Left);
      X    : constant Uint := UI_Abs (Right);
      N, D : Uint;

   begin
      --  Raise numerator and denominator to absolute value of exponent

      N := UI_Exponentiate (Lrat.Num, X);
      D := UI_Exponentiate (Lrat.Den, X);

      --  All done if exponent is non-negative. Note that in the case of
      --  a zero exponent, N and D are set to 1, which gives the correct
      --  result of 1.0 (i.e. 1/1).

      if not UI_Is_Negative (Right) then
         return Rational_To_Real (Sloc (Left), (N, D));

      --  Return reciprocal if negative (note that we will blow up on an
      --  assert error in Rational_To_Real if the base is zero in this case,
      --  which is fine)

      else
         return Rational_To_Real (Sloc (Left), (D, N));
      end if;
   end Rat_Exponentiate;

   ------------------
   -- Rat_From_Int --
   ------------------

   function Rat_From_Int (Int : Integer_Literal) return Real_Literal is
   begin
      Loc := Sloc (Int);
      return
        Rational_To_Real (Sloc (Int), (Num => Intval (Int), Den => Uint_1));
   end Rat_From_Int;

   -------------------
   -- Rat_From_Ints --
   -------------------

   function Rat_From_Ints (Num, Den : Integer_Literal) return Real_Literal is
   begin
      pragma Assert (Intval (Den) /= Uint_0);

      return
        Rational_To_Real (Sloc (Num),
          Normalize ((Num => Intval (Num), Den => Intval (Den))));

   end Rat_From_Ints;

   -------------------
   -- Rat_From_Uint --
   -------------------

   function Rat_From_Uint (Int_Value : Uint) return Real_Literal is
   begin
      return Make_Real_Literal (No_Location,
                Numerator => Int_Value,
                Denominator => Uint_1);
   end Rat_From_Uint;

   ------------
   -- Rat_Ge --
   ------------

   function Rat_Ge (Left, Right : Real_Literal) return Boolean is
      Lrat : constant Rational := Real_To_Rational (Left);
      Rrat : constant Rational := Real_To_Rational (Right);

   begin
      return UI_Ge (UI_Product (Lrat.Num, Rrat.Den),
                    UI_Product (Rrat.Num, Lrat.Den));

   end Rat_Ge;

   ------------
   -- Rat_Gt --
   ------------

   function Rat_Gt (Left, Right : Real_Literal) return Boolean is
      Lrat : constant Rational := Real_To_Rational (Left);
      Rrat : constant Rational := Real_To_Rational (Right);

   begin
      return UI_Gt (UI_Product (Lrat.Num, Rrat.Den),
                    UI_Product (Rrat.Num, Lrat.Den));
   end Rat_Gt;

   -----------------
   -- Rat_Is_Zero --
   -----------------

   function Rat_Is_Zero (Arg : Real_Literal) return Boolean is
      Rat : constant Rational := Real_To_Rational (Arg);

   begin
      return UI_Is_Zero (Rat.Den);
   end Rat_Is_Zero;

   ------------
   -- Rat_Le --
   ------------

   function Rat_Le (Left, Right : Real_Literal) return Boolean is
      Lrat : constant Rational := Real_To_Rational (Left);
      Rrat : constant Rational := Real_To_Rational (Right);

   begin
      return UI_Le (UI_Product (Lrat.Num, Rrat.Den),
                    UI_Product (Rrat.Num, Lrat.Den));
   end Rat_Le;

   ------------
   -- Rat_Lt --
   ------------

   function Rat_Lt (Left, Right : Real_Literal) return Boolean is
      Lrat : constant Rational := Real_To_Rational (Left);
      Rrat : constant Rational := Real_To_Rational (Right);

   begin
      return UI_Lt (UI_Product (Lrat.Num, Rrat.Den),
                    UI_Product (Rrat.Num, Lrat.Den));
   end Rat_Lt;

   -------------
   -- Rat_Max --
   -------------

   function Rat_Max (Left, Right : Real_Literal) return Real_Literal is
   begin
      if Rat_Ge (Left, Right) then
         return Val (Left);
      else
         return Val (Right);
      end if;
   end Rat_Max;

   -------------
   -- Rat_Min --
   -------------

   function Rat_Min (Left, Right : Real_Literal) return Real_Literal is
   begin
      if Rat_Le (Left, Right) then
         return Val (Left);
      else
         return Val (Right);
      end if;
   end Rat_Min;

   ------------
   -- Rat_Ne --
   ------------

   function Rat_Ne (Left, Right : Real_Literal) return Boolean is
      Lrat : constant Rational := Real_To_Rational (Left);
      Rrat : constant Rational := Real_To_Rational (Right);

   begin
      return Rrat.Num /= Lrat.Num or else Rrat.Den /= Lrat.Den;
   end Rat_Ne;

   ----------------
   -- Rat_Negate --
   ----------------

   function Rat_Negate (Right : Real_Literal) return Real_Literal is
      Rrat : constant Rational := Real_To_Rational (Right);

   begin
      return
        Rational_To_Real (Sloc (Right), (UI_Negate (Rrat.Num), Rrat.Den));
   end Rat_Negate;

   -----------------
   -- Rat_Product --
   -----------------

   function Rat_Product (Left, Right : Real_Literal) return Real_Literal is
      Lrat : constant Rational := Real_To_Rational (Left);
      Rrat : constant Rational := Real_To_Rational (Right);

   begin
      return Rational_To_Real (Sloc (Left), (
        UI_Product (Lrat.Num, Rrat.Num),
        UI_Product (Lrat.Den, Rrat.Den)));
   end Rat_Product;

   ------------------
   -- Rat_Quotient --
   ------------------

   function Rat_Quotient (Left, Right : Real_Literal) return Real_Literal is
      Lrat : constant Rational := Real_To_Rational (Left);
      Rrat : constant Rational := Real_To_Rational (Right);

   begin
      pragma Assert (Rrat.Num /= 0);

      return Rational_To_Real (Sloc (Left), (
        UI_Product (Lrat.Num, Rrat.Den),
        UI_Product (Lrat.Den, Rrat.Num)));
   end Rat_Quotient;

   ---------------
   -- Rat_Round --
   ---------------

   function Rat_Round (Real : Real_Literal; N : Uint) return Real_Literal is
      Norm_R : Real_Literal;
      Fact_R : Uint;

   begin
      Normal_Form (Real, Norm_R, Fact_R);
      return Rat_Product
               (Rat_Exponentiate
                  (Rlit_2,
                   UI_Difference (Fact_R, N)),
                Rat_From_Uint
                  (Rat_Trunc
                     (Rat_Sum
                        (Rat_Product (Rat_Exponentiate (Rlit_2, N), Norm_R),
                         Rat_Product
                           (Rat_From_Uint (Rat_Sign (Real)),
                            Rlit_Half)))));
   end Rat_Round;

   -------------
   -- Rat_Sum --
   -------------

   function Rat_Sum (Left, Right : Real_Literal) return Real_Literal is
      Lrat : constant Rational := Real_To_Rational (Left);
      Rrat : constant Rational := Real_To_Rational (Right);

   begin
      return Rational_To_Real (Sloc (Left), (
        UI_Sum (UI_Product (Lrat.Num, Rrat.Den),
                UI_Product (Rrat.Num, Lrat.Den)),
        UI_Product (Rrat.Den, Lrat.Den)));
   end Rat_Sum;

   ---------------
   -- Rat_Trunc --
   ---------------

   function Rat_Trunc (Real : Real_Literal) return Uint is
      Rat : constant Rational := Real_To_Rational (Real);

   begin
      return UI_Quotient (Rat.Num, Rat.Den);
   end Rat_Trunc;

   ----------------------
   -- Rational_To_Real --
   ----------------------

   function Rational_To_Real
     (Loc  : Source_Ptr;
      R    : Rational)
      return Real_Literal
   is
      Nrat : constant Rational := Normalize (R);

   begin
      return
        Make_Real_Literal (Loc,
          Numerator   => Nrat.Num,
          Denominator => Nrat.Den,
          Decimal     => False);

   end Rational_To_Real;

   ----------------------
   -- Real_To_Rational --
   ----------------------

   function Real_To_Rational (Real : Real_Literal) return Rational is
      Rlit : Node_Id;
      Ent  : Entity_Id;
      Decl : Node_Id;

   begin
      Rlit := Real;

      while Nkind (Rlit) in N_Entity_Name loop
         Ent := Entity (Rlit);
         pragma Assert (Is_Real_Type (Etype (Rlit)));
         Decl := Declaration_Node (Ent);
         pragma Assert (Nkind (Decl) = N_Object_Declaration);
         Rlit := Expression (Decl);
      end loop;

      pragma Assert (Nkind (Rlit) = N_Real_Literal);

      declare
         Num : constant Uint := Numerator (Rlit);
         Den : constant Uint := Denominator (Rlit);

      begin
         if Num = Uint_0 then
            return (Num => Uint_0, Den => Uint_1);

         elsif Decimal (Rlit) then
            if UI_Is_Positive (Den) then
               return Normalize ((Num, UI_Exponentiate (Uint_10, Den)));

            elsif Den = Uint_0 then
               return (Num => Num, Den => Uint_1);

            else -- UI_Is_Negative (Den)
               return
                 (UI_Product
                   (Num, UI_Exponentiate (Uint_10, UI_Negate (Den))), Uint_1);
            end if;

         elsif UI_Is_Negative (Den) then
            return Normalize ((UI_Negate (Num), UI_Negate (Den)));

         else
            return Normalize ((Num, Den));

         end if;
      end;
   end Real_To_Rational;

   --------------
   -- Rat_Sign --
   --------------

   function Rat_Sign (Real : Real_Literal) return Uint is
      Rat : constant Rational := Real_To_Rational (Real);

   begin

      if UI_Is_Negative (Rat.Num) then
         return Uint_Minus_1;
      else
         return Uint_1;
      end if;

   end Rat_Sign;

   ---------------
   -- Rlit_Half --
   ---------------

   function Rlit_Half return Real_Literal is
   begin
      return R_Half;
   end Rlit_Half;

   ------------
   -- Rlit_1 --
   ------------

   function Rlit_1 return Real_Literal is
   begin
      return R_1;
   end Rlit_1;

   ------------
   -- Rlit_2 --
   ------------

   function Rlit_2 return Real_Literal is
   begin
      return R_2;
   end Rlit_2;

   -------------
   -- Rlit_10 --
   -------------

   function Rlit_10 return Real_Literal is
   begin
      return R_10;
   end Rlit_10;

   ---------
   -- Val --
   ---------

   function Val (Real : Real_Literal) return Real_Literal is
   begin
      return Rational_To_Real (Sloc (Real), Real_To_Rational (Real));
   end Val;

end Rat;
