------------------------------------------------------------------------------
--                                                                          --
--                         GNAT COMPILER COMPONENTS                         --
--                                                                          --
--                                O S I N T                                 --
--                                                                          --
--                                 B o d y                                  --
--                                                                          --
--                            $Revision: 1.61 $                             --
--                                                                          --
--           Copyright (c) 1992,1993,1994 NYU, All Rights Reserved          --
--                                                                          --
-- GNAT is free software;  you can  redistribute it  and/or modify it under --
-- terms of the  GNU General Public License as published  by the Free Soft- --
-- ware  Foundation;  either version 2,  or (at your option) any later ver- --
-- sion.  GNAT is distributed in the hope that it will be useful, but WITH- --
-- OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
-- or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License --
-- for  more details.  You should have  received  a copy of the GNU General --
-- Public License  distributed with GNAT;  see file COPYING.  If not, write --
-- to the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA. --
--                                                                          --
------------------------------------------------------------------------------

with Namet;    use Namet;
with Output;   use Output;
with Switch;   use Switch;
with Unixlib;  use Unixlib;
with Opt;      use Opt;
with Sdefault; use Sdefault;

with System.Environment; use System.Environment;

package body Osint is

   File_Names : array (Int range 1 .. Int (Arg_Count)) of Arg_Ptr;
   --  As arguments are scanned in OS_Interface_Init, filenames are stored
   --  in this array. The string does not contain a terminating NUL.

   Number_File_Names : Int := 0;
   --  The total number of filenames found on command line and placed in
   --  File_Names

   Current_File_Name_Index : Int := 0;
   --  The index in File_Names of the last file opened by Next_Main_Source
   --  or Next_Main_Lib_File. The value 0 indicates that no files have been
   --  opened yet.

   In_Binder   : Boolean;
   In_Compiler : Boolean;
   In_Make     : Boolean;
   --  Exactly one of these flags is set True to indicate which program
   --  is bound and executing with Osint, which is used by all these programs.

   Source_Time_Stamp : Time_Stamp_Type;
   --  Time stamp for current source file

   Output_FD : Unix_FD;
   --  The file descriptor for the current library info or binder output

   Next_Source_Low_Bound : Source_Ptr := First_Source_Ptr;
   --  Value for low bound of next text buffer

   EOL : constant Character := Ascii.LF;
   --  End of line character

   Output_Filename : Arg_Ptr := null;
   --  The name after the -o option

   Save_Main_File_Name : File_Name_Type;
   --  Used to save a simple file name between calls to Next_Main_Source and
   --  Read_Source_File.  If the file name argument to Read_Source_File is
   --  No_File, that indicates that the file whose name was returned by the
   --  last call to Next_Main_Source (and stored here) is to be read.

   Save_Full_File_Name : Name_Id;
   --  Set to full name of source or library information file read by the
   --  most recent call to Read_Source_File (result returned by
   --  Full_Source_Name) or Read_Library_Info (result returned by
   --  Full_Library_Info_Name)

   function Normalize_Directory_Name (Directory : String) return String;
   --  Verify and normalize a directory name.  If directroy name is invalid,
   --  this will return an empty string.  Otherwise it will insure a trailing
   --  slash and make other normalizations.

   Primary_Directory : Natural := 0;
   --  This is index in the table created below for the first directory to
   --  search in for source or library information files.
   --  For the compiler (looking for sources) it is the directory containing
   --  the main unit.  For the binder (looking for library information files)
   --  it is the current working directory.

   package Search_Directories is new Table (
     Table_Component_Type => Arg_Ptr,
     Table_Index_Type     => Natural,
     Table_Low_Bound      => Primary_Directory,
     Table_Initial        => 12,
     Table_Increment      => 100,
     Table_Name           => "Osint.Search_Directories");
   --  Table of names of directories in which to search for source (Compiler)
   --  or ali (Binder) files.  This table is filled in the order in which
   --  the directories are to be searched, and then used in that order.

   function Locate_File (
     Dir_Index : Natural;
     File_Name : String)
     return Boolean;
   --  See if the file named by File_Name exists in the directory in
   --  Search_Directories indexed by Dir_Index.  Return True and
   --  (side effect) assign Save_Full_File_Name if found; otherwise
   --  return False.

   Directory_Separator : Character := '/';
   Path_Separator      : Character := ':';
   ALI_Suffix          : Arg_Ptr   := new String'("ali");
   Object_Suffix       : Arg_Ptr   := new String'("o");
   --  The character that is used to separate parts of a pathname; the
   --  character to separate paths in an environment variable value; the
   --  filename suffixes for ALI and object files.  Set here to Unix
   --  convention, reset in Initialize for other conventions.

   ----------------
   -- Initialize --
   ----------------

   procedure Initialize (P : Program_Type) is
      Already_Seen      : Boolean := False;
      Search_Path_Value : Arg_Ptr;
      Next_Arg          : Positive;

      function Is_MS_DOS return Int;
      pragma Import (C, Is_MS_DOS, "Is_MS_DOS");

   begin
      MS_DOS := Is_MS_DOS /= 0;

      Program := P;

      case Program is
         when Binder   => In_Binder   := True;
         when Compiler => In_Compiler := True;
         when Make     => In_Make     := True;
      end case;

      if MS_DOS then
         Directory_Separator := '\';
         Path_Separator      := ';';
         ALI_Suffix          := new String'("ali");
         Object_Suffix       := new String'("obj");
      end if;

      Search_Directories.Set_Last (Primary_Directory);

      --  Reserve the first slot in the search paths table.  For the compiler
      --  this is the directory of the main source file and is filled in by
      --  each call to Next_Main_Source.  For the binder, this is always empty
      --  so the current working directory is searched first.

      if In_Binder then
         Search_Directories.Table (Primary_Directory) := new String'("");
      end if;

      --  Loop through command line arguments, storing them for later access

      Next_Arg := 1;
      loop
         exit when Next_Arg >= Arg_Count;

         declare
            Next_Argv : constant Arg_Ptr := Arg_Value (Next_Arg);
            Search    : constant Arg_Ptr := new String'("-search");
            --  ??? Kludge - see below

         begin
            if Next_Argv'Length /= 0
               and then (Next_Argv (1) = '-'
                         or (MS_DOS and Next_Argv (1) = '/'))
            then

               --  Processing of search path arguments is handled here.
               --  All other options are single character and are handled
               --  by Scan_Switches.

               --  ???if Next_Argv.all = "-search" or "/search" or "/SEARCH"
               --     then
               if Next_Argv.all = Search.all then

                  --  Got another directory to search.  The directory name
                  --  is the following argument.  Make sure it is present and
                  --  add name to table.

                  Next_Arg := Next_Arg + 1;

                  if Next_Arg = Arg_Count then
                     Write_Str ("Missing argument for -search");
                     Write_Eol;
                     Exit_Program (E_Fatal);
                  end if;

                  Search_Directories.Increment_Last;
                  Search_Directories.Table (Search_Directories.Last)
                    := new String'(Arg_Value (Next_Arg).all);

               else
                  Scan_Switches (Next_Argv.all);
               end if;

            --  Not a switch, so must be a filename (if non-empty)

            elsif Next_Argv'Length /= 0 then -- Ignore empty arguments

               if Output_Filename_Present and not Already_Seen then
                  Already_Seen := True;
                  Output_Filename := new String'(Next_Argv.all);

               else
                  Number_File_Names := Number_File_Names + 1;
                  File_Names (Number_File_Names) := new String'(Next_Argv.all);
               end if;
            end if;
         end;

         Next_Arg := Next_Arg + 1;
      end loop;

      --  After the locations specified on the command line, the next places
      --  to look for files are the directories specified by the appropriate
      --  environment variable.  Get this value, extract the directory names
      --  and store in the table.

      if In_Compiler then
         Search_Path_Value := new String'(Unix_Getenv ("ADA_INCLUDE_PATH"));
      elsif In_Binder then
         Search_Path_Value := new String'(Unix_Getenv ("ADA_OBJECTS_PATH"));
      end if;

      if Search_Path_Value'Length > 0 then
         declare
            Lower_Bound : Positive := 1;
            Upper_Bound : Positive;

         begin
            loop
               while Lower_Bound <= Search_Path_Value'Last
                  and then Search_Path_Value.all (Lower_Bound)
                                                          = Path_Separator
               loop
                  Lower_Bound := Lower_Bound + 1;
               end loop;

               exit when Lower_Bound > Search_Path_Value'Last;

               Upper_Bound := Lower_Bound;
               while Upper_Bound <= Search_Path_Value'Last
                  and then Search_Path_Value.all (Upper_Bound)
                                                          /= Path_Separator
               loop
                  Upper_Bound := Upper_Bound + 1;
               end loop;

               Search_Directories.Increment_Last;
               Search_Directories.Table (Search_Directories.Last)
                 := new String'(Normalize_Directory_Name
                                  (Search_Path_Value.all
                                    (Lower_Bound .. Upper_Bound - 1)));

               Lower_Bound := Upper_Bound + 1;
            end loop;
         end;
      end if;

      --  The last place to look are the defaults.  OS/2 doesn't use
      --  them.

      if not MS_DOS then
         Search_Directories.Increment_Last;

         if In_Compiler then
            Search_Directories.Table (Search_Directories.Last)
              := new String'(Include_Dir_Default_Name);
         elsif In_Binder then
            Search_Directories.Table (Search_Directories.Last)
              := new String'(Object_Dir_Default_Name);
         end if;
      end if;

   end Initialize;

   ------------------------------
   -- Normalize_Directory_Name --
   ------------------------------

   function Normalize_Directory_Name (Directory : String) return String is
   begin
      --  For now this just insures that the string is terminated with
      --  the directory separator character.  Add more later?

      if Directory (Directory'Last) = Directory_Separator then
         return Directory;
      else
         --  ??? should be: return Directory & Directory_Separator;
         declare
            Return_String : String (1 .. Directory'Length + 1);
         begin
            Return_String (1 .. Directory'Length) := Directory;
            Return_String (Directory'Length + 1) := Directory_Separator;
            return Return_String;
         end;
      end if;
   end Normalize_Directory_Name;

   -----------------
   -- Locate_File --
   -----------------

   function Locate_File (
     Dir_Index : Natural;
     File_Name : String)
     return Boolean
   is
      --  ??? Full_Name : constant String
      --    := Search_Directories.Table (Dir_Index).all & File_Name;
      Dir_Name_Length : Natural := Search_Directories.Table (Dir_Index)'Length;
      Full_Name : String (1 .. Dir_Name_Length + File_Name'Length);

   begin
      --  ??? Change to above declaration when dynamic concatenation works
      Full_Name (1 .. Dir_Name_Length)
        := Search_Directories.Table (Dir_Index).all;
      Full_Name (Dir_Name_Length + 1 .. Full_Name'Length) := File_Name;

      if not Unix_Is_Regular_File (Full_Name) then
         return False;
      else
         Name_Len := Full_Name'Length;
         Name_Buffer (1 .. Name_Len) := Full_Name;
         Save_Full_File_Name := Name_Enter;
         return True;
      end if;

   end Locate_File;

   ------------------------
   -- Write_Program_Name --
   ------------------------

   procedure Write_Program_Name is
   begin
      Write_Str (Arg_Value (0).all);
   end Write_Program_Name;

   -------------------------------------
   -- System_Maximum_File_Name_Length --
   -------------------------------------

   function System_Maximum_File_Name_Length return Pos is
   begin
      return Int'Last;
   end System_Maximum_File_Name_Length;

   -----------------------
   -- More_Source_Files --
   -----------------------

   function More_Source_Files return Boolean is
   begin
      pragma Assert (In_Compiler or In_Make);
      return (Current_File_Name_Index < Number_File_Names);
   end More_Source_Files;

   ----------------------
   -- Next_Main_Source --
   ----------------------

   function Next_Main_Source return File_Name_Type is
      File_Name : Arg_Ptr;
      Fptr      : Natural;

   begin
      pragma Assert (In_Compiler or In_Make);
      Current_File_Name_Index := Current_File_Name_Index + 1;

      --  Fatal error if no more files (should call More_Source_Files)

      pragma Assert (Current_File_Name_Index <= Number_File_Names);

      --  Otherwise return name of the file

      File_Name := File_Names (Current_File_Name_Index);
      Fptr := File_Name'First;

      for I in reverse File_Name'range loop
         if File_Name (I) = Directory_Separator then
            Fptr := I + 1;
            exit;
         end if;
      end loop;

      --  Save name of directory in which main unit resides for use in
      --  locating other units

      Search_Directories.Table (Primary_Directory)
        := new String'(File_Name (File_Name'First .. Fptr - 1));

      Name_Len := File_Name'Last - Fptr + 1;

      Name_Buffer (1 .. Name_Len) := File_Name (Fptr .. File_Name'Last);
      Save_Main_File_Name := File_Name_Type (Name_Find);
      return Save_Main_File_Name;
   end Next_Main_Source;

   ----------------------
   -- Read_Source_File --
   ----------------------

   function Read_Source_File (N : File_Name_Type) return Source_Buffer_Ptr is

      Source_File_FD : Unix_FD;
      --  The file descriptor for the current source file. A negative value
      --  indicates failure to open the specified source file.

      Text : Source_Buffer_Ptr;
      --  Allocated text buffer

      File_Located : Boolean;

      Is_Main_Unit : constant Boolean := (N = Save_Main_File_Name);

   begin
      --  First Locate the file (setting Save_Full_File_Name in the
      --  process).  The first place to look is in the directory of the
      --  main unit.  If the file is the main unit and it is not found
      --  in the directory specified for it, it is an error.

      Get_Name_String (N);

      File_Located :=
        Locate_File (Primary_Directory, Name_Buffer (1 .. Name_Len));

      if not File_Located then

         if Is_Main_Unit then

            --  An error. Main unit was not found in its specified directory

            Get_Name_String (N);
            Write_Str ("Cannot find: ");
            Write_Str (Name_Buffer (1 .. Name_Len));
            Write_Eol;
            Exit_Program (E_Fatal);

         else
            --  This is not the main unit, so look for it in the other
            --  places on the search path.

            for Dir_Index in
              Primary_Directory + 1 .. Search_Directories.Last
            loop
               File_Located
                 := Locate_File (Dir_Index, Name_Buffer (1 .. Name_Len));

               exit when File_Located;
            end loop;
         end if;

      end if;

      --  If file has still not been found, then we have an error

      if not File_Located then
         return null;
      end if;

      --  File name has been set in Save_Full_File_Name. Open it.

      Get_Name_String (Save_Full_File_Name);
      Name_Buffer (Name_Len + 1) := NUL;
      Source_File_FD := Unix_Open_Read (Name_Buffer'Address);

      if Source_File_FD < 0 then
         return null;
      end if;

      --  Read data from the file

      declare
         Len : Int := Unix_File_Length (Source_File_FD);
         --  Length of source file text

         Lo : Source_Ptr := Next_Source_Low_Bound;
         --  Low bound for allocated text buffer

         Hi : Source_Ptr := Lo + Source_Ptr (Len);
         --  High bound for allocated text buffer. Note length is Len + 1
         --  which allows for extra EOF character at the end of the buffer.

         Block_Size : constant := 2**14;
         --  Block size for read

         Ptr : Source_Ptr := Lo;
         --  Next location in text buffer to fill

         Count : Int;
         --  Count of characters read

      begin
         --  Allocate text buffer, allowing extra character at end for EOF

         Text := new Source_Buffer (Lo .. Hi);

         loop
            Count :=
              Unix_Read (Source_File_FD, Text (Ptr)'Address, Block_Size);
            exit when Count < Block_Size;
            Ptr := Ptr + Block_Size;
         end loop;

         Text (Hi) := EOF;
         Next_Source_Low_Bound := Hi + 1;
      end;

      --  Read is complete, get time stamp, close file and we are done

      Source_Time_Stamp := Unix_File_Time_Stamp (Source_File_FD);
      Unixlib.Unix_Close (Source_File_FD);
      return Text;

   end Read_Source_File;

   -------------------------------
   -- Current_Source_File_Stamp --
   -------------------------------

   function Current_Source_File_Stamp return Time_Stamp_Type is
   begin
      return Source_Time_Stamp;
   end Current_Source_File_Stamp;

   ----------------------
   -- Full_Source_Name --
   ----------------------

   function Full_Source_Name return Name_Id is
   begin
      return Save_Full_File_Name;
   end Full_Source_Name;

   -----------------------
   -- Source_File_Stamp --
   -----------------------

   function Source_File_Stamp (Name : File_Name_Type) return Time_Stamp_Type
   is
      Text : Source_Buffer_Ptr;

   begin
      Text := Read_Source_File (Name);

      if Text = null then
         return "            ";
      else
         Free (Text);
         return Source_Time_Stamp;
      end if;
   end Source_File_Stamp;

   --------------------
   -- More_Lib_Files --
   --------------------

   function More_Lib_Files return Boolean is
   begin
      pragma Assert (In_Binder);
      return (Current_File_Name_Index < Number_File_Names);
   end More_Lib_Files;

   ------------------------
   -- Next_Main_Lib_File --
   ------------------------

   function Next_Main_Lib_File return File_Name_Type is
      File_Name : Arg_Ptr;
      Fptr      : Natural;

   begin
      pragma Assert (In_Binder);
      Current_File_Name_Index := Current_File_Name_Index + 1;

      --  Fatal error if no more files (should call More_Lib_Files)

      pragma Assert (Current_File_Name_Index <= Number_File_Names);

      --  Otherwise return name of the file

      File_Name := File_Names (Current_File_Name_Index);
      Fptr := File_Name'First;

      for J in reverse File_Name'range loop
         if File_Name (J) = Directory_Separator then
            Fptr := J + 1;
         end if;
      end loop;

      Name_Len := File_Name'Last - Fptr + 1;

      Name_Buffer (1 .. Name_Len) := File_Name (Fptr .. File_Name'Last);
      Next_Source_Low_Bound := 0;
      return File_Name_Type (Name_Find);
   end Next_Main_Lib_File;

   -----------------------
   -- Read_Library_Info --
   -----------------------

   function Read_Library_Info
     (Lib_File  : File_Name_Type;
      Fatal_Err : Boolean)
      return      Text_Buffer_Ptr
   is
      Lib_FD : Unix_FD;
      --  The file descriptor for the current library file. A negative value
      --  indicates failure to open the specified source file.

      Text : Source_Buffer_Ptr;
      --  Allocated text buffer.

      File_Located : Boolean;

   begin
      if Lib_File = No_File then
         Name_Len := File_Names (Current_File_Name_Index)'Length;
         Name_Buffer (1 .. Name_Len)
           := File_Names (Current_File_Name_Index).all;
         Save_Full_File_Name := Name_Enter;
         File_Located := Unix_Is_Regular_File (Name_Buffer (1 .. Name_Len));

      else
         Get_Name_String (Lib_File);

         for Dir_Index in
                 Search_Directories.First .. Search_Directories.Last
         loop
            File_Located
              := Locate_File (Dir_Index, Name_Buffer (1 .. Name_Len));
            exit when File_Located;
         end loop;
      end if;

      if not File_Located then
         if Fatal_Err then
            Write_Str ("Cannot find: ");
            Write_Str (Name_Buffer (1 .. Name_Len));
            Write_Eol;
            Exit_Program (E_Fatal);
         else
            return null;
         end if;
      end if;

      Get_Name_String (Save_Full_File_Name);
      Name_Buffer (Name_Len + 1) := NUL;
      Lib_FD := Unix_Open_Read (Name_Buffer'Address);

      if Lib_FD < 0 then
         if Fatal_Err then
            Write_Str ("Cannot open: ");
            Write_Str (Name_Buffer (1 .. Name_Len));
            Write_Eol;
            Exit_Program (E_Fatal);
         else
            return null;
         end if;
      end if;

      --  Read data from the file

      declare
         Len : Int := Unix_File_Length (Lib_FD);
         --  Length of source file text

         Lo : Source_Ptr := 0;
         --  Low bound for allocated text buffer

         Hi : Source_Ptr := Source_Ptr (Len);
         --  High bound for allocated text buffer. Note length is Len + 1
         --  which allows for extra EOF character at the end of the buffer.

         Block_Size : constant := 2**14;
         --  Block size for read

         Ptr : Source_Ptr := Lo;
         --  Next location in text buffer to fill

         Count : Int;
         --  Count of characters read

      begin
         Text := new Source_Buffer (Lo .. Hi);
         --  Note extra charater at end for EOF character

         loop
            Count :=
              Unix_Read (Lib_FD, Text (Ptr)'Address, Block_Size);
            exit when Count < Block_Size;
            Ptr := Ptr + Block_Size;
         end loop;

         Text (Hi) := EOF;
         Next_Source_Low_Bound := Hi + 1;
      end;

      --  Read is complete, close file and we are done

      Unixlib.Unix_Close (Lib_FD);
      return Text;

   end Read_Library_Info;

   ----------------------------
   -- Full_Library_Info_Name --
   ----------------------------

   function Full_Library_Info_Name return Name_Id is
   begin
      return Save_Full_File_Name;
   end Full_Library_Info_Name;

   ---------------------------
   -- Full_Object_File_Name --
   ---------------------------

   function Full_Object_File_Name return Name_Id is
   begin
      Get_Name_String (Full_Library_Info_Name);
      Name_Len := Name_Len - ALI_Suffix'Length;
      Name_Buffer (Name_Len + 1 .. Name_Len + Object_Suffix'Length) :=
                                                        Object_Suffix.all;
      Name_Len := Name_Len + Object_Suffix'Length;
      return Name_Enter;
   end Full_Object_File_Name;

   --------------------------------
   -- Create_Output_Library_Info --
   --------------------------------

   procedure Create_Output_Library_Info is
      --  ??? Needs to be coordinated wiht -o option
      Dot_Index : Natural;

   begin
      pragma Assert (In_Compiler);
      Get_Name_String (Save_Main_File_Name);

      Dot_Index := 0;
      for I in reverse 1 .. Name_Len loop
         if Name_Buffer (I) = '.' then
            Dot_Index := I;
            exit;
         end if;
      end loop;

      --  Should be impossible to not have an extension

      if Dot_Index = 0 then
         null;
         pragma Assert (False);
      end if;

      declare
         Name_Buf : String (1 .. Dot_Index + 4);

      begin
         Name_Buf (1 .. Dot_Index) := Name_Buffer (1 .. Dot_Index);
         Name_Buf (Dot_Index + 1 .. Dot_Index + 3) := "ali";
         Name_Buf (Dot_Index + 4) := NUL;

         Output_FD := Unix_Create_File (Name_Buf'Address);

         if Output_FD < 0 then
            Write_Str ("Cannot create: ");
            Write_Str (Name_Buf);
            Write_Eol;
            Exit_Program (E_Fatal);
         end if;

      end;

   end Create_Output_Library_Info;

   ------------------------
   -- Write_Library_Info --
   ------------------------

   procedure Write_Library_Info (Info : String) is
   begin
      pragma Assert (In_Compiler);
      Unix_Write (Output_FD, Info'Address, Info'Length);
      Unix_Write (Output_FD, EOL'Address, 1);
   end Write_Library_Info;

   -------------------------------
   -- Close_Output_Library_Info --
   -------------------------------

   procedure Close_Output_Library_Info is
   begin
      pragma Assert (In_Compiler);
      Unix_Close (Output_FD);
   end Close_Output_Library_Info;

   -------------------
   -- Lib_File_Name --
   -------------------

   function Lib_File_Name
     (Source_File : File_Name_Type)
      return        File_Name_Type
   is
      Fptr : Natural;
      --  Pointer to location to set extension in place

   begin
      Get_Name_String (Source_File);
      Fptr := Name_Len + 1;

      for I in reverse 1 .. Name_Len loop
         if Name_Buffer (I) = '.' then
            Fptr := I;
            exit;
         end if;
      end loop;

      Name_Buffer (Fptr .. Fptr + 3) := ".ali";
      Name_Buffer (Fptr + 4) := NUL;
      Name_Len := Fptr + 3;
      return Name_Find;
   end Lib_File_Name;

   --------------------------
   -- Create_Binder_Output --
   --------------------------

   procedure Create_Binder_Output is
      File_Name : Arg_Ptr;
      Name_Buf  : String (1 .. 200);
      I         : Natural;

   begin
      pragma Assert (In_Binder);

      if (Output_Filename_Present) then

         if Output_Filename /= null then

            Name_Buf (Output_Filename'range) := Output_Filename.all;
            Name_Buf (Output_Filename'Last + 1) := NUL;
         else

            Write_Str ("Output filename missing after -o");
            Write_Eol;
            Exit_Program (E_Fatal);
         end if;
      else

         File_Name := File_Names (Current_File_Name_Index);
         Name_Buf (1 .. 2) := "b_";
         Name_Buf (3 .. File_Name'Length + 2) := File_Name.all;
         Name_Buf (File_Name'Length + 3) := '.';

         I := 3;

         while Name_Buf (I) /=  '.' loop
            I := I + 1;
         end loop;

         Name_Buf (I + 1) := 'c';
         Name_Buf (I + 2) := NUL;
      end if;

      Output_FD := Unix_Create_File (Name_Buf'Address);

      if Output_FD < 0 then
         Write_Str ("Cannot create: ");
         Write_Str (Name_Buf);
         Write_Eol;
         Exit_Program (E_Fatal);
      end if;

   end Create_Binder_Output;

   ------------------------
   -- Write_Binder_Info --
   ------------------------

   procedure Write_Binder_Info (Info : String) is
   begin
      pragma Assert (In_Binder);
      Unix_Write (Output_FD, Info'Address, Info'Length);
      Unix_Write (Output_FD, EOL'Address, 1);
   end Write_Binder_Info;

   -------------------------
   -- Close_Binder_Output --
   -------------------------

   procedure Close_Binder_Output is
   begin
      pragma Assert (In_Binder);
      Unix_Close (Output_FD);
   end Close_Binder_Output;

   ---------------------
   -- Number_Of_Files --
   ---------------------

   function Number_Of_Files return Int is
   begin
      return Number_File_Names;
   end Number_Of_Files;

   ------------------------
   -- Create_Req_Output  --
   ------------------------

   procedure Create_Req_Output is
   begin
      pragma Assert (In_Compiler or In_Make);
      Output_FD := Unix_Create_File (Name_Buffer'Address);

      if Output_FD < 0 then
         Write_Str ("Cannot create REQ File :");
         Write_Str (Name_Buffer);
         Output.Write_Eol;
         Exit_Program (E_Fatal);
      end if;

   end Create_Req_Output;

   -------------------------
   -- Create_Xref_Output  --
   -------------------------

   procedure Create_Xref_Output is
      Multi_File_Xref : constant String := "X.ref";

   begin
      pragma Assert (In_Compiler);

      --  For now, always use X.ref, since cannot reference Lib

      Name_Buffer (1 .. Multi_File_Xref'Last) := Multi_File_Xref;
      Name_Len := Multi_File_Xref'Last;
      Name_Buffer (Name_Len + 1) := NUL;

      Output_FD := Unix_Create_File (Name_Buffer'Address);

      if Output_FD < 0 then
         Write_Str ("Cannot create Xref-File !!!");
         Write_Eol;
         Exit_Program (E_Fatal);
      end if;
   end Create_Xref_Output;

   -------------------------
   -- Write_Xref_Output  --
   -------------------------

   procedure Write_Xref_Info (Info : String; Eol : Boolean := True) is
   begin
      pragma Assert (In_Compiler);
      Unix_Write (Output_FD, Info'Address, Info'Length);

      if Eol then
         Unix_Write (Output_FD, Osint.EOL'Address, 1);
      end if;
   end Write_Xref_Info;

   -------------------------
   -- Close_Xref_Output   --
   -------------------------

   procedure Close_Xref_Output is
   begin
      pragma Assert (In_Compiler);
      Unix_Close (Output_FD);
   end Close_Xref_Output;

   ------------------
   -- Exit_Program --
   ------------------

   procedure Exit_Program (Exit_Code : Exit_Code_Type) is
   begin
      case Exit_Code is
         when E_Success    => Unix_Exit (0);
         when E_Warnings   => Unix_Exit (0);
         when E_Errors     => Unix_Exit (1);
         when E_Fatal      => Unix_Exit (2);
         when E_Abort      => Unix_Abort;
      end case;
   end Exit_Program;

   ----------------
   -- Do_Compile --
   ----------------

   function Do_Compile (Source_File : File_Name_Type) return Exit_Code_Type is
   begin
      return E_Success;
   end Do_Compile;

   -------------
   -- Do_Bind --
   -------------

   function Do_Bind (Lib_File : File_Name_Type) return Exit_Code_Type is
   begin
      return E_Success;
   end Do_Bind;

end Osint;
