------------------------------------------------------------------------------
--                                                                          --
--                         GNAT COMPILER COMPONENTS                         --
--                                                                          --
--                              B I N D G E N                               --
--                                                                          --
--                                 B o d y                                  --
--                                                                          --
--                            $Revision: 1.28 $                             --
--                                                                          --
--           Copyright (c) 1992,1993,1994 NYU, All Rights Reserved          --
--                                                                          --
-- GNAT is free software;  you can  redistribute it  and/or modify it under --
-- terms of the  GNU General Public License as published  by the Free Soft- --
-- ware  Foundation;  either version 2,  or (at your option) any later ver- --
-- sion.  GNAT is distributed in the hope that it will be useful, but WITH- --
-- OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
-- or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License --
-- for  more details.  You should have  received  a copy of the GNU General --
-- Public License  distributed with GNAT;  see file COPYING.  If not, write --
-- to the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA. --
--                                                                          --
------------------------------------------------------------------------------

with ALI;    use ALI;
with Binde;  use Binde;
with Namet;  use Namet;
with Opt;    use Opt;
with Osint;  use Osint;
with Types;  use Types;

package body Bindgen is

   Statement_Buffer : String (1 .. 1000);
   --  Buffer used for constructing output statements

   Withed_Text_IO : Boolean := False;
   --  Flag which indicates whether the program has a context clause for
   --  units Text_IO or Ada.Text_IO.

   -----------------------
   -- Local Subprograms --
   -----------------------

   procedure Gen_Elab_Calls;
   --  Generate sequence of elaboration calls

   procedure Gen_Main_Program_File;
   --  Generate lines for output file in main program case

   procedure Gen_Non_Main_Program_File;
   --  Generate lines for output file in non-main program case

   procedure List_Object_Files;
   --  Output a comment containing a list of the full names of the object
   --  files to be linked

   ---------------------
   -- Gen_Output_File --
   ---------------------

   procedure Gen_Output_File is
   begin
      Create_Binder_Output;

      if Bind_Main_Program then
         Gen_Main_Program_File;
      else
         Gen_Non_Main_Program_File;
      end if;

      Close_Binder_Output;
   end Gen_Output_File;

   --------------------
   -- Gen_Elab_Calls --
   --------------------

   procedure Gen_Elab_Calls is
      L   : Natural;
      Col : Natural;

   begin
      for E in Elab_Order.First .. Elab_Order.Last loop
         Get_Name_String (Unit.Table (Elab_Order.Table (E)).Uname);

         --  Temporary check to see if the unit Text_IO has been withed. If it
         --  has been we need to set the flag "Withed_Text_IO" so that there
         --  is a call to "ada__text_io__aux__textio_finalization" inserted
         --  immediately after the main program completes. When "finalization"
         --  is implemented this code will be removed and be replaced by the
         --  appropriate finalization action in Text_IO ???.

         if Name_Buffer (1 .. 11) = "ada.text_io" then
            Withed_Text_IO := True;
         end if;

         Statement_Buffer (1 .. 3) := "   ";

         --  Copy the unit name (and replace '.' by '__' for child unit)

         L := 4;

         for I in 1 .. Name_Len - 2 loop
            if Name_Buffer (I) /= '.' then
               Statement_Buffer (L) := Name_Buffer (I);
               L := L + 1;
            else
               Statement_Buffer (L .. L + 1) := "__";
               L := L + 2;
            end if;
         end loop;

         --  Complete call to elaboration routine

         Statement_Buffer (L .. L + 6) := "___elab";
         Statement_Buffer (L + 7) := Name_Buffer (Name_Len);
         Statement_Buffer (L + 8 .. L + 11) := " ();";
         L := L + 11;
         Write_Binder_Info (Statement_Buffer (1 .. L));
      end loop;
   end Gen_Elab_Calls;

   ---------------------------
   -- Gen_Main_Program_File --
   ---------------------------

   procedure Gen_Main_Program_File is
   begin
      Write_Binder_Info ("#include <string.h>");
      Write_Binder_Info ("  ");
      Write_Binder_Info ("/* predefined exceptions */");
      Write_Binder_Info ("char constraint_error = 0;");
      Write_Binder_Info ("char numeric_error    = 0;");
      Write_Binder_Info ("char program_error    = 0;");
      Write_Binder_Info ("char storage_error    = 0;");
      Write_Binder_Info ("char tasking_error    = 0;");
      Write_Binder_Info ("char _abort_signal    = 1;");
      Write_Binder_Info (" ");
      Write_Binder_Info ("static int static_argc;");
      Write_Binder_Info ("static char * *static_argv;");
      Write_Binder_Info (" ");
      Write_Binder_Info ("int arg_count () { return static_argc; }");
      Write_Binder_Info (" ");
      Write_Binder_Info ("int len_arg (arg_num)");
      Write_Binder_Info ("   int arg_num;");
      Write_Binder_Info ("   { return strlen(static_argv[arg_num]); }");
      Write_Binder_Info (" ");
      Write_Binder_Info ("int fill_arg (a, i)");
      Write_Binder_Info ("   char * a;");
      Write_Binder_Info ("   int i;");
      Write_Binder_Info ("{ strncpy (a, static_argv[i],");
      Write_Binder_Info ("     strlen(static_argv[i])); }");
      Write_Binder_Info (" ");
      Write_Binder_Info ("extern void "
        & "(*system__tasking_soft_links__abort_defer) ();  ");
      Write_Binder_Info ("extern char "
        & "*system__task_specific_data__get_gnat_exception ();");
      Write_Binder_Info ("extern int "
        & "*system__task_specific_data__get_jmpbuf_address ();");
      Write_Binder_Info ("extern char debug__get_debug_flag_k (); ");
      Write_Binder_Info ("");
      Write_Binder_Info ("void");
      Write_Binder_Info ("__gnat_raise_nodefer (except)");
      Write_Binder_Info ("     char *except;");
      Write_Binder_Info ("{");
      Write_Binder_Info ("  int *ptr = "
        & "system__task_specific_data__get_jmpbuf_address ();");
      Write_Binder_Info ("");
      Write_Binder_Info ("  system__task_specific_data__set_gnat_exception"
        & " (except);");
      Write_Binder_Info ("  if (ptr)");
      Write_Binder_Info ("    longjmp (ptr, 1);");
      Write_Binder_Info ("");
      Write_Binder_Info ("  else ");
      Write_Binder_Info ("    {");
      Write_Binder_Info ("      if (except == &constraint_error) ");
      Write_Binder_Info ("        puts (""\nraised Constraint_Error\n""); ");
      Write_Binder_Info ("      else if (except == &numeric_error) ");
      Write_Binder_Info ("        puts (""\nraised Numeric_Error\n""); ");
      Write_Binder_Info ("      else if (except == &program_error) ");
      Write_Binder_Info ("        puts (""\nraised Program_Error\n""); ");
      Write_Binder_Info ("      else if (except == &storage_error)");
      Write_Binder_Info ("        puts (""\nraised Storage_Error\n"");");
      Write_Binder_Info ("      else if (except == &tasking_error)");
      Write_Binder_Info ("        puts (""\nraised Tasking_Error\n"");");
      Write_Binder_Info ("      else if (!ptr)");
      Write_Binder_Info ("        puts (""\nraised unhandled exception\n"");");
      Write_Binder_Info ("");
      Write_Binder_Info ("      exit (1);");
      Write_Binder_Info ("");
      Write_Binder_Info ("    }");
      Write_Binder_Info ("}");
      Write_Binder_Info ("");
      Write_Binder_Info ("void ");
      Write_Binder_Info ("__gnat_raise (except)");
      Write_Binder_Info ("     char *except;");
      Write_Binder_Info ("{");
      Write_Binder_Info ("  (*system__tasking_soft_links__abort_defer) ();");
      Write_Binder_Info ("  __gnat_raise_nodefer (except);");
      Write_Binder_Info ("}");
      Write_Binder_Info ("");
      Write_Binder_Info ("void");
      Write_Binder_Info ("__gnat_reraise (flag)");
      Write_Binder_Info ("     int flag;");
      Write_Binder_Info ("{");
      Write_Binder_Info ("  char *except = "
        & "system__task_specific_data__get_gnat_exception ();");
      Write_Binder_Info ("");
      Write_Binder_Info ("  if (flag)");
      Write_Binder_Info ("    __gnat_raise (except);");
      Write_Binder_Info ("  else");
      Write_Binder_Info ("    __gnat_raise_nodefer (except);");
      Write_Binder_Info ("}");
      Write_Binder_Info ("void");
      Write_Binder_Info ("__gnat_raise_constraint_error ()");
      Write_Binder_Info ("{");
      Write_Binder_Info ("  __gnat_raise (&constraint_error);");
      Write_Binder_Info ("}");
      --  Generate __main_priority function

      declare
         Ctr : Integer;
         P   : Int;

         procedure Set_Int (N : Nat) is
         begin
            if N > 9 then
               Set_Int (N / 10);
            else
               Statement_Buffer (Ctr) :=
                 Character'Val (N mod 10 + Character'Pos ('0'));
               Ctr := Ctr + 1;
            end if;
         end Set_Int;

      begin
         Statement_Buffer (1 .. 32) := "int __main_priority () { return ";
         Ctr := 33;
         P := ALIs.Table (ALIs.First).Main_Priority;

         if P < 0 then
            P := -P;
            Statement_Buffer (Ctr) := '-';
            Ctr := Ctr + 1;
         end if;

         Set_Int (P);
         Statement_Buffer (Ctr .. Ctr + 2) := "; }";
         Write_Binder_Info (Statement_Buffer (1 .. Ctr + 2));
         Write_Binder_Info (" ");
      end;

      --  Generate main

      if ALIs.Table (ALIs.First).Main_Program = Proc then
         Write_Binder_Info ("void main (argc, argv)");
      else
         Write_Binder_Info ("int main (argc, argv)");
      end if;

      Write_Binder_Info ("   int argc;");
      Write_Binder_Info ("   char * argv[];");
      Write_Binder_Info ("{");
      Write_Binder_Info ("   static_argc = argc;");
      Write_Binder_Info ("   static_argv = argv;");
      Write_Binder_Info (" ");

      Gen_Elab_Calls;

      Write_Binder_Info (" ");
      Get_Name_String (Unit.Table (First_Unit_Entry).Uname);

      --  Main program is procedure case

      if ALIs.Table (ALIs.First).Main_Program = Proc then
         Statement_Buffer (1 .. 8) := "   _ada_";
         Statement_Buffer (9 .. Name_Len + 6) :=
           Name_Buffer (1 .. Name_Len - 2);
         Statement_Buffer (Name_Len + 7 .. Name_Len + 10) := " ();";
         Write_Binder_Info (Statement_Buffer (1 .. Name_Len + 10));

      --  Main program is function case

      else -- ALIs.Table (ALIs_First).Main_Program = Func
         Statement_Buffer (1 .. 16) := "   return (_ada_";
         Statement_Buffer (17 .. Name_Len + 14) :=
           Name_Buffer (1 .. Name_Len - 2);
         Statement_Buffer (Name_Len + 15 .. Name_Len + 19) := " ());";
         Write_Binder_Info (Statement_Buffer (1 .. Name_Len + 19));
      end if;

      --  Generate a call to the text_io procedure which performs cleanup
      --  operations on temporary files. See comment above at the beginning
      --  of Gen_Elab_Calls.

      if Withed_Text_IO then
         Write_Binder_Info ("   ada__text_io__aux__text_io_finalization ();");
      end if;

      Write_Binder_Info ("   exit (0);");
      Write_Binder_Info ("}");
      List_Object_Files;
   end Gen_Main_Program_File;

   -------------------------------
   -- Gen_Non_Main_Program_File --
   -------------------------------

   procedure Gen_Non_Main_Program_File is
   begin
      Write_Binder_Info ("void ada__bind ()");
      Write_Binder_Info ("{");
      Gen_Elab_Calls;
      Write_Binder_Info ("}");
      List_Object_Files;
   end Gen_Non_Main_Program_File;

   -----------------------
   -- List_Object_Files --
   -----------------------

   procedure List_Object_Files is
   begin
      Write_Binder_Info ("/* BEGIN Object file list");

      for E in Elab_Order.First .. Elab_Order.Last loop
         Get_Name_String (Unit.Table (Elab_Order.Table (E)).Uname);

         --  If not spec that has an associated body, then generate a
         --  comment giving the name of the corresponding ALI file

         if Unit.Table (Elab_Order.Table (E)).Utype /= Is_Spec then

            --  Now output the file name as a comment

            Get_Name_String
              (ALIs.Table
                (Unit.Table (Elab_Order.Table (E)).My_ALI).Ofile_Full_Name);
            Write_Binder_Info (Name_Buffer (1 .. Name_Len));
         end if;
      end loop;

      Write_Binder_Info ("   END Object file list */");
   end List_Object_Files;

end Bindgen;
