/*
 *      Copyright (C) 1993 Ning and David Mosberger.
 *      Original code:
 *      Copyright (C) 1993 Bas Laarhoven.
 *      Copyright (C) 1992 David L. Brown, Jr.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2, or (at
 * your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; see the file COPYING.  If not, write to
 * the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139,
 * USA.
 *
 * $Source: /usr/src/distr/ftape-1.12/RCS/tecc.c,v $
 * $Author: bas $
 *
 * $Revision: 1.8 $
 * $Date: 1994/05/26 23:43:45 $
 * $State: Beta $
 *
 *      This is a test program for the code implementing the QIC-40/80
 *      Reed-Solomon Error-Correction Code.
 */
static const char RCSid[] =
  "$Id: tecc.c,v 1.8 1994/05/26 23:43:45 bas Beta bas $";

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/time.h>

#include "ecc.h"
#include "ftape.h"	/* to get SECTOR_SIZE */

#define KB			(1024)
#define MB			(KB*KB)
#define MICRO_SEC_PER_SEC	(1000000)

static int verbose;
static int nerrors;
static int nerasures;
static struct memory_segment mseg;
static unsigned char test_data[SECTOR_SIZE * 32];
static unsigned char backup_data[SECTOR_SIZE * 32];


/*
 * Fills DATA array with test codewords from QIC-40/80 specifications.
 * The spec gives data for the first seven columns only, so we fill
 * the remaining with pseudo random data (columns 7 through
 * SECTOR_SIZE).
 */
static void
generate_test_patterns(unsigned char *data)
{
    int i, j;

    for (i = 0; i < 32 * SECTOR_SIZE; ++i) {
	data[i] = 0;
    } /* for */

    data[28 * SECTOR_SIZE]     = 0x01;
    data[27 * SECTOR_SIZE + 1] = 0x01;
    data[26 * SECTOR_SIZE + 2] = 0x01;
    data[25 * SECTOR_SIZE + 3] = 0x01;
    data[24 * SECTOR_SIZE + 4] = 0x01;

    data[ 2 * SECTOR_SIZE + 5] = 0x01;
    data[ 3 * SECTOR_SIZE + 5] = 0xc0;
    data[ 4 * SECTOR_SIZE + 5] = 0xc0;
    data[ 5 * SECTOR_SIZE + 5] = 0x01;
    data[ 6 * SECTOR_SIZE + 5] = 0x01;
    data[ 8 * SECTOR_SIZE + 5] = 0x67;
    data[ 9 * SECTOR_SIZE + 5] = 0xa6;
    data[10 * SECTOR_SIZE + 5] = 0xc0;
    data[11 * SECTOR_SIZE + 5] = 0x01;
    data[14 * SECTOR_SIZE + 5] = 0xff;
    data[15 * SECTOR_SIZE + 5] = 0x99;
    data[16 * SECTOR_SIZE + 5] = 0x67;
    data[17 * SECTOR_SIZE + 5] = 0x01;
    data[21 * SECTOR_SIZE + 5] = 0xa3;
    data[22 * SECTOR_SIZE + 5] = 0x5d;
    data[23 * SECTOR_SIZE + 5] = 0xff;
    data[24 * SECTOR_SIZE + 5] = 0x01;

    for (i = 0; i < 32; ++i) {
	data[i * SECTOR_SIZE + 6] = i + 1;
    } /* for */

    /*
     * Even though we want pseudo-random data, we want it to be
     * reproducible, so we always use the same seed value.
     */
    srand48(0);
    for (i = 0; i < 29; ++i) {
	for (j = 7; j < SECTOR_SIZE; ++j) {
	    data[i * SECTOR_SIZE + j] = (unsigned char) lrand48();
	} /* for */
    } /* for */
} /* generate_test_patterns */


static int
verify_parity(unsigned char *data)
{
    int i, j;
    unsigned char p;
    int parity_errors = 0;
    static unsigned char test_parity[7][3] = {
	{ 0xc0, 0xc0, 0x01, },
	{ 0x67, 0xa6, 0xc0, },
	{ 0xff, 0x99, 0x67, },
	{ 0xa3, 0x5d, 0xff, },
	{ 0xad, 0x0f, 0xa3, },
	{ 0xad, 0x0f, 0xa3, },
	{ 0x5d, 0xff, 0xa3, },
    };

    for (i = 0; i < 7; ++i) {
	if (verbose) {
	    printf("parity of test code word #%d:", i + 1);
	    for (j = 0; j < 3; ++j) {
		printf("  0x%02x",  data[(29 + j) * SECTOR_SIZE + i]);
	    } /* for */
	    printf("\n");
	} /* if */
	for (j = 0; j < 3; ++j) {
	    p = data[(29 + j) * SECTOR_SIZE + i];
	    if (p != test_parity[i][j]) {
		printf("wrong parity: %s parity byte should be 0x%02x\n",
		       (j == 0) ? "1st" : (j == 1) ? "2nd" : "3rd",
		       test_parity[i][j]);
		++parity_errors;
	    } /* if */
	} /* for */
    } /* for */
    return parity_errors;
} /* verify_parity */


static int
next_pattern(int len, int *pat, int max_val)
{
    int i;

    i = len - 1;
    while (i >= 0 && (++pat[i] > max_val || pat[i] + len - 1 - i > max_val)) {
	--i;
    } /* while */
    if (i < 0) {
	/* no more patterns: */
	return 0;
    } /* if */
    while (i < len - 1) {
	pat[i+1] = pat[i] + 1;
	++i;
    } /* while */
    return 1;
} /* next_pattern */


/*
 * This function exhaustively tests the ECC with the specified
 * error pattern, number of erasures, and number of errors.
 * LENIENCY=0 means all errors must be correctable, LENIENCY=1
 * means all errors must be detectable, LENIENCY=2 means all
 * bets are off.  A value of 2 also implies most printing to
 * be turned off.
 */
static int
verify_correction(int error_pattern, int nerasures, int nerrors, int nblocks,
		  struct memory_segment *mseg, unsigned char *data,
		  int leniency)
{
    int i, j, k, nlocs = nerasures + nerrors;
    int loc[nlocs];
    int error_loc[nerrors];
    int nbad, nfailures, result;
    BAD_SECTOR bad_mask, crc_mask;
    unsigned char *dp1, *dp2;

    printf("Verifying %d blocks with %d CRC errors, %d CRC failures, "
	   "error pattern 0x%02x.\n",
	   nblocks, nerasures, nerrors, error_pattern);

    for (i = 0; i < nerasures + nerrors; ++i) {
	loc[i] = i;
    } /* for */

    nfailures = 0;
    k = 1;
    do {
	for (i = 0; i < nerrors; ++i) {
	    error_loc[i] = i;
	} /* for */

	do {
	    bad_mask = 0;
	    /* add in the errors: */
	    for (i = 0; i < nlocs;  ++i) {
		bad_mask |= 1 << loc[i];
		for (j = 0; j < SECTOR_SIZE; ++j) {
		    mseg->data[loc[i] * SECTOR_SIZE + j] = error_pattern;
		} /* for */
	    } /* for */
	    /* create CRC failures: */
	    crc_mask = bad_mask;
	    for (i = 0; i < nerrors; ++i) {
		crc_mask ^= 1 << loc[error_loc[i]];
	    } /* for */

	    mseg->read_bad  = crc_mask;
	    mseg->corrected = 0x5a5a5a5a;	/* force junk value */
	    result = ecc_correct_data(mseg);

	    if (result == ECC_OK || result == ECC_CORRECTED) {
		/* check for correct computation of mseg->corrected: */
		if (mseg->corrected != bad_mask && leniency < 2) {
		    printf("ECC failure: mseg->corrected=%lx, should be %lx\n",
			   mseg->corrected, bad_mask);
		    ++nfailures;
		} /* if */

		/* let's be paranoid and verify each and every byte: */
		nbad = 0;
		dp1 = mseg->data;
		dp2 = data;
		for (i = 0; i < nblocks; ++i) {
		    for (j = 0; j < SECTOR_SIZE; ++j) {
			if (*dp1 != *dp2) {
			    if (leniency < 2) {
				if (++nbad <= 10) {
				    printf("ECC failure: "
					   "data[row=%2d][col=%4d]=0x%02x "
					   "instead of 0x%02x!\n",
					   i, j, (int) *dp1, (int) *dp2);
				} else if (nbad == 11) {
				    printf("ECC failure: (remaining "
					   "errors supressed...)\n");
				} /* if */
			    } /* if */
			    /* restore correct data: */
			    *dp1 = *dp2;
			} /* if */
			++dp1; ++dp2;
		    } /* for */
		} /* for */
		nfailures += nbad;
		if (nbad) {
		    printf("%d: ecc_correct_data(): "
			   "failed to detect %d errors (returned %s)\n", k,
			   nbad,
			   (result == ECC_OK) ? "ECC_OK" : "ECC_CORRECTED");
		} else if (verbose) {
		    printf("%d: ecc_correct_data(): succeeded (returned %s)\n",
			   k, (result == ECC_OK) ? "ECC_OK" : "ECC_CORRECTED");
		} /* if */
	    } else {
		if (!leniency) {
		    printf("%d: ecc_correct_data(): ", k);
		    printf("ECC failed on correctable errors!\n");
		    ++nfailures;
		} else if (leniency < 2 && verbose) {
		    printf("%d: ecc_correct_data(): ECC failed, "
			   "but this is OK\n", k);
		} /* if */
		/* restore data: */
		memcpy(mseg->data, data, nblocks * SECTOR_SIZE);
	    } /* if */
	    ++k;
	} while (next_pattern(nerrors, error_loc, nlocs - 1));
    } while (next_pattern(nlocs, loc, nblocks - 1));
    return nfailures;
} /* verify_correction */


/*
 * Do some tests to check the correctness of the error-correction.
 * In each test, the *error patterns* are tested exhaustively, but
 * not all *data patterns* are tested (would take a little bit
 * too long...).  If RIGOROSITY > 1, also do long running tests.
 */
static int
verify_qic_40_80_specs(int rigorosity)
{
    int nerasures, nerrors, nblocks;
    int nfailures = 0;

    /* first, test handling of small segments: */
    for (nblocks = 10; nblocks > 3; --nblocks) {
	mseg.data = test_data;
	mseg.blocks = nblocks;
	mseg.read_bad = 0;
	ecc_set_segment_parity(&mseg);
	memcpy(backup_data, mseg.data, sizeof(backup_data));
	nfailures = verify_correction(0x5a, 1, 1, nblocks, &mseg,
				      backup_data, 0);
	if (nfailures) {
	    printf("verify_qic_40_80_specs: test failed (%d failures)!\n",
		   nfailures);
	    return 0;
	} /* if */
    } /* for */

    generate_test_patterns(test_data);
    mseg.read_bad = 0;
    mseg.marked_bad = 0;		/* not used... */
    mseg.blocks = 32;
    mseg.data = test_data;
    ecc_set_segment_parity(&mseg);

    memcpy(backup_data, mseg.data, sizeof(backup_data));

    /* first, test cases ECC should be able to correct: */
    for (nerasures = 0; nerasures <= 3; ++nerasures) {
	for (nerrors = 0; nerrors <= 1; ++nerrors) {
	    if (2 * nerrors + nerasures <= 3) {
		nfailures = verify_correction(0xe5, nerasures, nerrors, 32,
					      &mseg, backup_data, 0);
		if (nfailures) {
		    printf("verify_qic_40_80_specs: test failed "
			   "(%d failures)!\n", nfailures);
		    return 0;
		} /* if */
	    } /* if */
	} /* for */
    } /* for */

    /* now, test cases ECC should be able to *detect*: */
    for (nerasures = 0; nerasures <= 3; ++nerasures) {
	for (nerrors = 0; nerrors <= 2; ++nerrors) {
	    if (2 * nerrors + nerasures == 4) {
		nfailures = verify_correction(0xe5, nerasures, nerrors, 32,
					      &mseg, backup_data, 1);
		if (nfailures) {
		    printf("verify_qic_40_80_specs: test failed "
			   "(%d failures)!\n", nfailures);
		    return 0;
		} /* if */
	    } /* if */
	} /* for */
    } /* for */

    if (rigorosity > 1) {
	/* test cases for which ECC cannot guarantee anything: */

	printf("WARNING: the following tests may take several weeks to\n"
	       "\tfinish.  They are not extremely important, so you might\n"
	       "\twanna stop now.  For correct operation, the ecc code must\n"
	       "\thave been compiled with ECC_SANITY_CHECK on and you have\n"
	       "\tto check the output for error messages (the test doesn't\n"
	       "\tstop when errors occur).\n");

	for (nerasures = 0; nerasures <= 4; ++nerasures) {
	    for (nerrors = 0; nerrors <= 3; ++nerrors) {
		if (2 * nerrors + nerasures > 4) {
		    nfailures = verify_correction(0xe5, nerasures, nerrors, 32,
						  &mseg, backup_data, 2);
		    if (nfailures) {
			printf("verify_qic_40_80_specs: %d undetected "
			       "failures.\n",
			       nfailures);
		    } /* if */
		} /* if */
	    } /* for */
	} /* for */
    } /* if */

    return 1;
} /* verify_qic_40_80_specs */


static int
verify_pattern_gen(void)
{
#   define K 4
#   define N 9
    int i, j, pat[K];
    int n_choose_k, k_factorial;
    int OK = 0;

    n_choose_k  = N;
    k_factorial = 1;
    for (i = 1; i < K; ++i) {
	n_choose_k  *= N - i;
	k_factorial *= i + 1;
    } /* for */
    n_choose_k /= k_factorial;

    printf("Testing error pattern generator.  Should result in %d unique "
	   "vectors\n(ignoring permutations). "
	   "The program tests the number of patterns but not\nuniqueness.\n",
	   n_choose_k);

    for (i = 0; i < K; ++i) {
	pat[i] = i;
    } /* for */

    j = 0;
    do {
	printf("(");
	for (i = 0; i < K; ++i) {
	    if (i > 0) {
		printf(",");
	    } /* if */
	    printf("%d", pat[i]);
	} /* for */
	printf(") ");
	if (i % 8 == 7) {
	    printf("\n");
	} /* if */
	++j;
    } while (next_pattern(K, pat, N-1));
    printf("\n");

    if (j == n_choose_k) {
	printf("Test pattern generator generated expected number of "
	       "patterns.\n");
	OK = 1;
    } else {
	printf("Test pattern generator test failed: %d patterns instead of "
	       "the expected %d\n", j, n_choose_k);
    } /* if */
    return OK;
} /* verify_pattern_gen */


/*
 * Just a no-operation to measure timing overheads.
 */
void
null_function(void)
{
} /* null_function */


/*
 * Encode one memory segment.
 */
void
encode_segment(void)
{
    ecc_set_segment_parity(&mseg);
} /* encode_segment */


/*
 * Add NERASURES erasures and NERRORS errors to the segment.
 */
void
add_erasures_and_errors(void)
{
    int nbad = nerasures + nerrors;
    int i, j, row;

    mseg.read_bad = 0;
    for (j = 0; j < nbad; ++j) {
	row = j * (32/nbad);
	if (j < nerasures) {
	    mseg.read_bad |= 1 << row;
	} /* if */
	for (i = 0; i < SECTOR_SIZE; ++i) {
	    mseg.data[row * SECTOR_SIZE + i] = i;
	} /* for */
    } /* for */
} /* add_erasures_and_errors */


/*
 * Decode one memory segment after adding erasures and errors.
 */
void
decode_segment(void)
{
    add_erasures_and_errors();
    if (ecc_correct_data(&mseg) == ECC_FAILED) {
	printf("decode_segment: ECC correction failed!\n");
	exit(1);
    } /* if */
} /* decode_segment */


/*
 * Invokes FUNC for NREPETITIONS number of times.  The elapsed time
 * is returned in micro-seconds.
 */
static double
lapsed_time(int nrepetitions, void (*func)(void))
{
    struct timeval start, stop;
    int i;

    gettimeofday(&start, NULL);
    for (i = 0; i < nrepetitions; ++i) {
	(*func)();
    } /* for */
    gettimeofday(&stop, NULL);

    if (start.tv_usec > stop.tv_usec) {
	stop.tv_usec += 1000000;
	--stop.tv_sec;
    } /* if */
    return 1e6*(stop.tv_sec - start.tv_sec) + (stop.tv_usec - start.tv_usec);
} /* lapsed_time */


static void
measure_throughput(void)
{
    double avg_overhead, overhead, avg_lapsed, lapsed;
    int i, saved_tracing;
#   define TIMES 5

    /* setup test data: */
    generate_test_patterns(test_data);
    mseg.read_bad = 0;
    mseg.marked_bad = 0;		/* not used... */
    mseg.blocks = 32;
    mseg.data = test_data;

    /* measure timing overheads: */
    avg_overhead = 0.0;
    for (i = 0; i < TIMES; ++i) {
	overhead = lapsed_time(100000, null_function);
	overhead /= 1e5;
	if (verbose) {
	    printf("    %g micro-sec\n", overhead);
	} /* if */
	avg_overhead += overhead;
    } /* for */
    avg_overhead /= TIMES;
    if (verbose) {
	printf("Average timing overhead: %g microseconds\n\n", avg_overhead);
    } /* if */

    /* measure encoding rate: */
    avg_lapsed = 0.0;
    for (i = 0; i < TIMES; ++i) {
	lapsed = lapsed_time(100, encode_segment);
	lapsed /= 100;
	lapsed -= avg_overhead;
	if (verbose) {
	    printf("    %g KB/sec\n",
		   MICRO_SEC_PER_SEC/lapsed * sizeof(test_data)/KB);
	} /* if */
	avg_lapsed += lapsed;
    } /* for */
    avg_lapsed /= TIMES;
    printf("Average encoding throughput: %g KB/sec\n\n",
	   MICRO_SEC_PER_SEC/avg_lapsed * sizeof(test_data)/KB);

    saved_tracing = ftape_ecc_tracing;
    ftape_ecc_tracing = 0;			/* show bugs only */
    for (nerrors = 0; nerrors < 2; ++nerrors) {
	for (nerasures = 0; nerasures < 4; ++nerasures) {
	    if (2 *  nerrors + nerasures > 3) {
		continue;
	    } /* if */

	    /* measure time to add erasures and errors to data: */
	    avg_overhead = 0.0;
	    for (i = 0; i < TIMES; ++i) {
		overhead = lapsed_time(100, add_erasures_and_errors);
		overhead /= 100;
		avg_overhead += overhead;
	    } /* for */
	    avg_overhead /= TIMES;

	    /* measure decoding rate with erasures and errors: */
	    avg_lapsed = 0.0;
	    for (i = 0; i < TIMES; ++i) {
		lapsed = lapsed_time(100, decode_segment);
		lapsed /= 100;
		lapsed -= avg_overhead;
		if (verbose)  {
		    printf("    %g KB/sec\n",
			   MICRO_SEC_PER_SEC/lapsed * sizeof(test_data)/KB);
		} /* if */
		avg_lapsed += lapsed;
	    } /* for */
	    avg_lapsed /= TIMES;
	    printf("Average throughput with %d erasures, %d errors: "
		   "%g KB/sec\n\n", nerasures, nerrors,
		   MICRO_SEC_PER_SEC/avg_lapsed * sizeof(test_data)/KB);
	} /* for */
    } /* for */
    ftape_ecc_tracing = saved_tracing;
} /* measure_throughput */


static void
usage(char *pgm_name)
{
    fprintf(stderr, "usage: %s [ -tcavh ]\n", pgm_name);
    fprintf(stderr, "\t-t\tmeasure encoding/decoding throughput\n");
    fprintf(stderr, "\t-c\ttest correctness of ECC code\n");
    fprintf(stderr, "\t-c -c\tas above, but also do long running tests"
	    "(these may take a few weeks...)\n");
    fprintf(stderr, "\t-a\tdo all of the above tests\n");
    fprintf(stderr, "\t-v\tturn on verbose messages\n");
    fprintf(stderr, "\t-h\tprints this usage information\n");
} /* usage */


void
main(int argc, char **argv)
{
    int i;
    int throughput = 0;
    int correctness = 0;

    if (argc < 2) {
	usage(argv[0]);
	exit(1);
    } /* if */
    for (i = 0; i < argc; ++i) {
	if (strcmp(argv[i], "-v") == 0) {
	    ++verbose;
	} else if (strcmp(argv[i], "-t") == 0) {
	    ++throughput;
	} else if (strcmp(argv[i], "-c") == 0) {
	    ++correctness;
	} else if (strcmp(argv[i], "-a") == 0) {
	    ++throughput; ++correctness;
	} else if (strcmp(argv[i], "-h") == 0) {
	    usage(argv[0]);
	    exit(0);
	} /* if */
    } /* for */

    if (verbose) {
	ftape_ecc_tracing = 1;	/* show bugs and errors only */
    } else {
	ftape_ecc_tracing = 0;	/* show bugs only */
    } /* if */

    generate_test_patterns(test_data);
    mseg.data = test_data;
    mseg.read_bad = 0;
    mseg.blocks = 32;
    
    ecc_set_segment_parity(&mseg);
    if (verify_parity(mseg.data)) {
	printf("QIC 40/80 spec. parity generation failed.\n");
	exit(1);
    } /* if */

    if (correctness) {
	if (!verify_pattern_gen()) {
	    exit(1);
	} /* if */
	if (verify_qic_40_80_specs(correctness)) {
	    printf("Passed QIC-40/80 ECC tests!\n");
	} else {
	    printf("QIC-40/80 ECC tests FAILED!\n");
	} /* if */
    } /* if */

    if (throughput) {
	measure_throughput();
    } /* if */
    exit(0);
} /* main */

			/*** end of tecc.c */
