/*
 *      Copyright (C) 1994 Bas Laarhoven.

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; see the file COPYING.  If not, write to
the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

 $Source: /usr/src/distr/ftape-0.9.10/RCS/ftape-eof.c,v $
 $Author: bas $
 *
 $Revision: 1.5 $
 $Date: 1994/02/20 15:26:55 $
 $State: Alpha $
 *
 *      This file contains the reading and writing code
 *      for the QIC-40/80 floppy-tape driver for Linux.
 */

static char RCSid[] =
"$Id: ftape-eof.c,v 1.5 1994/02/20 15:26:55 bas Alpha bas $";


#include <linux/string.h>
#include <linux/errno.h>
#include <linux/sched.h>
#include <asm/dma.h>
#include <asm/segment.h>
#include <asm/system.h>

#include "ftape.h"
#include "ftape-eof.h"
#include "ftape-write.h"
#include "ftape-read.h"
#include "ftape-rw.h"
#include "ftape-io.h"

/*      Global vars.
 */
int failed_sector_log_changed = 0;

/*      Local vars.
 */
static struct failed_sector_entry {
  unsigned short sector;
  unsigned short segment;
} *last_failed_sector_mark;
static union {
  struct failed_sector_entry mark;
  unsigned long entry;
} failed_sector_log[ (2048 - 256) / 4];
static unsigned last_failed_sector_index;
static unsigned failed_sector_count = 0;


void
reset_eof_list( void)
{
  TRACE_FUN( 8, "reset_eof_list");

  last_failed_sector_mark = &failed_sector_log[ 0].mark;
  last_failed_sector_index = 0;
  TRACE_EXIT;
}

void
clear_eof_mark( unsigned segment, unsigned byte_count)
{
  TRACE_FUN( 5, "clear_eof_mark");
  static unsigned last_segment = INT_MAX;

  if (segment <= last_segment) {
    reset_eof_list();
  }
  last_segment = segment;

  for (;;) {
    if (last_failed_sector_index >= failed_sector_count ||
        segment < last_failed_sector_mark->segment) {
      TRACE( 5, "not found");
      break;
    }
    /*  at or past next eof mark.
     */
    if (segment == last_failed_sector_mark->segment) {
      if (byte_count < last_failed_sector_mark->sector * SECTOR_SIZE) {
        TRACE( 5, "not past");
        break;
      }
      /*  at eof mark.
       */
      memmove( &failed_sector_log[ last_failed_sector_index],
              &failed_sector_log[ last_failed_sector_index + 1],
              (failed_sector_count - last_failed_sector_index) *
              sizeof( *failed_sector_log));
      --failed_sector_count;
      failed_sector_log_changed = 1;
      TRACE( 5, "cleared");
    } else {
      /*  past eof mark, scan for following marks.
       */
      ++last_failed_sector_mark;
      ++last_failed_sector_index;
      TRACE( 5, "search next");
    }
  }
  TRACE_EXIT;
}

int
check_for_eof( unsigned segment)
{
  TRACE_FUN( 8, "check_for_eof");
  static unsigned last_segment = INT_MAX;
  int result = 0;

  if (segment <= last_segment) {
    reset_eof_list();
  }
  last_segment = segment;
  for (;;) {
    if (last_failed_sector_index >= failed_sector_count ||
        segment < last_failed_sector_mark->segment) {
      TRACE( 5, "not found");
      break;
    }
    /*  at or past next eof mark.
     */
    if (segment == last_failed_sector_mark->segment) {
      /*  at eof mark.
       */
      TRACE( 5, "hit");
      result = last_failed_sector_mark->sector;
      break;
    } else {
      /*  past eof mark, scan for following marks.
       */
      ++last_failed_sector_mark;
      ++last_failed_sector_index;
      TRACE( 5, "search next");
    }
  }
  TRACE_EXIT;
  return result;
}

void
put_file_mark_in_map( unsigned segment, unsigned sector)
{
  TRACE_FUN( 8, "put_file_mark_in_map");
  unsigned long new_entry = (segment << 16) + sector;
  unsigned long entry = 0;
  int i;

  for (i = 0 ; i < failed_sector_count ; ++i) {
    entry = failed_sector_log[ i].entry;
    if (new_entry <= entry) {
      break;                    /* insert at pos. i */
    }
  }
  if (entry == new_entry) {
    TRACEx1( 5, "0x%04x already exists", new_entry);
  } else {
    if (i < failed_sector_count) {
      TRACEx3( 5, "insert 0x%08x at index %d before 0x%08x",
              new_entry, i, entry);
      memmove( &failed_sector_log[ i + 1], &failed_sector_log[ i],
              (failed_sector_count - i) * sizeof( *failed_sector_log));
    } else {
      TRACEx1( 5, "appending 0x%08x", new_entry);
    }
    failed_sector_log[ i].entry = new_entry;
    ++failed_sector_count;
    failed_sector_log_changed = 1;
  }
  TRACE_EXIT;
}

/*  Write count file marks to tape starting at first non-bad
 *  sector following the given segment and sector.
 *  sector = base 1 !
 */
int
ftape_weof( unsigned count, unsigned segment, unsigned sector)
{
  TRACE_FUN( 5, "ftape_weof");
  int result = 0;
  unsigned long mask = get_bad_sector_map_entry( segment);
  unsigned sector_nr = 0;

  ftape_seg_pos += count;       /* new position */
  if (sector < 1 || sector > 29 || count > 29 ||
      segment >= ftape_last_segment.id) {
    TRACEx3( 5, "parameter out of range: %d, %d, %d", count, segment, sector);
    result = -EIO;
  } else {
    while (count-- > 0) {
      do {                        /* count logical sectors */
        do {                      /* skip until good sector */
          while (mask & 1) {      /* skip bad sectors */
            ++sector_nr;
            mask >>= 1;
          }
          if (sector_nr >= 29) {
            if (++segment >= ftape_last_segment.id) {
              TRACEx1( 5, "segment out of range: %d", segment);
              result = -EIO;
              break;
            }
            mask = get_bad_sector_map_entry( segment);
            sector_nr = 0;
          }
        } while (mask & 1);
        ++sector_nr;                /* point to good sector */
        mask >>= 1;
      } while (--sector);
      if (result >= 0) {
        TRACEx2( 5, "putting filemark in segment %d at sector %d",
                segment, sector_nr);
        put_file_mark_in_map( segment, sector_nr);
        ++segment;                  /* next segment */
        sector_nr = 0;
        sector = 1;                 /* first sector */
      }
    }
  }
  TRACE_EXIT;
  return result;
}

int
ftape_erase( void)
{
  TRACE_FUN( 5, "ftape_erase");
  int result = 0;
  int i;
  unsigned long now = 0;
  byte* buffer = deblock_buffer;

  if (write_protected) {
    result = -EROFS;
  } else {
    result = read_header_segment( buffer);
    if (result >= 0) {
      TRACEx1( 5, "tape label: `%s'", (char*)(buffer + 30));
      if (strncmp( buffer + 30, LINUX_TAPE_LABEL, LINUX_TAPE_LABEL_LEN) != 0) {
        TRACE( 5, "invalid label, overwriting with new");
        memset( buffer + 30, ' ', 44);
        memcpy( buffer + 30, LINUX_TAPE_LABEL, sizeof( LINUX_TAPE_LABEL));
        PUT4( buffer, 74, now);
        for (i = 0; i < failed_sector_count; ++i) {
          unsigned failing_segment = failed_sector_log[ i].mark.segment;

          if (!valid_segment_no( failing_segment)) {
            TRACEi( 4, "bad entry in failed sector log:", failing_segment);
          } else {
            add_segment_to_bad_sector_map( failing_segment);
            TRACEx2( 4, "moved entry %d from failed sector log (%d)",
                    i, failing_segment);
          }
        }
      }
      /*  Clear failed sector log: remove all tape marks
       */
      memset( failed_sector_log, 0, sizeof( failed_sector_log));
      failed_sector_count = 0;
      failed_sector_log_changed = 1;
      result = ftape_update_header_segments( buffer, 1);
      prevent_flush();          /* prevent flush_buffers writing file marks */
    }
  }
  TRACE_EXIT;
  return result;
}

void
extract_file_marks( byte* address)
{
  TRACE_FUN( 8, "extract_file_marks");
  int i;

  memcpy( failed_sector_log, address + 256, sizeof( failed_sector_log));
  failed_sector_count = GET2( address, 144);
  TRACEi( 4, "number of file marks:", failed_sector_count);
  for (i = 0; i < failed_sector_count; ++i) {
    TRACEx2( 4, "eof mark in segment %d, sector %d",
            failed_sector_log[ i].mark.segment,
            failed_sector_log[ i].mark.sector);
  }
  reset_eof_list();
  TRACE_EXIT;
}

int
update_failed_sector_log( byte* buffer)
{
  if (failed_sector_log_changed) {
    memcpy( buffer + 256, failed_sector_log, sizeof( failed_sector_log));
    PUT2( buffer, 144, failed_sector_count);
    failed_sector_log_changed = 0;
    return 1;
  }
  return 0;
}

int
ftape_seek_eom( void)
{
  TRACE_FUN( 5, "ftape_seek_eom");
  int result = 0;
  unsigned eom;
  int i;

  if (first_data_segment == -1) {
    result = read_header_segment( deblock_buffer);
  }
  if (result >= 0) {
    eom = first_data_segment;
    /*  If fresh tape, count should be zero but we don't
     *  want to worry about the case it's one.
     */
    if (failed_sector_count > 1) {
      for (i = 1 ; i < failed_sector_count ; ++i) {
        /*  The eom is recorded as two eof marks in succeeding segments
         *  where the second one is always at segment number 1.
         */
        if (failed_sector_log[ i].mark.sector == 1) {
          if (failed_sector_log[ i].mark.segment == 
              failed_sector_log[ i - 1].mark.segment + 1) {
            eom = failed_sector_log[ i].mark.segment;
            break;
          }
        }
      }
    }
    ftape_seg_pos = eom;
    TRACEx1( 5, "eom found at segment %d", eom);
  }
  TRACE_EXIT;
  return result;
}

int
ftape_seek_eof( unsigned count)
{
  TRACE_FUN( 5, "ftape_seek_eof");
  int result = 0;
  int bad_seek = 0;
  int i;

  if (first_data_segment == -1) {
    result = read_header_segment( deblock_buffer);
  }
  TRACEx1( 5, "tape positioned at segment %d", ftape_seg_pos);
  if (result >= 0) {
    if (count != 0) {
      for (i = 0 ; i <= failed_sector_count ; ++i) {
        if (i >= failed_sector_count || /* start seeking after last mark */
            ftape_seg_pos <= failed_sector_log[ i].mark.segment) {
          i += count;
          if (i < 1) {          /* begin of tape */
            ftape_seg_pos = first_data_segment;
            if (i < 0) {        /* `before' begin of tape */
              bad_seek = 1;
            }
          } else if (i > failed_sector_count) { /* `after' end of tape */
            ftape_seg_pos = segments_per_track * tracks_per_tape;
            if (i > failed_sector_count + 1) {
              bad_seek = 2;
            }
          } else {              /* after requested file mark */
            ftape_seg_pos = failed_sector_log[ i - 1].mark.segment + 1;
          }
          break;
        }
      }
    }
  }
  if (bad_seek) {
    TRACEx1( 1, "seek reached %s of tape",
            (bad_seek == 1) ? "begin" : "end");
    result = -EIO;
  } else {
    TRACEx1( 5, "tape repositioned to segment %d", ftape_seg_pos);
  }
  TRACE_EXIT;
  return result;
}
