/*
 *	fs/msdos/mmap.c
 *
 *	Written by Jacques Gelinas (jacques@solucorp.qc.ca)
 *	Inspired by fs/nfs/mmap.c (Jaon Tombs 15 Aug 1993)
 *
 *	msdos mmap handling
 */
#include <linux/stat.h>
#include <linux/sched.h>
#include <linux/kernel.h>
#include <linux/mm.h>
#include <linux/shm.h>
#include <linux/errno.h>
#include <linux/mman.h>
#include <linux/string.h>
#include <linux/malloc.h>
#include <asm/segment.h>
#include <asm/system.h>
#include <linux/msdos_fs.h>

extern int share_page(struct vm_area_struct * area, struct task_struct * tsk,
	struct inode * inode, unsigned long address, unsigned long error_code,
	unsigned long newpage);

extern unsigned long put_page(struct task_struct * tsk,unsigned long page,
	unsigned long address,int prot);

extern void file_mmap_free(struct vm_area_struct * area);
extern int file_mmap_share(struct vm_area_struct * from, struct vm_area_struct * to,
				unsigned long address);

/*
	Perform the mapping of an adresse in memory
*/
static void msdos_file_mmap_nopage(
	int error_code,
	struct vm_area_struct * area,
	unsigned long address)
{
	struct inode * inode = area->vm_inode;
	unsigned int clear;
	unsigned long page;
	int pos;
	long gap;	/* distance from eof to pos */

	address &= PAGE_MASK;
	pos = address - area->vm_start + area->vm_offset;

	page = __get_free_page(GFP_KERNEL);
	if (share_page(area, area->vm_task, inode, address, error_code, page)) {
		++area->vm_task->mm->min_flt;
		return;
	}

	++area->vm_task->mm->maj_flt;
	if (!page) {
		oom(current);
		put_page(area->vm_task, BAD_PAGE, address, PAGE_PRIVATE);
		return;
	}

	clear = 0;
	gap = inode->i_size - pos;
	if (gap <= 0){
		/* mmaping beyong end of file */
		clear = PAGE_SIZE;
	}else{
		int cur_read;
		int need_read;
		struct file filp;
		if (gap < PAGE_SIZE){
			clear = PAGE_SIZE - gap;
		}
		filp.f_pos = pos;
		need_read = PAGE_SIZE - clear;
		cur_read = msdos_file_read_kmem (inode,&filp,(char*)page,need_read);
		if (cur_read != need_read){
			printk ("MSDOS: Error while reading an mmap file %d <> %d\n"
				,cur_read,need_read);
			free_page (page);
			oom(current);
			put_page(area->vm_task, BAD_PAGE, address, PAGE_PRIVATE);
			return;
		}
	}
	if (!(error_code & PAGE_RW)) {
		if (share_page(area, area->vm_task, inode, address, error_code, page))
			return;
	}
	if (clear > 0){
		#if 1
			memset ((char*)page+PAGE_SIZE-clear,0,clear);
		#else
			unsigned long tmp = page + PAGE_SIZE;
			while (clear--) {
				*(char *)--tmp = 0;
			}
		#endif
	}
	if (put_page(area->vm_task,page,address,area->vm_page_prot))
		return;
	free_page(page);
	oom(current);
}
struct vm_operations_struct msdos_file_mmap = {
	NULL,			/* open */
	file_mmap_free,		/* close */
	msdos_file_mmap_nopage,/* nopage */
	NULL,			/* wppage */
	file_mmap_share,	/* share */
	NULL,			/* unmap */
};
/*
	Initialise a mmap operation on a file.	
	Return 0 if ok, or a negative error code if not.
*/
int msdos_mmap(
	struct inode * inode,
	struct file * file,
	unsigned long addr,
	size_t len,
	int prot,
	unsigned long off)
{
	struct vm_area_struct * mpnt;

	if (prot & PAGE_RW)	/* only PAGE_COW or read-only supported now */
		return -EINVAL;
	if (off & (inode->i_sb->s_blocksize - 1))
		return -EINVAL;
	if (!inode->i_sb || !S_ISREG(inode->i_mode))
		return -EACCES;
	if (!IS_RDONLY(inode)) {
		inode->i_atime = CURRENT_TIME;
		inode->i_dirt = 1;
	}

	mpnt = (struct vm_area_struct * ) kmalloc(sizeof(struct vm_area_struct), GFP_KERNEL);
	if (!mpnt)
		return -ENOMEM;

	unmap_page_range(addr, len);
	mpnt->vm_task = current;
	mpnt->vm_start = addr;
	mpnt->vm_end = addr + len;
	mpnt->vm_page_prot = prot;
	mpnt->vm_share = NULL;
	mpnt->vm_inode = inode;
	inode->i_count++;
	mpnt->vm_offset = off;
	mpnt->vm_ops = &msdos_file_mmap;
	insert_vm_struct (current,mpnt);
	merge_segments (current->mm->mmap,NULL,NULL);
	return 0;
}
