/*
 * QUOTA    An implementation of the diskquota system for the LINUX operating
 *          system. QUOTA is implemented using the BSD systemcall interface as the
 *          means of communication with the user level. Should work for all
 *          filesystems because of integration into the VFS layer of the operating
 *          system. This is based on the Melbourne quota system wich uses both user
 *          and group quota files.
 * 
 *          Program to check disk quotas.
 * 
 * Authors: Marco van Wieringen <v892273@si.hhs.nl>
 *          Edvard Tuinder <v892231@si.hhs.nl>
 * 
 *          This program is free software; you can redistribute it and/or modify it under
 *          the terms of the GNU General Public License as published by the Free
 *          Software Foundation; either version 2 of the License, or (at your option)
 *          any later version.
 */

#include <sys/types.h>
#include <sys/file.h>
#include <sys/param.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <stdarg.h>
#include <dirent.h>
#include <pwd.h>
#include <grp.h>
#include <limits.h>
#include <stdio.h>
#include <unistd.h>
#include <string.h>
#include <getopt.h>
#include <mntent.h>
#define __KERNEL__
#include <linux/quota.h>
#undef __KERNEL__

#ifndef lint
static char RCS_checkquota[] = "$Id: quotacheck.c,v 1.6 1993/06/16 17:07:47 root Exp root $";
#endif

struct dirs {
   char *dirname;
   struct dirs *next;
};

struct dlinks {
   ino_t ino;
   size_t size;
   size_t blksize;
   u_long id;
   struct dlinks *next;
};

static char bits[] = "|/-\\";

dev_t cur_dev;
char dflag = 0, vflag = 0;
char check_usr, check_grp;

struct dquot *dquot_list[MAXQUOTAS] = {(struct dquot *)0, (struct dquot *)0};
struct dquot *mru_dquot[MAXQUOTAS] = {(struct dquot *)0, (struct dquot *)0};
struct dirs *dirs_root, *cur_dirs;
struct dlinks *stored_links[MAXQUOTAS] = {(struct dlinks *)0, (struct dlinks *)0};

long dir_to_do, dir_done, files_done;
char *qfname = QUOTAFILENAME;
char *qfextension[] = INITQFNAMES;

char cwd[MAXPATHLEN + 1];
short bitc = 0;

static inline char *typetoname(int type)
{
   switch (type) {
      case USRQUOTA:
         return "uid";
      case GRPQUOTA:
         return "gid";
      default:
   }
}

/*
 * This simple algorithm calculates the size of a file in blocks.
 * It is not perfect but works most of the time.
 */
u_long isize_to_blocks(size_t isize, size_t blksize)
{
   u_long blocks;
   u_long indirect;

   if (!blksize)
     blksize = BLOCK_SIZE;

   blocks = (isize / blksize) + ((isize % blksize) ? 1 : 0);
   if (blocks > 10) {
      indirect = (blocks - 11) / 256 + 1; /* single indirect blocks */
      if (blocks > (10 + 256)) {
         indirect += (blocks - 267) / (256 * 256) + 1; /* double indirect blocks */
         if (blocks > (10 + 256 + (256 * 256)))
            indirect++; /* triple indirect blocks */
      }
      blocks += indirect;
   }
   return blocks;
}

void abort_now(const char *perror_mes, const char *fmt,...)
{
   va_list args;

   va_start(args, fmt);
   vfprintf(stderr, fmt, args);
   va_end(args);

   fflush(stderr);
   perror(perror_mes);
   exit(-1);
}

void store_dlinks(struct stat * st, int type)
{
   struct dlinks *lptr;

   for (lptr = stored_links[type]; lptr != (struct dlinks *) 0; lptr = lptr->next)
      if (lptr->ino == st->st_ino)
         return;

   lptr = (struct dlinks *) calloc(sizeof(struct dlinks), sizeof(void *));
   if (lptr == (struct dlinks *) 0)
      abort_now("calloc", "store_dlinks(\"%s\"): %d: ", cwd, st->st_ino);

   switch (type) {
      case USRQUOTA:
         lptr->id = st->st_uid;
         break;
      case GRPQUOTA:
         lptr->id = st->st_gid;
         break;
      default:
   }
   lptr->ino = st->st_ino;
   lptr->size = st->st_size;
   lptr->blksize = st->st_blksize;

   lptr->next = stored_links[type];
   stored_links[type] = lptr;
}

struct dquot *lookup_dquot(int id, int type)
{
   register struct dquot *lptr = (struct dquot *) 0;

   /*
    * First fast lookup when same as used before.
    */
   if (mru_dquot[type] != (struct dquot *) 0 && mru_dquot[type]->dq_id == id)
      return (mru_dquot[type]);

   for (lptr = dquot_list[type]; lptr != (struct dquot *) 0; lptr = lptr->next)
      if (lptr->dq_id == id) {
         mru_dquot[type] = lptr;
         return (lptr);
      }
   return ((struct dquot *) 0);
}

struct dquot *add_dquot(int id, int type)
{
   register struct dquot *lptr;

   if (dflag)
      fprintf(stderr, "Adding dquot for %s %d\n", typetoname(type), id);

   if ((lptr = (struct dquot *) calloc(sizeof(struct dquot), 1)) == (struct dquot *) 0)
      abort_now("calloc", "add_dquot(%d): allocating space for new %s\n", id, typetoname(type));

   lptr->dq_id = id;
   lptr->next = dquot_list[type];
   dquot_list[type] = lptr;
   lptr->dq_btime = lptr->dq_itime = (time_t) 0;

   return (lptr);
}

void add_to_quota(struct stat * st, int type)
{
   int wanted;
   struct dquot *lptr;

   if (st->st_nlink != 1 && !S_ISDIR(st->st_mode)) {
      store_dlinks(st, type);
      return;
   }
   switch (type) {
      case USRQUOTA:
         wanted = st->st_uid;
         break;
      case GRPQUOTA:
         wanted = st->st_gid;
         break;
      default:
   }

   if ((lptr = lookup_dquot(wanted, type)) == (struct dquot *) 0)
      if ((lptr = add_dquot(wanted, type)) == (struct dquot *) 0)
         abort_now("add_to_quota", "Can't add %s %d !\n", typetoname(type), wanted);

   lptr->dq_curinodes++;
   if (st->st_size)
      lptr->dq_curblocks += isize_to_blocks(st->st_size, st->st_blksize);
}

static inline void blit()
{
   putc(bits[bitc], stderr);
   putc('\b', stderr);
   fflush(stderr);
   bitc++;
   if (bitc == (sizeof(bits) - 1))
      bitc = 0;
}

void store_directory(struct dirent * d_info)
{
   if (dflag)
      fprintf(stderr, "Storing `%s' in directory list\n", d_info->d_name);

   cur_dirs->dirname = (char *) calloc(strlen(d_info->d_name) + strlen(cwd) + 4, sizeof(char));
   if (cur_dirs->dirname == (char *) 0)
      abort_now("calloc", "store_directory(\"%s\"): %s: ", cwd, d_info->d_name);

   if (strcmp(cwd, "/"))
      sprintf(cur_dirs->dirname, "%s/%s", cwd, d_info->d_name);
   else {
      strcpy(cur_dirs->dirname, "/");
      strcat(cur_dirs->dirname, d_info->d_name);
   }

   cur_dirs->next = (struct dirs *) calloc(sizeof(struct dirs), sizeof(void *));
   if (cur_dirs->next == (struct dirs *) 0)
      abort_now("calloc", "store_directory(\"%s\"): cur_dirs->next: ", cwd);

   cur_dirs = cur_dirs->next;
   dir_to_do++;
   dir_done++;
}

short read_dir_entry(DIR * dp)
{
   register struct dirent *dir;
   struct stat st;

   dir = readdir(dp);
   if (dir == (struct dirent *) 0)
      return 0;

   if ((lstat(dir->d_name, &st)) == -1)
      abort_now("stat", "read_dir_entry(\"%s\") %s: ", cwd, dir->d_name);

   if (S_ISDIR(st.st_mode)) {
      if (st.st_dev == cur_dev) {
         if ((strcmp(dir->d_name, ".")) && (strcmp(dir->d_name, ".."))) {
            store_directory(dir);
            if (check_usr && st.st_uid > ID_NO_QUOTA)
               add_to_quota(&st, USRQUOTA);
            if (check_grp && st.st_gid > ID_NO_QUOTA)
               add_to_quota(&st, GRPQUOTA);
         }
      }
      return read_dir_entry(dp);
   } else {
      if (dflag)
         fprintf(stderr, "Found `%s' in directory `%s'\n", dir->d_name, cwd);

      if (st.st_nlink == 1)
         files_done++;
      if (check_usr && st.st_uid > ID_NO_QUOTA)
         add_to_quota(&st, USRQUOTA);
      if (check_grp && st.st_gid > ID_NO_QUOTA)
         add_to_quota(&st, GRPQUOTA);
   }
   return 1; /* more to be found... */
}

void add_dlinks(u_long * inode, u_long * block, int id, int type)
{
   struct dlinks  *lptr, *done;

   if (dflag)
      fprintf(stderr, "Adding blocks from hardlinks for %s %d\n", typetoname(type), id);

   for (lptr = stored_links[type]; lptr != (struct dlinks *) 0; lptr = lptr->next) {
      if (lptr->id == id) {
         (*inode)++;
         if (lptr->size)
            *block += isize_to_blocks(lptr->size, lptr->blksize);
         files_done++;
      }
   }
}

void clean_list()
{
   int cnt;
   struct dquot *dquot, *dquot_free;
   struct dlinks *dlink, *dlink_free;

   for (cnt = 0; cnt < MAXQUOTAS; cnt++) {
      if (dquot_list[cnt] != (struct dquot *)0) {
         dquot = dquot_list[cnt];
         while (dquot != (struct dquot *)0) {
            dquot_free = dquot;
            dquot = dquot->next;
            free(dquot_free);
         }
         mru_dquot[cnt] = (struct dquot *)0;
      }
      dquot_list[cnt] = (struct dquot *)0;
      if (stored_links[cnt] != (struct dlinks *)0) {
         dlink = stored_links[cnt];
         while (dlink != (struct dlinks *)0) {
            dlink_free = dlink;
            dlink = dlink->next;
            free(dlink_free);
         }
      }
      stored_links[cnt] = (struct dlinks *)0;
   }
}

void init_list()
{
   cur_dirs = (struct dirs *)calloc(sizeof(struct dirs), sizeof(void *));
   if (cur_dirs == (struct dirs *)0)
      abort_now ("calloc", "store_directory(\"%s\"): cur_dirs: ", cwd);
   dirs_root = cur_dirs;
   dir_to_do = dir_done = files_done = 0;
}

void read_dir(const char *dir_name)
{
   register DIR *dp;
   struct stat st;

   if (dir_name == (char *) 0) {
      fprintf(stderr, "Got empty directory name in read_dir()\n");
      fprintf(stderr, "dir_to_do = %d\n", dir_to_do);
      return;
   }
   if (dflag)
      fprintf(stderr, "read_dir(%s)\n", dir_name);

   chdir(dir_name);   /* pretty darn important ! */

   bzero(cwd, MAXPATHLEN);
   getwd(cwd);

   if (((lstat(dir_name, &st)) == -1) || (!S_ISDIR(st.st_mode)))
      abort_now("stat", "read_dir(\"%s\"): ", dir_name);

   dp = opendir(dir_name);
   if (dp == (DIR *) 0)
      abort_now("opendir", "read_dir(\"%s\"): ", dir_name);

   while (read_dir_entry(dp));
   closedir(dp);
   chdir("..");
}

void scan_dir(const char *devicename, const char *pathname)
{
   register struct dirs *tmp, *to_free;
   struct stat st;

   if ((stat(pathname, &st)) == -1)
      abort_now("stat", "scan_dir(\"%s\",\"%s\"): ", devicename, pathname);

   chdir(pathname);

   if (vflag) {
      fprintf(stderr, "Scanning filesystem: %s (%s) ", devicename, pathname);
      fflush(stderr);
   }

   memset((caddr_t *)cwd, 0, MAXPATHLEN);
   getwd(cwd);
   cur_dev = st.st_dev;

   if (check_usr && st.st_uid > ID_NO_QUOTA)
      add_to_quota(&st, USRQUOTA);   /* root dir */

   if (check_grp && st.st_gid > ID_NO_QUOTA)
      add_to_quota(&st, GRPQUOTA);   /* root dir */

   read_dir(pathname);   /* root dir */

   dir_done++;

   if (dir_to_do) { /* scan dirs found in root dir */
      for (tmp = dirs_root; dir_to_do; dir_to_do--) {
         read_dir(tmp->dirname);
         free(tmp->dirname);
         to_free = tmp;
         tmp = tmp->next;
         free(to_free);
         if (vflag && !dflag)
            blit();
      }
   }
   if (vflag)
      fprintf(stderr, "done\nChecked %d directories and %d files\n", dir_done, files_done);
}

int getmax_ent(int type)
{
   int high = 0;
   struct passwd *pwd;
   struct group *grp;

   if (dflag) {
      fprintf(stderr, "Searching for highest %s: ", typetoname(type));
      fflush(stderr);
   }
   switch (type) {
      case USRQUOTA:
         pwd = getpwent();
         while (pwd != (struct passwd *) 0) {
            if (pwd->pw_uid > high)
               high = pwd->pw_uid;
            if (high == USHRT_MAX)   /* nobody = -1 -> would generate 65535 quota fields.. */
               high = 0;
            pwd = getpwent();
         }
         endpwent();
         break;
      case GRPQUOTA:
         grp = getgrent();
         while (grp != (struct group *) 0) {
            if (grp->gr_gid > high)
               high = grp->gr_gid;
            if (high == USHRT_MAX)   /* nobody = -1 -> would generate 65535 quota fields.. */
               high = 0;
            grp = getgrent();
         }
         endgrent();
         break;
      default:
   }

   if (dflag)
      fprintf(stderr, "%d\n", high);
   return high;
}

void dump_to_file(char *fsname, char *quotafile, int type)
{
   struct dqblk dq_dqb;
   struct dquot *dquot;
   int quota_enabled = 0, max_id;
   int fd, id = 0;

   if (vflag)
      fprintf(stderr, "Writing result to %s\n", quotafile);

   if (quotactl(QCMD(Q_GETQUOTA, type), fsname, 0, &dq_dqb) == 0)
      quota_enabled = 1;

   if (vflag && quota_enabled)
      fprintf(stderr, "Updating in core quotas\n");

   if ((fd = open(quotafile, O_RDWR | O_CREAT, 0600)) < 0)
      abort_now("open", "dump_to_file(%s)\n", quotafile);

   if (flock(fd, LOCK_EX) < 0)
      abort_now("flock", "dump_to_file(%s)\n", quotafile);

   max_id = getmax_ent(type);
   while (id <= max_id) {
      if (lseek(fd, dqoff(id), SEEK_SET))
         read(fd, &dq_dqb, sizeof(struct dqblk));
      if ((dquot = lookup_dquot(id, type)) != (struct dquot *) 0) {
         dq_curinodes = dquot->dq_curinodes;
         dq_curblocks = dquot->dq_curblocks;
         if (dflag)
            fprintf(stderr, "%s: %d curinodes: %d curblocks: %d\n", typetoname(type),
               id, dq_curinodes, dq_curblocks);
         add_dlinks(&dq_curinodes, &dq_curblocks, id, type);
         if (dflag)
            fprintf(stderr, "%s: %d curinodes: %d curblocks: %d\n", typetoname(type),
               id, dq_curinodes, dq_curblocks);
      } else
         memset((caddr_t *)&dq_dqb, 0, sizeof(struct dqblk));

      if (quota_enabled)
         quotactl(QCMD(Q_SETUSE, type), fsname, id, &dq_dqb);

      if (lseek(fd, dqoff(id), SEEK_SET))
         write(fd, &dq_dqb, sizeof(struct dqblk));
      id++;
   }
   flock(fd, LOCK_UN);
   close(fd);
}

void usage()
{
   fprintf(stderr, "Usage:\n\tquotacheck [-g] [-u] [-vd] -a\n");
   fprintf(stderr, "\tquotacheck [-g] [-u] [-vd] filesys ...\n");
   exit(1);
}

/*
 * Check to see if target appears in list of size cnt.
 */
oneof(char *target, char *list[], int cnt)
{
   register int i;

   for (i = 0; i < cnt; i++)
      if (strcmp(target, list[i]) == 0)
         return (i);
   return (-1);
}

#define CORRECT_FSTYPE(type) (!strcmp(type,MNTTYPE_EXT) || !strcmp(type,MNTTYPE_EXT2) || \
!strcmp(type,MNTTYPE_XIAFS) || !strcmp(type,MNTTYPE_MINIX))

/*
 * Check to see if a particular quota is to be enabled.
 */
hasquota(struct mntent *mnt, int type, char **qfnamep)
{
   char *buf;

   if (!CORRECT_FSTYPE(mnt->mnt_type))
      return (0);

   if ((type == USRQUOTA) && hasmntopt(mnt, MNTOPT_USRQUOTA) ||
       (type == GRPQUOTA) && hasmntopt(mnt, MNTOPT_GRPQUOTA)) {
      buf = (char *) calloc(BUFSIZ, 1);
      (void) sprintf(buf, "%s/%s.%s", mnt->mnt_dir, qfname, qfextension[type]);
      *qfnamep = buf;
      return (1);
   } else
      return (0);
}

void main(int argc, char **argv)
{
   FILE *fp;
   int cnt, ch, errs;
   char aflag = 0, gflag = 0, uflag = 0;
   long argnum, done;
   char *usr_qfnp, *grp_qfnp;
   register struct mntent *mnt;

   while ((ch = getopt(argc, argv, "avugd")) != EOF) {
      switch (ch) {
         case 'a':
            aflag++;
            break;
         case 'g':
            gflag++;
            break;
         case 'u':
            uflag++;
            break;
         case 'd':
            dflag++;
         case 'v':
            vflag++;
            break;
         default:
            usage();
      }
   }
   argc -= optind;
   argv += optind;

   if (!uflag && !gflag)
      uflag++;

   if (!aflag && argc == 0)
      usage();

   fp = setmntent(FSTAB, "r");
   while ((mnt = getmntent(fp)) != (struct mntent *) 0) {
      check_usr = check_grp = 0;
      if (argc && (argnum = oneof(mnt->mnt_dir, argv, argc)) >= 0 ||
                    (argnum = oneof(mnt->mnt_fsname, argv, argc)) >= 0) {
          done |= 1 << argnum;
      } else
         if (!aflag || hasmntopt(mnt, MNTOPT_NOAUTO))
            continue;
      if (gflag && hasquota(mnt, GRPQUOTA, &grp_qfnp))
         check_grp++;
      if (uflag && hasquota(mnt, USRQUOTA, &usr_qfnp))
         check_usr++;
      if (check_usr || check_grp) {
         init_list();
         scan_dir(mnt->mnt_fsname, mnt->mnt_dir);
         if (check_usr)
            dump_to_file(mnt->mnt_fsname, usr_qfnp, USRQUOTA);
         if (check_grp)
            dump_to_file(mnt->mnt_fsname, grp_qfnp, GRPQUOTA);
         clean_list();
      }
   }
   endmntent(fp);

   for (cnt = 0; cnt < argc; cnt++)
      if ((done & (1 << cnt)) == 0)
         fprintf(stderr, "%s not found in fstab\n",
            argv[cnt]);

   exit(0);
}
