import {
  AbsoluteTime,
  assertUnreachable,
  Challenge,
  ChallengeRequestResponse,
  ChallengeResponse,
  Duration,
  HttpStatusCode,
  TalerErrorCode,
  TanChannel,
} from "@gnu-taler/taler-util";
import {
  Time,
  undefinedIfEmpty,
  useTranslationContext,
} from "@gnu-taler/web-util/browser";
import { Fragment, h, VNode } from "preact";
import { useEffect, useState } from "preact/hooks";
import { useSessionContext } from "../context/session.js";
import { Notification } from "../utils/types.js";
import { AsyncButton } from "./exception/AsyncButton.js";
import { FormErrors, FormProvider } from "./form/FormProvider.js";
import { Input } from "./form/Input.js";
import { NotificationCard } from "./menu/index.js";
import {
  datetimeFormatForSettings,
  usePreference,
} from "../hooks/preference.js";
import { format } from "date-fns";

export interface Props {
  onCompleted(challenges: string[]): void;
  onCancel(): void;
  currentChallenge: ChallengeResponse;
}

interface Form {
  code: string;
}

function SolveChallenge({
  challenge,
  expiration,
  onCancel,
  onSolved,
}: {
  onCancel: () => void;
  challenge: Challenge;
  expiration: AbsoluteTime;
  onSolved: () => void;
}): VNode {
  const { i18n } = useTranslationContext();
  const { state: session, lib, logIn } = useSessionContext();
  const [notif, setNotif] = useState<Notification | undefined>(undefined);
  const [value, setValue] = useState<Partial<Form>>({});

  const [showExpired, setExpired] = useState(
    expiration !== undefined && AbsoluteTime.isExpired(expiration),
  );
  const [settings] = usePreference();

  useEffect(() => {
    if (showExpired) return;
    const remain = AbsoluteTime.remaining(expiration).d_ms;
    if (remain === "forever") return;
    const handler = setTimeout(() => {
      setExpired(true);
    }, remain);
    return () => {
      clearTimeout(handler);
    };
  }, []);

  const errors = undefinedIfEmpty<FormErrors<Form>>({
    code: showExpired
      ? i18n.str`Expired`
      : !value.code
        ? i18n.str`Required`
        : undefined,
  });
  function valueHandler(s: (d: Partial<Form>) => Partial<Form>): void {
    const next = s(value);
    const v: Form = {
      code: next.code ?? "",
    };
    setValue(v);
  }

  async function doVerificationImpl() {
    try {
      const resp = await lib.instance.confirmChallenge(challenge.challenge_id, {
        tan: value.code!,
      });
      if (resp.case === "ok") {
        return onSolved();
      }
      switch (resp.case) {
        case HttpStatusCode.Unauthorized: {
          setNotif({
            message: i18n.str`Failed to validate the verification code.`,
            type: "ERROR",
            description: resp.detail?.hint,
          });
          return;
        }
        case TalerErrorCode.MERCHANT_TAN_CHALLENGE_FAILED: {
          setNotif({
            message: i18n.str`Failed to validate the verification code.`,
            type: "ERROR",
            description: resp.detail?.hint,
          });
          return;
        }
        case TalerErrorCode.MERCHANT_TAN_CHALLENGE_UNKNOWN: {
          setNotif({
            message: i18n.str`Failed to validate the verification code.`,
            type: "ERROR",
            description: resp.detail?.hint,
          });
          return;
        }
        case TalerErrorCode.MERCHANT_TAN_TOO_MANY_ATTEMPTS: {
          setNotif({
            message: i18n.str`Failed to validate the verification code.`,
            type: "ERROR",
            description: resp.detail?.hint,
          });
          return;
        }
        default: {
          assertUnreachable(resp);
        }
      }
    } catch (error) {
      setNotif({
        message: i18n.str`Failed to verify code`,
        type: "ERROR",
        description: error instanceof Error ? error.message : String(error),
      });
    }
  }

  return (
    <Fragment>
      <NotificationCard notification={notif} />

      <div class="columns is-centered" style={{ margin: "auto" }}>
        <div class="column is-two-thirds ">
          <div class="modal-card" style={{ width: "100%", margin: 0 }}>
            <header
              class="modal-card-head"
              style={{ border: "1px solid", borderBottom: 0 }}
            >
              <p class="modal-card-title">
                <i18n.Translate>Validation code sent.</i18n.Translate>
              </p>
            </header>
            <section
              class="modal-card-body"
              style={{ border: "1px solid", borderTop: 0, borderBottom: 0 }}
            >
              {(function (): VNode {
                switch (challenge.tan_channel) {
                  case TanChannel.SMS:
                    return (
                      <i18n.Translate>
                        The verification code sent to the phone number starting
                        with "<b>{challenge.tan_info}</b>"
                      </i18n.Translate>
                    );
                  case TanChannel.EMAIL:
                    return (
                      <i18n.Translate>
                        The verification code sent to the email address starting
                        with "<b>{challenge.tan_info}</b>"
                      </i18n.Translate>
                    );
                }
              })()}
              <FormProvider<Form>
                name="settings"
                errors={errors}
                object={value}
                valueHandler={valueHandler}
              >
                <Input<Form>
                  label={i18n.str`Verification code`}
                  name="code"
                  readonly={showExpired}
                />
              </FormProvider>
              {expiration.t_ms === "never" ? undefined : (
                <p>
                  <i18n.Translate>
                    It will expired at{" "}
                    {format(
                      expiration.t_ms,
                      datetimeFormatForSettings(settings),
                    )}
                  </i18n.Translate>
                </p>
              )}
              {showExpired ? (
                <p>
                  <i18n.Translate>
                    The challenge is expired and can't be solved but you can go
                    back and create a new challenge.
                  </i18n.Translate>
                </p>
              ) : undefined}
            </section>
            <footer
              class="modal-card-foot "
              style={{
                justifyContent: "space-between",
                border: "1px solid",
                borderTop: 0,
              }}
            >
              <button class="button" onClick={onCancel}>
                <i18n.Translate>Back</i18n.Translate>
              </button>
              <AsyncButton
                type="is-info"
                disabled={errors !== undefined}
                onClick={doVerificationImpl}
              >
                <i18n.Translate>Verify</i18n.Translate>
              </AsyncButton>
            </footer>
          </div>
        </div>
      </div>
    </Fragment>
  );
}

export function SolveMFAChallenges({
  currentChallenge,
  onCompleted,
  onCancel,
}: Props): VNode {
  const { i18n } = useTranslationContext();
  const { state: session, lib, logIn } = useSessionContext();
  const [notif, setNotif] = useState<Notification | undefined>(undefined);
  const [solved, setSolved] = useState<string[]>([]);
  // FIXME: we should save here also the expiration of the
  // tan channel to be used when the user press "i have the code"
  const [retransmission, setRetransmission] = useState<
    Record<TanChannel, AbsoluteTime>
  >({
    email: AbsoluteTime.now(),
    sms: AbsoluteTime.now(),
  });

  const [selected, setSelected] = useState<{
    ch: Challenge;
    expiration: AbsoluteTime;
  }>();
  const [settings] = usePreference();

  if (selected) {
    return (
      <SolveChallenge
        onCancel={() => setSelected(undefined)}
        challenge={selected.ch}
        expiration={selected.expiration}
        onSolved={() => {
          setSelected(undefined);
          const newSolved = [...solved, selected.ch.challenge_id];

          const done = currentChallenge.combi_and
            ? newSolved.length === currentChallenge.challenges.length
            : newSolved.length > 0;

          if (done) {
            onCompleted(newSolved);
          } else {
            setSolved(newSolved);
          }
        }}
      />
    );
  }

  const hasSolvedEnough = currentChallenge.combi_and
    ? solved.length === currentChallenge.challenges.length
    : solved.length > 0;

  async function doSendCodeImpl(ch: Challenge) {
    try {
      const resp = await lib.instance.sendChallenge(ch.challenge_id);
      if (resp.case === "ok") {
        if (resp.body.earliest_retransmission) {
          setRetransmission({
            ...retransmission,
            [ch.tan_channel]: AbsoluteTime.fromProtocolTimestamp(
              resp.body.earliest_retransmission,
            ),
          });
        }
        return setSelected({
          ch,
          expiration: !resp.body.solve_expiration
            ? AbsoluteTime.never()
            : AbsoluteTime.fromProtocolTimestamp(resp.body.solve_expiration),
        });
      }
      switch (resp.case) {
        case HttpStatusCode.Unauthorized: {
          setNotif({
            message: i18n.str`Failed to send the verification code.`,
            type: "ERROR",
            description: resp.detail?.hint,
          });
          return;
        }
        case HttpStatusCode.Forbidden: {
          setNotif({
            message: i18n.str`The request was valid, but the server is refusing action.`,
            type: "ERROR",
            description: resp.detail?.hint,
          });
          return;
        }
        case TalerErrorCode.MERCHANT_TAN_CHALLENGE_UNKNOWN: {
          setNotif({
            message: i18n.str`The backend is not aware of the specified MFA challenge.`,
            type: "ERROR",
            description: resp.detail?.hint,
          });
          return;
        }
        case TalerErrorCode.MERCHANT_TAN_MFA_HELPER_EXEC_FAILED: {
          setNotif({
            message: i18n.str`The backend failed to launch a helper process required for the multi-factor authentication step.`,
            type: "ERROR",
            description: resp.detail?.hint,
          });
          return;
        }
        case TalerErrorCode.MERCHANT_TAN_CHALLENGE_SOLVED: {
          setNotif({
            message: i18n.str`The challenge was already solved.`,
            type: "ERROR",
            description: resp.detail?.hint,
          });
          return;
        }
        case TalerErrorCode.MERCHANT_TAN_TOO_EARLY: {
          setNotif({
            message: i18n.str`It is too early to request another transmission of the challenge.`,
            type: "ERROR",
            description: resp.detail?.hint,
          });
          return;
        }
        default: {
          assertUnreachable(resp);
        }
      }
    } catch (error) {
      setNotif({
        message: i18n.str`Failed to send the verification code.`,
        type: "ERROR",
        description: error instanceof Error ? error.message : String(error),
      });
    }
  }

  return (
    <Fragment>
      <NotificationCard notification={notif} />

      <div class="columns is-centered" style={{ margin: "auto" }}>
        <div class="column is-two-thirds ">
          <div class="modal-card" style={{ width: "100%", margin: 0 }}>
            <header
              class="modal-card-head"
              style={{ border: "1px solid", borderBottom: 0 }}
            >
              <p class="modal-card-title">
                <i18n.Translate>
                  Multi-factor authentication required.
                </i18n.Translate>
              </p>
            </header>
            <section
              class="modal-card-body"
              style={{ border: "1px solid", borderTop: 0, borderBottom: 0 }}
            >
              {currentChallenge.combi_and ? (
                <i18n.Translate>
                  You need to complete all of this requirements.
                </i18n.Translate>
              ) : (
                <i18n.Translate>
                  You need to complete at least one of this requirements.
                </i18n.Translate>
              )}
            </section>
            {currentChallenge.challenges.map((d) => {
              const time = retransmission[d.tan_channel];
              const alreadySent = !AbsoluteTime.isExpired(time);

              return (
                <section
                  class="modal-card-body"
                  style={{ border: "1px solid", borderTop: 0, borderBottom: 0 }}
                >
                  {(function (): VNode {
                    switch (d.tan_channel) {
                      case TanChannel.SMS:
                        return (
                          <i18n.Translate>
                            An SMS to the phone number starting with{" "}
                            {d.tan_info}
                          </i18n.Translate>
                        );
                      case TanChannel.EMAIL:
                        return (
                          <i18n.Translate>
                            An email to the address starting with {d.tan_info}
                          </i18n.Translate>
                        );
                    }
                  })()}

                  {alreadySent && time.t_ms !== "never" ? (
                    <p>
                      <i18n.Translate>
                        You have to wait until{" "}
                        {format(time.t_ms, datetimeFormatForSettings(settings))}
                        to send a new code.
                      </i18n.Translate>
                    </p>
                  ) : undefined}
                  <div
                    style={{
                      justifyContent: "space-between",
                      display: "flex",
                    }}
                  >
                    <button
                      disabled={
                        hasSolvedEnough || solved.indexOf(d.challenge_id) !== -1
                      }
                      class="button"
                      onClick={() => {
                        setSelected({
                          ch: d,
                          expiration: AbsoluteTime.never(),
                        });
                      }}
                    >
                      <i18n.Translate>I have a code</i18n.Translate>
                    </button>
                    <AsyncButton
                      disabled={
                        hasSolvedEnough ||
                        solved.indexOf(d.challenge_id) !== -1 ||
                        alreadySent
                      }
                      onClick={() => doSendCodeImpl(d)}
                    >
                      <i18n.Translate>Send me a message</i18n.Translate>
                    </AsyncButton>
                  </div>
                </section>
              );
            })}
            <footer
              class="modal-card-foot "
              style={{
                justifyContent: "space-between",
                border: "1px solid",
                borderTop: 0,
              }}
            >
              <button class="button" onClick={onCancel}>
                <i18n.Translate>Cancel</i18n.Translate>
              </button>
              <AsyncButton
                type="is-info"
                disabled={!hasSolvedEnough}
                onClick={async () => onCompleted(solved)}
              >
                <i18n.Translate>Complete</i18n.Translate>
              </AsyncButton>
            </footer>
          </div>
        </div>
      </div>
    </Fragment>
  );
}
