/**********************************************************************
* Name:
*        dapibkup.c
*
* Function:
*        Functions invoked via menu selections
*
* Description:
*        The file contains the functions that are invoked from the
*        menu processing code and tables.
***********************************************************************/
#include <ctype.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "dsmapitd.h"      /* Standard Tivoli Storage Manager API    */
#include "dsmapifp.h"
#include "dsmrc.h"         /* RC defs.                               */
#include "dapitype.h"      /* Sample API data types.                 */
#include "dapint64.h"      /* 64 bit routines.                       */
#include "dapiutil.h"      /* Handy utility routines.                */
#include "dapiproc.h"

/*----------------------------------------------------------------------+
| Private routines
.----------------------------------------------------------------------*/
static dsInt16_t ourBeginTxn(dsUint32_t  dsmHandle,    /* Tivoli Storage Manager API sess handle  */
                         dsUint32_t  *txn_objs,        /* Objects in this txn. */
                         llist   *txn_llist_h,     /* Linked list head.    */
                         llist   *txn_llist_t,     /* Linked list tail.    */
                         char    *last_copy_dest);
static dsInt16_t register_fs(dsUint32_t dsmHandle,regFSData *fsData);
/*----------------------------------------------------------------------+
| Public routines
.----------------------------------------------------------------------*/

/*----------------------------------------------------------------------+
| Name:    perform_backup()
|
| Action:  Take the information passed in the dialog to this function,
|          build the backup objects dynamically, and send them to the
|          server.
|
| Input:   sel_dialog  - Pointer to dialog table to process.
|          send_type   - Indicates whether archive or backup being done
|
| Returns: RC_OK             - Successful
|          RC_NOT_IN_SESSION - Client is not in session with the server
|          RC_SESSION_FAILED - A dsm api call failed stopping execution
|
| Side
| Effects: None
|
| Notes:   This routine handles 64bit objects and number of objects.  Thus
|          the number of files to be sent and the size of the files to be
|          sent are 64bit numbers.  Memory buffers are still 32bit values.
|          Be very careful changing the numbers to insure a type mismatch
|          does not occur.
+----------------------------------------------------------------------*/
dsInt16_t perform_send(dialog *sel_dialog, dsUint16_t send_type)

{
/*----------------------------------------------------------------------
| Note this procedure stack is getting quite big... may need to malloc
| the larger structures to keep it down in size for smaller systems.
.---------------------------------------------------------------------*/

int                     x;
char                    *seed, *desc, *end_of_buff;
char                    *bkup_buff=NULL, *cursor;
char                    t[50], t2[50], t3[50], t4[50], t5[50];
char                    last_copy_dest[DSM_MAX_CG_DEST_LENGTH + 1];
char                    *ll;
char                    *temp_string;
char                    smpAPI[20];
char                    seed_count[15] = "0";
char                    mcOverride[DSM_MAX_MC_NAME_LENGTH];  /* new mgmt class override */
dsBool_t                new_txn_needed = bFalse;
dsBool_t                in_txn = bFalse;
dsBool_t                done = bFalse;
dsBool_t                mountWait = bFalse;
dsInt16_t               rc;
dsUint16_t              txn_reason;
dsUint32_t              seed_size, desc_size;
dsUint32_t              txn_objs;
dsUint32_t              i;
dsUint32_t              buff_size, send_amt;
dsStruct64_t            t64;
dsStruct64_t            filesize64;
dsStruct64_t            numfiles64, files64, count64;
dsStruct64_t            buff_size64;
dsStruct64_t            bytes_left64;
dsStruct64_t            buff_num64;
dsmObjName              objName;   /* Big object for some compiler stacks......*/
mcBindKey               MCBindKey;
dialog                  *dlg;
llist                   *txn_llist_h = NULL, *txn_llist_t = NULL;
regFSData               fsData;
ObjAttr                 objAttr;
DataBlk                 dataBlk;
sndArchiveData          archData;
smpObjInfo              *objInfo=NULL, *tobjInfo;
dsmEndSendObjExIn_t     endSendObjExIn;
dsmEndSendObjExOut_t    endSendObjExOut;
const char              *validChars = "ABCDEFGHIJKLMNOPQRSTUVWXYZ";

strcpy(smpAPI, "Sample");

if (!dsmHandle)
{
   printf("You must signon before performing this action.\n");
   return(0);
}

memset(&objName,0x00,sizeof(objName));
memset(&objAttr,0x00,sizeof(objAttr));

/*----------------------------------------------------------------------
| First pull all values out of the passed dialog for our use.
.----------------------------------------------------------------------*/
i = 0;
dlg = &sel_dialog[i];
while (dlg->item_type != DSMAPI_END)
{
   switch (dlg->item_type)
   {
      case DSMAPI_FILESPACE :
         strcpy(objName.fs,dlg->item_buff);
         break;
      case DSMAPI_HIGHLEVEL :
         strcpy(objName.hl,dlg->item_buff);
         break;
      case DSMAPI_LOWLEVEL :
         ll = dlg->item_buff;
         strcpy(objName.ll,dlg->item_buff);
         break;
      case DSMAPI_OBJTYPE :
         switch (*dlg->item_buff)
         {
            case 'F':
            case 'f':
               objName.objType = DSM_OBJ_FILE; break;
            case 'D':
            case 'd':
               objName.objType = DSM_OBJ_DIRECTORY; break;
            default :
               objName.objType = DSM_OBJ_FILE;
         }
         break;
      case DSMAPI_OBJOWNER  :
         strcpy(objAttr.owner,dlg->item_buff);
         /* Note that for Intel platforms, the owner field should */
         /* be an empty string since these are single user systems. */
         break;
      case DSMAPI_OBJCOMP :
         switch (*dlg->item_buff)
         {
            case 'Y':
            case 'y':
               objAttr.objCompressed   = bTrue;     break;
            case 'N':
            case 'n':
               objAttr.objCompressed   = bFalse;    break;
            default :
               objAttr.objCompressed   = bFalse;    break;
         }
         break;
      case DSMAPI_MNTWAIT :
         switch (*dlg->item_buff)
         {
            case 'Y':
            case 'y':
               mountWait = bTrue;              break;
            case 'N':
            case 'n':
               mountWait = bFalse;             break;
            default :
               mountWait = bFalse;             break;
         }
         break;
      case DSMAPI_FILESIZE :
         filesize64 = dChtoI64(dlg->item_buff,10);
         break;
      case DSMAPI_NUM_FILES :
         numfiles64 = dChtoI64(dlg->item_buff,10);
         break;
      case DSMAPI_SEED_STRING :
         seed = dlg->item_buff;
         seed_size = strlen(seed);
         break;
      case DSMAPI_DESCRIPTION :
         desc = dlg->item_buff;
         desc_size = strlen(desc);
         break;
      case DSMAPI_MC  :   /* new mgmt class override */
         strcpy(mcOverride,dlg->item_buff);
         objAttr.mcNameP = mcOverride;
      default :
         ;
   }
   i++;
   dlg = &sel_dialog[i];
}


dI64toCh(&numfiles64,t4,10);  format_number(t4,t);
dI64toCh(&filesize64,t5,10);  format_number(t5,t2);
printf("Creating %s object(s) called %s%s%s(nnn) each of size %s.\n",
       t,objName.fs,objName.hl,objName.ll,t2);

/*----------------------------------------------------------------------+
| Use a simplistic brute force mechanism to stream data to the server.
| The seed string is duplicated as needed in sends (and buffers) until
| the requested object size is reached.  For ease of demonstration, we
| just duplicate the seed as many times as will fit in a buffer of size
| given in the preferences structure.  The seed string will never be
| partially duplicated since it is used as a validation string for the
| object data on restores or retrieves.
| We then use this buffer over and over (or just part of it)
| until the total object size is sent.  Not glamarous but OK for samples.
.----------------------------------------------------------------------*/
if (filesize64.hi || filesize64.lo) /* Any data to send */
{
   if (!seed_size)
   {
      printf("*** Error: Send action terminating since no seed string");
      printf(" was specified for the object data.\n");
      return 0;
   }
   if (pref.sz_send_buff < seed_size)
   {
      printf("*** Error: Send action terminating since send buffer size");
      printf("(%u)\n is less than seed string size (%u).\n",
             pref.sz_send_buff,seed_size);
      return 0;
   }

   if (!(bkup_buff = (char *)malloc(pref.sz_send_buff))) /* Allocate the buff.      */
     return RC_NO_MEMORY;

   cursor = bkup_buff;                         /* Position within buffer. */
   end_of_buff = bkup_buff + pref.sz_send_buff;
   while (end_of_buff > (cursor + seed_size + strlen(seed_count) + 1)) /* Duplicate seed in buff. */
   {
      memcpy(cursor,seed,seed_size);
      cursor += seed_size;
      x = sprintf(seed_count, "%d", (int)((cursor - bkup_buff) + 1));
      memcpy(cursor,seed_count,strlen(seed_count));
      cursor += strlen(seed_count);
   }
   for (i = end_of_buff - cursor; i ; --i )
   {
      *cursor = 0xFF ;
      cursor += 1 ;
   }
   buff_size = pref.sz_send_buff;

}
/*----------------------------------------------------------------------+
| Next make sure the filespace entered is registered at the server.
| The filespace registration must be done before sending the object(s)
| to the server.  Filespace registration and queries are not allowed
| within the bounds of a transaction so must be done here.  This is
| also for system performance reasons so that we do not have to break
| out of a transaction to perform a registration.  The call below will
| check to see if the fs is registered.  If so a 0 rc is returned.  If
| not then the registration is attempted with the results returned in
| the rc.
.----------------------------------------------------------------------*/
memset(&fsData,0x00,sizeof(fsData));
fsData.stVersion = regFSDataVersion;
fsData.fsName = objName.fs;
fsData.fsType = smpAPI;
initFS(&fsData);   /* fill in platform specific info */

fsData.occupancy.hi=0;
fsData.occupancy.lo=100;
fsData.capacity.hi=0;
fsData.capacity.lo=300;

if ((rc = register_fs(dsmHandle,&fsData)))
{
   printf("Filespace registration failed: ");
   rcApiOut(dsmHandle, rc);
   free(bkup_buff);
   return (RC_SESSION_FAILED);
}

if (!(objInfo = (smpObjInfo *)malloc(seed_size + 1 + sizeof(smpObjInfo))))
{
   free(bkup_buff);
   return RC_NO_MEMORY;
}

memset(objInfo, 0x00, (seed_size + 1 + sizeof(smpObjInfo)) );

/*----------------------------------------------------------------------+
| Now we are set to start sending each object to the server w/in txns.
| This is the outer loop for each object to be sent.  Since 64bit numbers
| are supported we could be here a long time!
.----------------------------------------------------------------------*/
for (files64.hi=0, files64.lo=0;               /* Init to 0 start       */
     (dCmp64(&files64,&numfiles64) == DS_LESSTHAN);  /* Still more to process?*/
     files64 = dIncr64(&files64,1))              /* Bump counter by 1.    */
{
   /*--------------------------------------------------------------------+
   | First create the name of the object to send and init object counters.
   | If we were asked to create more than 1 file in this round then we will
   | add a sequence number onto the end of each low level name qualifier to
   | uniquely identify it to the system.
   .--------------------------------------------------------------------*/
   if ((numfiles64.lo > 1) || (numfiles64.hi > 0))
   {
      strcpy(objName.ll,ll);
      dI64toCh(&files64,t,10);
      strcat(objName.ll,t);
   }

   if (filesize64.hi || filesize64.lo) /* Any data to send */
   {
       /*=== modify the seed ===*/
       for(i = 0; i < seed_size; i++ )
             seed[i] = validChars[rand() % 25];
       seed[i] = '\0';

    
       /*=== modify each file ===*/
       cursor = bkup_buff;                         /* Position within buffer. */
       end_of_buff = bkup_buff + pref.sz_send_buff;
       while (end_of_buff > (cursor + seed_size + strlen(seed_count) + 1)) /* Duplicate seed in buff. */
       {
          memcpy(cursor,seed,seed_size);
          cursor += seed_size;
          x = sprintf(seed_count, "%d", (int)((cursor - bkup_buff) + 1));
          memcpy(cursor,seed_count,strlen(seed_count));
          cursor += strlen(seed_count);
       }
       for (i = end_of_buff - cursor; i ; --i )
       {
          *cursor = 0xFF ;
          cursor += 1 ;
       }
       buff_size = pref.sz_send_buff;
   }

   /* Convert 64bit nums to printable strings and format them */
   dI64toCh(&numfiles64,t4,10); format_number(t4,t);
   dI64toCh(&filesize64,t4,10); format_number(t4,t2);
   count64 = dIncr64(&files64,1);    /* format counter for user */
   dI64toCh(&count64,t4,10);    format_number(t4,t3);

   printf("Creating object %s of %s   Size=%s    Name=%s%s%s\n",
          t3,t,t2,objName.fs,objName.hl,objName.ll);
   bytes_left64.hi = filesize64.hi;
   bytes_left64.lo = filesize64.lo;
   done = bFalse;
   buff_num64.hi = 0; buff_num64.lo = 1;

   if (!in_txn)
   {
      if ((rc = ourBeginTxn(dsmHandle,        /* Tivoli Storage Manager API sess handle */
                           &txn_objs,         /* Objects in this txn.                   */
                           txn_llist_h,       /* Linked list head.                      */
                           txn_llist_t,       /* Linked list tail.                      */
                           last_copy_dest)) )
      {
         printf("*** BeginTxn(1) failed: ");
         rcApiOut(dsmHandle, rc);
         rc = (RC_SESSION_FAILED);
         break;
      }
      /*-----------------------------------------------------------------.
      | If we get here we are in session and within the bounds of a txn
      | and all of our transaction counters, etc., are initialized.
      .-----------------------------------------------------------------*/
      new_txn_needed = bFalse;
      in_txn = bTrue;
   }

   /*--------------------------------------------------------------------.
   | In transaction.  Just about to send object but a BindMC must first
   | be done and further checks done to
   .--------------------------------------------------------------------*/
   MCBindKey.stVersion = mcBindKeyVersion;

   switch (send_type)
   {
      case (Backup_Send) :
         rc = dsmBindMC(dsmHandle,&objName,stBackup,&MCBindKey);
         temp_string = MCBindKey.backup_copy_dest;
         break;
      case (Archive_Send) :
         rc = dsmBindMC(dsmHandle,&objName,stArchive,&MCBindKey);
         temp_string = MCBindKey.archive_copy_dest;
         break;
      default : ;
   }

   if (rc)
   {
      printf("*** dsmBindMC failed: ");
      rcApiOut(dsmHandle, rc);
      rc = (RC_SESSION_FAILED);
      break;
   }

   if (pref.verbose)
   {
      printf("      MC Name: %s\n",MCBindKey.mcName);
      printf(" MC copy dest: %s\n",temp_string);
   }

   /*--------------------------------------------------------------------.
   | Got a MC bound.  Now see if we can backup the item based on the
   | copygroup.
   .--------------------------------------------------------------------*/
   if (((!MCBindKey.backup_cg_exists)
         && send_type == Backup_Send)
         ||
       ((!MCBindKey.archive_cg_exists)
         && send_type == Archive_Send))
   {
      printf("*** Object %s%s%s will not be sent because management class\n",
         objName.fs,objName.hl,objName.ll);
      printf("%s does not have a backup/archive copygroup defined.\n",
         MCBindKey.mcName);
      break;
   }
   /*--------------------------------------------------------------------.
   | Do the last and current copy destinations match (if not first one)?
   .--------------------------------------------------------------------*/
   if (((!strcmp(last_copy_dest,MCBindKey.backup_copy_dest) && (txn_objs))
       && send_type == Backup_Send)
        ||
       ((!strcmp(last_copy_dest,MCBindKey.archive_copy_dest) && (txn_objs))
       && send_type == Archive_Send))
   {
      new_txn_needed = bTrue;

      printf("****** New txn: Last copy dest(%s) != current copy dest",
                  last_copy_dest);
      switch (send_type)
      {
         case (Backup_Send) :
            printf("(%s).\n",MCBindKey.backup_copy_dest);
            break;
         case (Archive_Send) :
            printf("(%s).\n",MCBindKey.archive_copy_dest);
            break;
         default : ;
      }
   }
   /*--------------------------------------------------------------------.
   | Are we within the max number of objects for a transaction?
   .--------------------------------------------------------------------*/
   if (txn_objs >= dsmSessInfo.maxObjPerTxn)
   {
      new_txn_needed = bTrue;

      printf("***** New txn: Current txn objects (%u) ",txn_objs);
      printf(">= max txn objects(%u).\n",dsmSessInfo.maxObjPerTxn);
   }

   /*--------------------------------------------------------------------.
   | See if a new txn is needed.  If so end current txn and start a new
   | one.
   .--------------------------------------------------------------------*/
   if (new_txn_needed)
   {
      txn_reason = 0;
      rc = dsmEndTxn(dsmHandle,          /* Tivoli Storage Manager API sess handle */
                     DSM_VOTE_COMMIT,    /* Commit transaction.                    */
                     &txn_reason);       /* Reason if txn aborted.                 */
      if (rc || txn_reason)
      {
         printf("*** dsmEndTxn(1) failed: rc = ");
         rcApiOut(dsmHandle, rc);
         printf("   Reason = ");
         rcApiOut(dsmHandle, txn_reason);
         rc = (RC_SESSION_FAILED);
         in_txn = bFalse;
         break;
      }

      if ((rc = ourBeginTxn(dsmHandle,       /* Tivoli Storage Manager API sess handle */
                          &txn_objs,         /* Objects in this txn.                   */
                          txn_llist_h,       /* Linked list head.                      */
                          txn_llist_t,       /* Linked list tail.                      */
                          last_copy_dest)) )
      {
         printf("*** BeginTxn(2) failed: ");
         rcApiOut(dsmHandle, rc);
         rc = (RC_SESSION_FAILED);
         break;
      }
      new_txn_needed = bFalse;
   }

   /*--------------------------------------------------------------------.
   | Believe it or not if we get here we can start sending the object to
   | the server!  We are in a transaction and are within the bounds of a
   | single or multiple transaction.  For now we will issue the
   | dsmSendObj without any data to pass all obj info, and the issue
   | dsmSendData call(s) as needed for the data.
   .--------------------------------------------------------------------*/
   objAttr.stVersion = ObjAttrVersion;

   if (!pref.use_est)
   {
      /* Fake out size of object so server will let us send huge objects.*/
      objAttr.sizeEstimate.hi = 0;
      objAttr.sizeEstimate.lo = DSM_MIN_COMPRESS_SIZE + 1;
   }
   else
   {
      objAttr.sizeEstimate.hi = filesize64.hi; /* Size of object being sent*/
      objAttr.sizeEstimate.lo = filesize64.lo;
   }

   objInfo->objType        = objName.objType; /* Type of object.        */
   objInfo->size.hi        = filesize64.hi;  /* Size of obj. being sent.*/
   objInfo->size.lo        = filesize64.lo;
   strcpy(objInfo->eyecatcher,api_eyecatcher); /* Indicate our data.    */
   strcpy(objInfo->seed,seed);          /* Seed string used to gen file */
   


   objAttr.objInfo         = (char *) malloc(DSM_MAX_OBJINFO_LENGTH) ; 
   objAttr.objInfoLength   = strlen(seed) + 1 + sizeof(smpObjInfo);
   memcpy(objAttr.objInfo, (char *)objInfo, objAttr.objInfoLength);


   if (pref.verbose)
   {
      tobjInfo = (smpObjInfo *)objAttr.objInfo;
      printf("Object info for %s%s%s :\n",objName.fs,objName.hl,objName.ll);
      dI64toCh(&(tobjInfo->size),t4,10);  format_number(t4,t);
      printf("   Size of object : %s\n",t);
      printf("   Seed string ==>%s<==\n",tobjInfo->seed);
   }

   switch (send_type)
   {
      case (Backup_Send) :
       if (mountWait)
         rc = dsmSendObj(dsmHandle,
                         stBackupMountWait,
                         NULL,
                         &objName,
                         &objAttr,
                         NULL);
       else
         rc = dsmSendObj(dsmHandle,
                         stBackup,
                         NULL,
                         &objName,
                         &objAttr,
                         NULL);
         break;
      case (Archive_Send) :
       archData.stVersion = sndArchiveDataVersion;
       archData.descr = desc;
       if (mountWait)
         rc = dsmSendObj(dsmHandle,
                         stArchiveMountWait,
                         &archData,
                         &objName,
                         &objAttr,
                         NULL);
       else
         rc = dsmSendObj(dsmHandle,
                         stArchive,
                         &archData,
                         &objName,
                         &objAttr,
                         NULL);
         break;
      default : ;
   }
   free(objAttr.objInfo); 
   if (rc)
   {
      printf("*** dsmSendObj failed: ");
      rcApiOut(dsmHandle, rc);
      rc = (RC_SESSION_FAILED);
      break;
   }

   buff_size64 = dMake64(buff_size);  /* Need for compares */

   if (!(bytes_left64.lo || bytes_left64.hi)) done = bTrue;
   while (!done)
   {
      if ( dCmp64(&bytes_left64, &buff_size64) == DS_LESSTHAN)
      {
         send_amt = bytes_left64.lo;    /* Only need lo 4 bytes here! */
      }
      else
      {
         send_amt = buff_size;
      }

      dataBlk.stVersion = DataBlkVersion;
      dataBlk.bufferLen = send_amt;
      dataBlk.numBytes  = 0;
      dataBlk.bufferPtr = bkup_buff;
      rc = dsmSendData(dsmHandle,&dataBlk);
      if (rc)
      {
         printf("*** dsmSendData failed: ");
         rcApiOut(dsmHandle, rc);
         done = bTrue;
      }

      if (send_amt < buff_size)  done = bTrue;
      bytes_left64 = dDecr64(&bytes_left64,send_amt);

      count64 = dIncr64(&files64,1);    /* format counter for user */
      dI64toCh(&count64,t4,10);      format_number(t4,t);
      dI64toCh(&buff_num64,t4,10);   format_number(t4,t2);
      dI64toCh(&bytes_left64,t4,10); format_number(t4,t3);
      t64 = dMake64(dataBlk.numBytes);
      dI64toCh(&t64,t4,10);          format_number(t4,t5);
      printf("   Object: %6s Buffer: %4s Bytes sent: %7s Bytes left: %s\n",
             t,t2,t5,t3);

      if (dataBlk.numBytes != send_amt)
        printf("*** WARNING ***  Only sent %u bytes out of %u.\n",
               dataBlk.numBytes, send_amt);
      buff_num64 = dIncr64(&buff_num64,1);
   }

   memset(&endSendObjExIn,  0x00, sizeof(dsmEndSendObjExIn_t));
   memset(&endSendObjExOut, 0x00, sizeof(dsmEndSendObjExOut_t));

   endSendObjExIn.stVersion  = dsmEndSendObjExInVersion;
   endSendObjExIn.dsmHandle  = dsmHandle;

   endSendObjExOut.stVersion = dsmEndSendObjExOutVersion;

   rc = dsmEndSendObjEx(&endSendObjExIn, &endSendObjExOut);
   if (rc)
   {
       printf("*** dsmEndSendObjEx failed: ");
       rcApiOut(dsmHandle, rc);
   }
   else
   {
       dI64toCh(&endSendObjExOut.totalBytesSent,t,10);
       format_number(t,t2);
       printf("Total bytes sent:     %s\n", t2);
       printf("Compression:          %s\n", 
           endSendObjExOut.objCompressed == bTrue ? "YES" : "NO");

       dI64toCh(&endSendObjExOut.totalCompressSize,t,10);
       format_number(t,t2);
       printf("Compressed size:      %s\n", t2);

       dI64toCh(&endSendObjExOut.totalLFBytesSent,t,10);
       format_number(t,t2);
       printf("LAN-free bytes sent:  %s\n", t2);

       printf("Encryption:           %s\n",
              endSendObjExOut.encryptionType & DSM_ENCRYPT_CLIENTENCRKEY ? "CLIENTENCRKEY" :
              endSendObjExOut.encryptionType & DSM_ENCRYPT_USER ? "USER" : "NO");
       printf("Encryption Strength:  %s\n",
              endSendObjExOut.encryptionType & DSM_ENCRYPT_AES_256BIT ? "AES_256BIT" :
              endSendObjExOut.encryptionType & DSM_ENCRYPT_AES_128BIT ? "AES_128BIT" :
              endSendObjExOut.encryptionType & DSM_ENCRYPT_DES_56BIT ? "DES_56BIT" : "NONE");
       printf("Object Deduplicated:  %s\n",
              endSendObjExOut.objDeduplicated ? "YES" : "NO");
       dI64toCh(&endSendObjExOut.totalDedupSize,t,10);
       format_number(t,t2);
       printf("Deduplicated size:    %s\n\n", t2);

    }
   txn_objs++;
}

if (in_txn)
{
   txn_reason = 0;
   rc = dsmEndTxn(dsmHandle,          /* Tivoli Storage Manager API sess handle */
                 DSM_VOTE_COMMIT,     /* Commit transaction.                    */
                 &txn_reason);        /* Reason if txn aborted.                 */
   if (rc || txn_reason)
   {
      printf("*** dsmEndTxn(2) failed: rc = ");
      rcApiOut(dsmHandle, rc);
      printf("   Reason = ");
      rcApiOut(dsmHandle, txn_reason);
      rc = (RC_SESSION_FAILED);
   }
}

if (bkup_buff)
{
   free(bkup_buff);
   bkup_buff = NULL;
}
if (objInfo)
{
   free(objInfo);
   objInfo = NULL;
}

return rc;
}

/*----------------------------------------------------------------------+
| Name:    ourBeginTxn()
|
| Action:  Do a begin txn to the API as well as .............
|          Eventually this routine should add the objects in the txn to
|          a linked list so that if the txn is aborted the objects in the
|          list can be retried or marked as failed.
|
| Input:
|
| Returns:
|
| Side
| Effects: None
|
| Notes:   None
+----------------------------------------------------------------------*/
static dsInt16_t ourBeginTxn(dsUint32_t  dsmHandle,        /* Tivoli Storage Manager API sess handle  */
                         dsUint32_t  *txn_objs,        /* Objects in this txn. */
                         llist   *txn_llist_h,     /* Linked list head.    */
                         llist   *txn_llist_t,     /* Linked list tail.    */
                         char    *last_copy_dest)
{
dsInt16_t rc;

rc = dsmBeginTxn(dsmHandle);
*txn_objs = 0;
*last_copy_dest = 0x00;
return rc;
}

/*----------------------------------------------------------------------+
| Name:    register_fs()
|
| Action:  Do a register_fs to the server.  This routine could be
|          enhanced to check to see if the fs is already registered, but
|          since the api already does that we will let it do it for us.
|          Thus the cast of the already registered rc to 0 since this is
|          a valid situation.
|
| Input:
|
| Returns:
|
| Side
| Effects: None
|
| Notes:   None
+----------------------------------------------------------------------*/
/* See if fs needs to be registered, and if so do so. */
static dsInt16_t register_fs(dsUint32_t dsmHandle,regFSData *fsData)
{
dsInt16_t rc;
rc = dsmRegisterFS(dsmHandle,fsData);
if (rc == DSM_RC_FS_ALREADY_REGED) rc = DSM_RC_OK;

return rc;
}

