/*
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License as
 *  published by the Free Software Foundation; either version 2 of
 *  the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details:
 *
 *  http://www.gnu.org/copyleft/gpl.txt
 */

#include "display.h"
#include "shared.h"

/*------------------------------------------------------------------------*/

/* IFFT_Bin_Value()
 *
 * Calculates IFFT bin values with auto level control
 */
  int
IFFT_Bin_Value( int sum_i, int sum_q, gboolean reset )
{
 /* Value of ifft output "bin" */
  static int bin_val = 0;

  /* Maximum value of ifft bins */
  static int bin_max = 1000, max = 0;

  /* Scale output from IFFT */
  sum_i /= rc_data.ifft_scale;
  sum_q /= rc_data.ifft_scale;

  /* Calculate sliding window average of max bin value */
  if( reset )
  {
    bin_max = max;
    if( !bin_max ) bin_max = 1;
    max = 0;
  }
  else
  {
    /* Calculate average signal power at each frequency (bin) */
    bin_val  = bin_val * AMPL_AVE_MUL;
    bin_val += sum_i * sum_i + sum_q * sum_q;
    bin_val /= AMPL_AVE_WIN;

    /* Record max bin value */
    if( max < bin_val )
      max = bin_val;

    /* Scale bin values to 255 depending on max value */
    int ret = (255 * bin_val) / bin_max;
    if( ret > 255 ) ret = 255;
    return( ret );
  }

  return( 0 );
} /* IFFT_Bin_Value() */

/*------------------------------------------------------------------------*/

/* Color codes the pixels of the
 * waterfall according to their value
 */
  static void
Color_Code( guchar *pix, int pixel_val )
{
  int n;

  if( pixel_val < 64 ) // From black to blue
  {
    pix[0] = 0;
    pix[1] = 0;
    pix[2] = 3 + (guchar)pixel_val * 4;
  }
  else if( pixel_val < 128 ) // From blue to green
  {
    n = pixel_val - 127; // -63 <= n <= 0
    pix[0] = 0;
    pix[1] = 255 + (guchar)n * 4;
    pix[2] = 3   - (guchar)n * 4;
  }
  else if( pixel_val < 192 ) // From green to yellow
  {
    n = pixel_val - 191; // -63 <= n <= 0
    pix[0] = 255 + (guchar)n * 4;
    pix[1] = 255;
    pix[2] = 0;
  }
  else // From yellow to reddish-orange
  {
    n = pixel_val - 255; // -63 <= n <= 0
    pix[0] = 255;
    pix[1] = 66 - (guchar)n * 3;
    pix[2] = 0;
  }

} /* Color_Code() */

/*------------------------------------------------------------------------*/

/* Display_Waterfall()
 *
 * Displays IFFT Spectrum as "waterfall"
 */
  void
Display_Waterfall( void )
{
  int
    vert_lim,  /* Limit of vertical index for copying lines */
    idh, idv,  /* Index to hor. and vert. position in warterfall */
    pixel_val, /* Greyscale value of pixel derived from ifft o/p  */
    idf,       /* Index to ifft output array */
    i, len;

  /* Pointer to current pixel */
  static guchar *pix;


  /* Copy each line of waterfall to next one */
  vert_lim = wfall_height - 2;
  for( idv = vert_lim; idv > 0; idv-- )
  {
    pix = wfall_pixels + wfall_rowstride * idv + wfall_n_channels;

    for( idh = 0; idh < wfall_width; idh++ )
    {
      pix[0] = pix[ -wfall_rowstride];
      pix[1] = pix[1-wfall_rowstride];
      pix[2] = pix[2-wfall_rowstride];
      pix += wfall_n_channels;
    }
  }

  /* Go to top left +1 hor. +1 vert. of pixbuf */
  pix = wfall_pixels + wfall_rowstride + wfall_n_channels;

  /* Do real fft from the 2.4 kHz subcarrier or
   * complex fft from the RTLSDR I/Q output */
  if( isFlagClear(CARRIER_SPECTRUM) )
  {
    IFFT_Real( ifft_data );

    idf = 2;
    len = ifft_data_length / 2;
    for( i = 1; i < len; i++ )
    {
      /* Calculate vector magnitude of
       * signal at each freq. ("bin") */
      pixel_val = IFFT_Bin_Value(
          ifft_data[idf], ifft_data[idf + 1], FALSE );
      idf += 2;

      /* Color code signal strength */
      Color_Code( pix, pixel_val );
      pix += wfall_n_channels;

    } /* for( i = 1; i < len; i++ ) */
  }
  else
  {
    /* IFFT produces an output of positive and negative
     * frequencies and it output is handled accordingly */
    IFFT( ifft_data );

    /* Calculate bin values after IFFT */
    len = ifft_data_length / 4;

    /* Do the "positive" frequencies */
    idf = ifft_data_length / 2;
    for( i = 0; i < len; i++ )
    {
      /* Calculate vector magnitude of
       * signal at each freq. ("bin") */
      pixel_val = IFFT_Bin_Value(
          ifft_data[idf], ifft_data[idf + 1], FALSE );
      idf += 2;

      /* Color code signal strength */
      Color_Code( pix, pixel_val );
      pix += wfall_n_channels;

    } /* for( i = 1; i < len; i++ ) */

    /* Do the "negative" frequencies */
    idf = 2;
    for( i = 1; i < len; i++ )
    {
      /* Calculate vector magnitude of
       * signal at each freq. ("bin") */
      pixel_val = IFFT_Bin_Value(
          ifft_data[idf], ifft_data[idf + 1], FALSE );
      idf += 2;

      /* Color code signal strength */
      Color_Code( pix, pixel_val );
      pix += wfall_n_channels;

    } /* for( i = 1; i < len; i++ ) */
  } /* if( isFlagClear(CARRIER_SPECTRUM) ) */

  /* Reset function */
  IFFT_Bin_Value( ifft_data[0], ifft_data[0], TRUE );

  /* At last draw waterfall */
  gtk_widget_queue_draw( waterfall );

} /* Display_Waterfall() */

/*------------------------------------------------------------------------*/

/*  Display_Signal()
 *
 *  Displays the sub-carrier signal amplitude
 */

/* Points to plot */
static GdkPoint *points = NULL;

  void
Display_Signal( int plot )
{
  static int points_idx = 0;

  /* Initialize on first call */
  if( points == NULL )
  {
    if( !mem_alloc( (void *)&points,
          (size_t)scope_width * sizeof(GdkPoint)) )
      return;
  }

  /* Save values to be plotted (scaled to fit display) */
  points[points_idx].y = scope_height - plot - 1;
  if( points[points_idx].y <= 0 )
    points[points_idx].y = 1;
  if( points[points_idx].y >= scope_height )
    points[points_idx].y = scope_height - 1;

  points[points_idx].x = points_idx;

  /* Recycle buffer idx when full and plot */
  if( ++points_idx >= scope_width )
  {
    SetFlag( ENABLE_SCOPE );
    gtk_widget_queue_draw( scope );
    points_idx = 0;
  } /* ++points_idx >= scope_width */

} /* Display_Signal */

/*------------------------------------------------------------------------*/

/* Draw_Signal()
 *
 * Draws the signal detector's output
 */
  void
Draw_Signal( cairo_t *cr )
{
  int idx;

  /* Draw scope backgrounds */
  cairo_set_source_rgb( cr, 0.0, 0.3, 0.0 );
  cairo_rectangle(
      cr, 0.0, 0.0,
      (double)scope_width,
      (double)scope_height );
  cairo_fill( cr );

  /* Plot signal graph */
  cairo_set_source_rgb( cr, 0.0, 1.0, 0.0 );

  cairo_move_to( cr,
      (double)points[0].x,
      (double)points[0].y );
  for( idx = 1; idx < scope_width; idx++ )
    cairo_line_to( cr,
        (double)points[idx].x,
        (double)points[idx].y );

  /* Stroke paths */
  cairo_stroke( cr );

  ClearFlag( ENABLE_SCOPE );

} /* Draw_Signal() */

/*------------------------------------------------------------------------*/

