// -*- C++ -*-
#include "Rivet/Analysis.hh"
#include "Rivet/Projections/Beam.hh"
#include "Rivet/Projections/UnstableParticles.hh"

namespace Rivet {


  /// @brief Longitudinal and transverse momenta of neutral pions in the forward-rapidity region
  class LHCF_2016_I1385877 : public Analysis {
  public:

    /// Constructor
    RIVET_DEFAULT_ANALYSIS_CTOR(LHCF_2016_I1385877);

    /// @name Analysis methods
    /// @{

    /// Book histograms and initialise projections before the run
    void init() {

      // Initialise and register projections
      declare(UnstableParticles(), "UFS");
      declare(Beam(), "Beam");

      // Calculate beam rapidity
      const Particle bm1 = beams().first;
      const Particle bm2 = beams().second;
      MSG_DEBUG("Beam 1 : momentum=" << bm1.pz() << " PID=" << bm1.pid() << " rapidity=" << bm1.rap());
      MSG_DEBUG("Beam 2 : momentum=" << bm2.pz() << " PID=" << bm2.pid() << " rapidity=" << bm2.rap());
      MSG_DEBUG("CM energy: " << sqrtS() );
      _beam_rap = bm1.rap();

      // Book histos for p-p or p-Pb mode
      for (double eVal : allowedEnergies()) {
        const string en = toString(round(eVal));
        if (isCompatibleWithSqrtS(eVal))  _sqs = en;
        if (en == "7000"s) {
          book(_g[en+"rap_pT"], {8.8, 9., 9.2, 9.4, 9.6, 9.8, 10., 10.2, 10.4, 10.6, 10.8});
          for (auto& b : _g[en+"rap_pT"]->bins()) {
            book(b, 1+b.index(), 1, 2);
          }
          book(_g[en+"pT_pZ"], {0., 0.2, 0.4, 0.6, 0.8, 1.});
          for (auto& b : _g[en+"pT_pZ"]->bins()) {
            book(b, 11+b.index(), 1, 2);
          }
          book(_p[en+"rap_apT"],      1, 1, 2);
          book(_h[en+"rap"],         21, 1, 2);
          book(_p[en+"raploss_apT"], 22, 1, 2);
          book(_h[en+"raploss"],     23, 1, 2);
        }
        if (en == "2760"s) {
          book(_p[en+"rap_apT"], 1, 1, 1);
          book(_g[en+"rap_pT"], {8.8, 9., 9.2, 9.4, 9.6, 9.8});
          for (auto& b : _g[en+"rap_pT"]->bins()) {
            book(b, 1+b.index(), 1, 1);
          }
          book(_g[en+"pT_pZ"], {0., 0.2, 0.4}, {"d12-x01-y01", "d13-x01-y01"});
          book(_h[en+"rap"], 21, 1, 1);
          book(_p[en+"raploss_apT"], 22, 1, 1);
          book(_h[en+"raploss"], 23, 1, 1);
        }
        else if (en == "1044160"s) {
          book(_g[en+"rap_pT"], {8.8, 9., 9.2, 9.4, 9.6, 9.8, 10., 10.2, 10.4, 10.6, 10.8});
          for (auto& b : _g[en+"rap_pT"]->bins()) {
            book(b, 1+b.index(), 1, 3);
          }
          book(_g[en+"pT_pZ"], {0., 0.2, 0.4, 0.6, 0.8, 1.0});
          for (auto& b : _g[en+"pT_pZ"]->bins()) {
            book(b, 11+b.index(), 1, 3);
          }
          book(_p[en+"rap_apT"],      1, 1, 3);
          book(_p[en+"raploss_apT"], 22, 1, 3);
        }
      }
      raiseBeamErrorIf(_sqs.empty());
    }


    /// Perform the per-event analysis
    void analyze(const Event& event) {

      // Select neutral pions
      const UnstableParticles& ufs = apply<UnstableParticles> (event, "UFS");
      const Particles pions = ufs.particles(Cuts::pz > 0 && Cuts::abspid == PID::PI0 && Cuts::pT > 0.01*GeV);
      for (const Particle& p : pions) {
        const double pT = p.pT()/GeV;
        const double rap = p.rap();
        const double raploss = _beam_rap - p.rap();
        _p[_sqs+"rap_apT"]->fill(rap, p.pT()/MeV);
        _p[_sqs+"raploss_apT"]->fill(raploss, p.pT()/MeV);
        _g[_sqs+"rap_pT"]->fill(rap, pT, 1.0/pT);
        _g[_sqs+"pT_pZ"]->fill(pT, p.pz()/GeV, p.E()/GeV/pT);
        if (_sqs != "1044160"s) {
          _h[_sqs+"rap"]->fill(rap);
          _h[_sqs+"raploss"]->fill(raploss);
        }
      }

    }


    /// Normalise histograms etc., after the run
    void finalize() {

      const double sf = 1.0/sumOfWeights()/TWOPI;
      const double bw = 0.2;
      for (auto& grp : _g) {
        for (auto& hist : grp.second->bins()) {
          if (grp.first.find("pT_") != string::npos) {
            if (hist.index() == 1)  hist->scaleW(sf/(bw-0.01));
            else  hist->scaleW(sf/bw);
          }
          else {
            size_t cutoff_bin = hist->indexAt(0.01);
            if (cutoff_bin > 0) {
              const double cutoff_wdt = hist->bin(cutoff_bin).xWidth();
              hist->bin(cutoff_bin).scaleW((cutoff_wdt)/(cutoff_wdt-0.01));
            }
            hist->scaleW(sf/bw);
          }
        }
      }
      scale(_h, 1.0/sumOfWeights());
    }

    /// @}



    // Store the beam rapidity for rap-loss calculation (could just re-access this in analyze())
    double _beam_rap;

    string _sqs = "";

    /// @name Histograms
    /// @{
    map<string,Histo1DPtr> _h;
    map<string,Profile1DPtr> _p;
    map<string,Histo1DGroupPtr> _g;
    /// @}

  };



  RIVET_DECLARE_PLUGIN(LHCF_2016_I1385877);
}
