/*
 *                           TERMS AND CONDITIONS
 *                                   FOR
 *                         OPEN SOURCE CODE LICENSE
 *                               Version 1.1
 * 
 * Japan Registry Services Co., Ltd. ("JPRS"), a Japanese corporation
 * having its head office at Chiyoda First Bldg. East 13F 3-8-1 Nishi-Kanda,
 * Chiyoda-ku, Tokyo 101-0065, Japan, grants you the license for open source
 * code specified in EXHIBIT A the "Code" subject to the following Terms and
 * Conditions ("OSCL").
 * 
 * 1. License Grant.
 *   JPRS hereby grants you a worldwide, royalty-free, non-exclusive
 *   license, subject to third party intellectual property claims:
 *   (a) under intellectual property rights (other than patent or
 *       trademark) licensable by JPRS to use, reproduce, modify, display,
 *       perform, sublicense and distribute the Code (or portions thereof)
 *       with or without modifications, and/or as part of a derivative work;
 *       or
 *   (b) under claims of the infringement through the making, using,
 *       offering to sell and/or otherwise disposing the JPRS Revised Code
 *       (or portions thereof);
 *   (c) the licenses granted in this Section 1(a) and (b) are effective on
 *       the date JPRS first distributes the Code to you under the terms of
 *       this OSCL;
 *   (d) Notwithstanding the above stated terms, no patent license is
 *       granted:
 *       1)  for a code that you delete from the Code;
 *       2)  separate from the Code; or
 *       3)  for infringements caused by:
 *            i) modification of the Code; or
 *           ii) combination of the Code with other software or devices.
 * 
 * 2. Consents.
 *   You agree that:
 *   (a) you must include a copy of this OSCL and the notice set forth in
 *       EXHIBIT A with every copy of the Code you distribute;
 *   (b) you must include a copy of this OSCL and the notice set forth in
 *       EXHIBIT A with every copy of binary form of the Code in the
 *       documentation and/or other materials provided with the distribution;
 *   (c) you may not offer or impose any terms on any source code version
 *       that alters or restricts the applicable version of this OSCL or
 *       the recipients' rights hereunder.
 *   (d) If the terms and conditions are set forth in EXHIBIT A, you must
 *       comply with those terms and conditions.
 * 
 * 3. Proprietary Information.
 *   All trademarks, service marks, patents, copyrights, trade secrets, and
 *   other proprietary rights in or related to the Code are and will remain
 *   the exclusive property of JPRS or its licensors, whether or not
 *   specifically recognized or perfected under local law except specified
 *   in this OSCL; provided however you agree and understand that the JPRS
 *   name may not be used to endorse or promote this Code without prior
 *   written approval of JPRS.
 * 
 * 4. WARRANTY DISCLAIMER.
 *   JPRS MAKES NO REPRESENTATIONS AND WARRANTIES REGARDING THE USE OF THE
 *   CODE, NOR DOES JPRS MAKE ANY REPRESENTATIONS THAT THE CODE WILL BECOME
 *   COMMERCIALLY AVAILABLE. JPRS, ITS AFFILIATES, AND ITS SUPPLIERS DO NOT
 *   WARRANT OR REPRESENT THAT THE CODE IS FREE OF ERRORS OR THAT THE CODE
 *   IS SUITABLE FOR TRANSLATION AND/OR LOCALIZATION. THE CODE IS PROVIDED
 *   ON AN "AS IS" BASIS AND JPRS AND ITS SUPPLIERS HAVE NO OBLIGATION TO
 *   CORRECT ERRORS OR TO SUPPORT THE CODE UNDER THIS OSCL FOR ANY REASON.
 *   TO THE FULL EXTENT PERMITTED BY LAW, ALL OBLIGATIONS ARE HEREBY
 *   EXCLUDED WHETHER EXPRESS, STATUTORY OR IMPLIED UNDER LAW, COURSE OF
 *   DEALING, CUSTOM, TRADE USAGE, ORAL OR WRITTEN STATEMENT OR OTHERWISE,
 *   INCLUDING BUT NOT LIMITED TO ANY IMPLIED WARRANTIES OF MERCHANTABILITY
 *   OR FITNESS FOR A PARTICULAR PURPOSE CONCERNING THE CODE.
 * 
 * 5. NO LIABILITY.
 *   UNDER NO CIRCUMSTANCES SHALL JPRS AND/OR ITS AFFILIATES, LICENSORS, OR
 *   REPRESENTATIVES BE LIABLE FOR ANY DAMAGES INCLUDING BUT NOT LIMITED TO
 *   CONSEQUENTIAL, INDIRECT, SPECIAL, PUNITIVE OR INCIDENTAL DAMAGES,
 *   WHETHER FORESEEABLE OR UNFORESEEABLE, BASED ON YOUR CLAIMS, INCLUDING,
 *   BUT NOT LIMITED TO, CLAIMS FOR LOSS OF DATA, GOODWILL, PROFITS, USE OF
 *   MONEY, INTERRUPTION IN USE OR AVAILABILITY OF DATA, STOPPAGE, IMPLIED
 *   WARRANTY, BREACH OF CONTRACT, MISREPRESENTATION, NEGLIGENCE, STRICT
 *   LIABILITY IN TORT, OR OTHERWISE.
 * 
 * 6. Indemnification.
 *   You hereby agree to indemnify, defend, and hold harmless JPRS for any
 *   liability incurred by JRPS due to your terms of warranty, support,
 *   indemnity, or liability offered by you to any third party.
 * 
 * 7. Termination.
 * 7.1 This OSCL shall be automatically terminated in the events that:
 *   (a) You fail to comply with the terms herein and fail to cure such
 *       breach within 30 days of becoming aware of the breach;
 *   (b) You initiate patent or copyright infringement litigation against
 *       any party (including a cross-claim or counterclaim in a lawsuit)
 *       alleging that the Code constitutes a direct or indirect patent or
 *       copyright infringement, in such case, this OSCL to you shall
 *       terminate as of the date such litigation is filed;
 * 7.2 In the event of termination under Sections 7.1(a) or 7.1(b) above,
 *     all end user license agreements (excluding distributors and
 *     resellers) which have been validly granted by You or any distributor
 *     hereunder prior to termination shall survive termination.
 *
 * 
 * 8. General.
 *   This OSCL shall be governed by, and construed and enforced in
 *   accordance with, the laws of Japan. Any litigation or arbitration
 *   between the parties shall be conducted exclusively in Tokyo, Japan
 *   except written consent of JPRS provides other venue.
 * 
 * 
 *                                EXHIBIT A
 * 
 * The original open source code of idnkit-2 is idnkit-1.0 developed and
 * conceived by Japan Network Information Center ("JPNIC"), a Japanese
 * association, Kokusai-Kougyou-Kanda Bldg 6F, 2-3-4 Uchi-Kanda,
 * Chiyoda-ku, Tokyo 101-0047, Japan, and JPRS modifies above original code
 * under following Terms and Conditions set forth by JPNIC.
 * 
 *                                  JPNIC
 * 
 * Copyright (c) 2000-2002 Japan Network Information Center.  All rights reserved.
 * 
 * By using this file, you agree to the terms and conditions set forth bellow.
 * 
 *                       LICENSE TERMS AND CONDITIONS
 * 
 * The following License Terms and Conditions apply, unless a different
 * license is obtained from Japan Network Information Center ("JPNIC"),
 * a Japanese association, Kokusai-Kougyou-Kanda Bldg 6F, 2-3-4 Uchi-Kanda,
 * Chiyoda-ku, Tokyo 101-0047, Japan.
 * 
 * 1. Use, Modification and Redistribution (including distribution of any
 *    modified or derived work) in source and/or binary forms is permitted
 *    under this License Terms and Conditions.
 * 
 * 2. Redistribution of source code must retain the copyright notices as they
 *    appear in each source code file, this License Terms and Conditions.
 * 
 * 3. Redistribution in binary form must reproduce the Copyright Notice,
 *    this License Terms and Conditions, in the documentation and/or other
 *    materials provided with the distribution. For the purposes of binary
 *    distribution the "Copyright Notice" refers to the following language:
 *    "Copyright (c) 2000-2002 Japan Network Information Center.  All rights reserved."
 * 
 * 4. The name of JPNIC may not be used to endorse or promote products
 *    derived from this Software without specific prior written approval of
 *    JPNIC.
 * 
 * 5. Disclaimer/Limitation of Liability: THIS SOFTWARE IS PROVIDED BY JPNIC
 *    "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *    LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
 *    PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL JPNIC BE LIABLE
 *    FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 *    CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 *    SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
 *    BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 *    WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 *    OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
 *    ADVISED OF THE POSSIBILITY OF SUCH DAMAGES.
 * 
 * 
 *                        JPRS Public License Notice
 *                                   For
 *                                idnkit-2.
 * 
 * The contents of this file are subject to the Terms and Conditions for
 * the Open Source Code License (the "OSCL"). You may not use this file
 * except in compliance with above terms and conditions. A copy of the OSCL
 * is available at <http://jprs.co.jp/idn/>.
 * The JPRS Revised Code is idnkit-2.
 * The Initial Developer of the JPRS Revised Code is Japan Network
 * Information Center ("JPNIC"), a Japanese association,
 * Kokusai-Kougyou-Kanda Bldg 6F, 2-3-4 Uchi-Kanda, Chiyoda-ku, Tokyo
 * 101-0047, Japan.
 * "Copyright (c) 2000-2002 Japan Network Information Center.  All rights reserved."
 * "Copyright (c) 2010-2012 Japan Registry Services Co., Ltd.  All rights reserved."
 * Contributor(s): ______________________________________.
 * 
 * If you wish to allow use of your version of this file only under the
 * above License(s) and not to allow others to use your version of this
 * file, please indicate your decision by deleting the relevant provisions
 * above and replacing them with the notice and other provisions required
 * by the above License(s). If you do not delete the relevant provisions,
 * a recipient may use your version of this file under either the above
 * License(s).
 */

#include <config.h>

#include <stddef.h>
#include <ctype.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <idn/assert.h>
#include <idn/debug.h>
#include <idn/logmacro.h>
#include <idn/result.h>
#include <idn/foreignmap.h>
#include <idn/strhash32.h>
#include <idn/utf32.h>

/*
 * Maximum byte length of every line in a map definition file.
 */
#define MAX_MAPFILE_LINE_LENGTH		255

/*
 * The maximum number of code points as result of a mapping.
 */
#define MAX_MAP_RESULT_LENGTH		15

/*
 * Map entry.
 */
typedef struct foreignmap_entry *idn__foreignmap_entry_t;

struct foreignmap_entry {
	unsigned long from;
	unsigned long *to;
	size_t tolen;
	idn__foreignmap_entry_t next;
};

/*
 * Foreignmap object.
 */
struct idn__foreignmap {
	idn__strhash32_t hash;
	idn__foreignmap_entry_t entries_head;
	idn__foreignmap_entry_t entries_tail;
};

static char *		parse_utf32(char *p, unsigned long *vp);

/*
 * Create a foreignmap object.
 */
idn_result_t
idn__foreignmap_create(idn__foreignmap_t *ctxp) {
	idn_result_t r = idn_success;
	idn__foreignmap_t ctx = NULL;
	idn__strhash32_t new_hash = NULL;

	assert(ctxp != NULL);

	TRACE(("idn__foreignmap_create()\n"));

	ctx = (idn__foreignmap_t)malloc(sizeof(struct idn__foreignmap));
	if (ctx == NULL) {
		WARNING(("idn__foreignmap_create: malloc failed\n"));
		r = idn_nomemory;
		goto ret;
	}

	ctx->hash = NULL;
	ctx->entries_head = NULL;
	ctx->entries_tail = NULL;
	r = idn__strhash32_create(&new_hash);
	if (r != idn_success)
		goto ret;

	ctx->hash = new_hash;
	*ctxp = ctx;

ret:
	if (r != idn_success) {
		if (new_hash != NULL)
			idn__strhash32_destroy(new_hash, NULL);
		free(ctx);
	}
	TRACE(("idn__foreignmap_create(): %s\n", idn_result_tostring(r)));
	return (r);
}

/*
 * Release all the memory allocated for 'ctx'.
 */
void
idn__foreignmap_destroy(idn__foreignmap_t ctx) {
	idn__foreignmap_entry_t entry;
	idn__foreignmap_entry_t next;

	assert(ctx != NULL);

	TRACE(("idn__foreignmap_destroy()\n"));

	if (ctx->entries_head != NULL) {
		entry = ctx->entries_head;
		while (entry != NULL) {
			next = entry->next;
			free(entry->to);
			free(entry);
			entry = next;
		}
	}

	idn__strhash32_destroy(ctx->hash, NULL);
	free(ctx);

	TRACE(("idn__foreignmap_destroy(): the object is destroyed\n"));
}

/*
 * Add a mapping to 'ctx'.
 */
idn_result_t
idn__foreignmap_add(idn__foreignmap_t ctx, unsigned long from,
		   const unsigned long *to) {
	idn_result_t r = idn_success;
	unsigned long hash_key[2];
	idn__foreignmap_entry_t new_entry = NULL;
	const unsigned long *p;

	assert(ctx != NULL && to != NULL);

	hash_key[0] = from;
	hash_key[1] = '\0';
	TRACE(("idn__foreignmap_add(from=\"%s\", to=\"%s\")\n", 
	       idn__debug_utf32xstring(hash_key),
	       idn__debug_utf32xstring(to)));

	if (from == 0 || from > UTF32_MAX || IS_SURROGATE(from)) {
		r = idn_invalid_codepoint;
		goto ret;
	}
	for (p = to; *p != '\0'; p++) {
		if (*p > UTF32_MAX || IS_SURROGATE(*p)) {
			r = idn_invalid_codepoint;
			goto ret;
		}
	}

	new_entry = (idn__foreignmap_entry_t)malloc(sizeof(*new_entry));
	if (new_entry == NULL) {
		r = idn_nomemory;
		goto ret;
	}

	new_entry->from = from;
	new_entry->to   = idn__utf32_strdup(to);
	if (new_entry->to == NULL) {
		r = idn_nomemory;
		goto ret;
	}
	new_entry->tolen = idn__utf32_strlen(to);
	new_entry->next  = NULL;

	r = idn__strhash32_put(ctx->hash, hash_key, new_entry);
	if (r != idn_success)
		goto ret;

	if (ctx->entries_head == NULL) {
		ctx->entries_head = new_entry;
		ctx->entries_tail = new_entry;
	} else {
		ctx->entries_tail->next = new_entry;
		ctx->entries_tail = new_entry;
	}

ret:
	if (r != idn_success) {
		if (new_entry != NULL) {
			free(new_entry->to);
			free(new_entry);
		}
	}
	TRACE(("idn__foreignmap_add(): %s\n", idn_result_tostring(r)));
	return (r);
}

/*
 * Read a mapping definition file and add mappings in it to 'ctx'.
 */
idn_result_t
idn__foreignmap_addfromfile(idn__foreignmap_t ctx, const char *file) {
	idn_result_t r = idn_success;
	FILE *fp = NULL;
	char line[MAX_MAPFILE_LINE_LENGTH + 1];
	unsigned long from;
	unsigned long to[MAX_MAP_RESULT_LENGTH + 1];
	size_t tolen;
	int lineno = 0;

	assert(ctx != NULL && file != NULL);

	TRACE(("idn__foreignmap_addfromfile(file=\"%s\")\n",
	       idn__debug_xstring(file)));

	if (strncmp(file, "filemap:", 8) == 0)
		file += 8;
	fp = fopen(file, "r");
	if (fp == NULL) {
		r = idn_nofile;
		goto ret;
	}

	while (fgets(line, sizeof(line), fp) != NULL) {
		char *newline;
		char *p = line;

		lineno++;
		newline = strpbrk(line, "\r\n");
		if (newline != NULL)
			*newline = '\0';
		else if (fgetc(fp) != EOF) {
			ERROR(("libidnkit: too long line, line %d: \"%s\"\n",
			       lineno, idn__debug_xstring(line)));
			r = idn_invalid_syntax;
			goto ret;
		}
		while (isspace(*p))
			p++;
		if (*p == '\0' || *p == '#')
			continue;

		/*
		 * Parse a code point 'from'.
		 */
		while (isspace((unsigned char)*p))
			p++;
		p = parse_utf32(p, &from);
		if (p == NULL) {
			r = idn_invalid_syntax;
			goto ret;
		}
		while (isspace(*p))
			p++;
		if (*p != ';') {
			r = idn_invalid_syntax;
			goto ret;
		}
		p++;
		while (isspace((unsigned char)*p))
			p++;

		/*
		 * Parse code points 'to'.
		 */
		for (tolen = 0; tolen < MAX_MAP_RESULT_LENGTH; tolen++) {
			if (*p == '\0' || *p == '#' || *p == ';')
				break;
			p = parse_utf32(p, &to[tolen]);
			if (p == NULL) {
				r = idn_invalid_syntax;
				goto ret;
			} else if (to[tolen] == 0) {
				r = idn_invalid_codepoint;
				goto ret;
			}
			while (isspace(*p))
				p++;
		}
		if (tolen >= MAX_MAP_RESULT_LENGTH) {
			r = idn_invalid_syntax;
			goto ret;
		}
		to[tolen] = '\0';

		/*
		 * Add a maping ('from' -> 'to') to 'ctx'.
		 */
		r = idn__foreignmap_add(ctx, from, to);
		if (r != idn_success)
			goto ret;
	}

ret:
	TRACE(("idn__foreignmap_addfromfile(): %s\n", idn_result_tostring(r)));
	if (fp != NULL)
		fclose(fp);

	return (r);
}

/*
 * Parse an Unicode code point.
 */
static char *
parse_utf32(char *p, unsigned long *vp) {
	char *endp;

	/* Skip optional 'U+' */
	if (strncmp(p, "U+", 2) == 0)
		p += 2;
	if (!isxdigit(*p))
		return (NULL);
	*vp = strtoul(p, &endp, 16);
	if (p == endp || endp - p > 6)
		return (NULL);
	p = endp;

	return p;
}

/*
 * Perform mapping.
 */
idn_result_t
idn__foreignmap_map(idn__foreignmap_t ctx, const unsigned long *from,
		   unsigned long *to, size_t tolen) {
	idn_result_t r = idn_success;
	unsigned long *to_org = to;
	unsigned long key[2];
	idn__foreignmap_entry_t entry;

	assert(ctx != NULL && from != NULL && to != NULL);

	TRACE(("idn__foreignmap_map(from=\"%s\", tolen=%d)\n", 
	       idn__debug_utf32xstring(from), (int)tolen));

	if (tolen <= 0) {
		r = idn_buffer_overflow;
		goto ret;
	}
	to[0] = '\0';
		
	while (*from != '\0') {
		key[0] = *from;
		key[1] = '\0';

		entry = (idn__foreignmap_entry_t)
			idn__strhash32_get(ctx->hash, key);
		if (entry == NULL) {
			r = idn__utf32_strcpy(to, tolen, key);
			if (r != idn_success)
				goto ret;
			to++;
			tolen--;
		} else {
			r = idn__utf32_strcpy(to, tolen, entry->to);
			if (r != idn_success)
				goto ret;
			to += entry->tolen;
			tolen -= entry->tolen;
		}
		from++;
	}

ret:
	if (r == idn_success) {
		TRACE(("idn__foreignmap_map(): success (to=\"%s\")\n",
		       idn__debug_utf32xstring(to_org)));
	} else {
		TRACE(("idn__foreignmap_map(): %s\n",
		       idn_result_tostring(r)));
	}
	return (r);
}

/*
 * An entry point to 'mapproc' of the maplist module.
 */
idn_result_t
idn__foreignmap_mapproc(void *ctx, const unsigned long *from,
			unsigned long *to, size_t tolen) {
	return (idn__foreignmap_map((idn__foreignmap_t)ctx, from, to, tolen));
}

/*
 * An entry point to 'destroyproc' of the maplist module.
 */
void
idn__foreignmap_destroyproc(void *ctx) {
	idn__foreignmap_destroy((idn__foreignmap_t)ctx);
}

