import 'package:auto_route/auto_route.dart';
import 'package:flutter/material.dart';
import 'package:flutter_hooks/flutter_hooks.dart';
import 'package:hooks_riverpod/hooks_riverpod.dart';
import 'package:immich_mobile/modules/sharing/providers/album_title.provider.dart';
import 'package:immich_mobile/modules/sharing/providers/asset_selection.provider.dart';
import 'package:immich_mobile/modules/sharing/providers/shared_album.provider.dart';
import 'package:immich_mobile/modules/sharing/providers/suggested_shared_users.provider.dart';
import 'package:immich_mobile/modules/sharing/services/shared_album.service.dart';
import 'package:immich_mobile/routing/router.dart';
import 'package:immich_mobile/shared/models/user_info.model.dart';
import 'package:immich_mobile/shared/ui/immich_loading_indicator.dart';

class SelectUserForSharingPage extends HookConsumerWidget {
  const SelectUserForSharingPage({Key? key}) : super(key: key);
  @override
  Widget build(BuildContext context, WidgetRef ref) {
    final sharedUsersList = useState<Set<UserInfo>>({});
    AsyncValue<List<UserInfo>> suggestedShareUsers = ref.watch(suggestedSharedUsersProvider);

    _createSharedAlbum() async {
      var isSuccess = await SharedAlbumService().createSharedAlbum(
        ref.watch(albumTitleProvider),
        ref.watch(assetSelectionProvider).selectedNewAssetsForAlbum,
        sharedUsersList.value.map((userInfo) => userInfo.id).toList(),
      );

      if (isSuccess) {
        await ref.watch(sharedAlbumProvider.notifier).getAllSharedAlbums();
        ref.watch(assetSelectionProvider.notifier).removeAll();
        ref.watch(albumTitleProvider.notifier).clearAlbumTitle();

        AutoRouter.of(context).navigate(const TabControllerRoute(children: [SharingRoute()]));
      }

      const ScaffoldMessenger(child: SnackBar(content: Text('Failed to create album')));
    }

    _buildTileIcon(UserInfo user) {
      if (sharedUsersList.value.contains(user)) {
        return CircleAvatar(
          backgroundColor: Theme.of(context).primaryColor,
          child: const Icon(
            Icons.check_rounded,
            size: 25,
          ),
        );
      } else {
        return CircleAvatar(
          backgroundImage: const AssetImage('assets/immich-logo-no-outline.png'),
          backgroundColor: Theme.of(context).primaryColor.withAlpha(50),
        );
      }
    }

    _buildUserList(List<UserInfo> users) {
      List<Widget> usersChip = [];

      for (var user in sharedUsersList.value) {
        usersChip.add(
          Padding(
            padding: const EdgeInsets.symmetric(horizontal: 8.0),
            child: Chip(
              backgroundColor: Theme.of(context).primaryColor.withOpacity(0.15),
              label: Text(
                user.email,
                style: const TextStyle(fontSize: 12, color: Colors.black87, fontWeight: FontWeight.bold),
              ),
            ),
          ),
        );
      }
      return Column(
        crossAxisAlignment: CrossAxisAlignment.start,
        children: [
          Wrap(
            children: [...usersChip],
          ),
          const Padding(
            padding: EdgeInsets.all(16.0),
            child: Text(
              'Suggestions',
              style: TextStyle(fontSize: 14, color: Colors.grey, fontWeight: FontWeight.bold),
            ),
          ),
          ListView.builder(
            shrinkWrap: true,
            itemBuilder: ((context, index) {
              return ListTile(
                leading: _buildTileIcon(users[index]),
                title: Text(
                  users[index].email,
                  style: const TextStyle(fontSize: 14, fontWeight: FontWeight.bold),
                ),
                onTap: () {
                  if (sharedUsersList.value.contains(users[index])) {
                    sharedUsersList.value =
                        sharedUsersList.value.where((selectedUser) => selectedUser.id != users[index].id).toSet();
                  } else {
                    sharedUsersList.value = {...sharedUsersList.value, users[index]};
                  }
                },
              );
            }),
            itemCount: users.length,
          ),
        ],
      );
    }

    return Scaffold(
      appBar: AppBar(
        title: const Text(
          'Invite to album',
          style: TextStyle(color: Colors.black),
        ),
        elevation: 0,
        centerTitle: false,
        leading: IconButton(
          icon: const Icon(Icons.close_rounded),
          onPressed: () async {
            AutoRouter.of(context).pop();
          },
        ),
        actions: [
          TextButton(
              onPressed: sharedUsersList.value.isEmpty ? null : _createSharedAlbum,
              child: const Text(
                "Create Album",
                style: TextStyle(fontSize: 14, fontWeight: FontWeight.bold),
              ))
        ],
      ),
      body: suggestedShareUsers.when(
        data: (users) {
          return _buildUserList(users);
        },
        error: (e, _) => Text("Error loading suggested users $e"),
        loading: () => const Center(
          child: ImmichLoadingIndicator(),
        ),
      ),
    );
  }
}
