/*
The original code for the Metronome class was found here (UNLICENSE) :
https://github.com/MasterEx/BeatKeeper/blob/master/src/pntanasis/android/metronome/
It was modified to work asynchronously and to integrate features needed by this app
UNLICENSE terms below
*/

/*
This is free and unencumbered software released into the public domain.

Anyone is free to copy, modify, publish, use, compile, sell, or
distribute this software, either in source code form or as a compiled
binary, for any purpose, commercial or non-commercial, and by any
means.

In jurisdictions that recognize copyright laws, the author or authors
of this software dedicate any and all copyright interest in the
software to the public domain. We make this dedication for the benefit
of the public at large and to the detriment of our heirs and
successors. We intend this dedication to be an overt act of
relinquishment in perpetuity of all present and future rights to this
software under copyright law.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
IN NO EVENT SHALL THE AUTHORS BE LIABLE FOR ANY CLAIM, DAMAGES OR
OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
OTHER DEALINGS IN THE SOFTWARE.

For more information, please refer to <http://unlicense.org/>
 */

package agrigolo.chubbyclick.metronome;

import android.os.AsyncTask;

import agrigolo.chubbyclick.utilities.Preferences;


public class Metronome extends AsyncTask<Void, Integer, String> {

    private int sampleRate = 8000;
    private AudioGenerator audioGenerator = new AudioGenerator(sampleRate);
    private Preferences prefs = new Preferences();
    private int accentPitch, beatPitch, tick;
    private boolean play;
    private int silence;
    private double bpm;
    private int beat;
    private String beatSubdivisions;

    private MetronomeNotification notification;
    private MetronomeListener metronomeListener;

    public Metronome(double bpm, int beat, String beatSubdivisions) {

        this.bpm = bpm;
        this.beat = beat;
        this.beatSubdivisions = beatSubdivisions;

        audioGenerator.createPlayer();

        beatPitch = Integer.valueOf(prefs.getBeatPitch());
        accentPitch = Integer.valueOf(prefs.getAccentPitch());
        tick = Integer.valueOf(prefs.getNoteDuration());
        setVolume(Double.parseDouble(prefs.getVolume()));
        notification = new MetronomeNotification();
        this.metronomeListener = null;

    }

    @Override
    protected String doInBackground(Void... arg0) {
        notification.showRunningNotification(beat, bpm);
        play();
        return null;
    }

    private void calcSilence() {
        silence = (int) (((60 / bpm) * sampleRate) - tick);
    }

    public void play() {
        this.play = true;
        int measure = 0;

        double[] accentTick =
                audioGenerator.getSineWave(this.tick, sampleRate, accentPitch);
        double[] beatTick =
                audioGenerator.getSineWave(this.tick, sampleRate, beatPitch);

        double silence = 0;
        double[] sound = new double[sampleRate];
        int t = 0, s = 0, b = 0;
        do {
            //Calculate silence duration in every cycle to be able change the bpm without restarting the task
            calcSilence();
            for (int i = 0; i < sound.length && play; i++) {
                if (t < this.tick) {
                    if (beatSubdivisions.charAt(b) == 'a') {
                        sound[i] = accentTick[t];
                    } else if (beatSubdivisions.charAt(b) == 'b')
                        sound[i] = beatTick[t];
                    t++;
                } else {
                    sound[i] = silence;
                    s++;
                    if (s >= this.silence) {
                        t = 0;
                        s = 0;
                        b++;
                        if (b > (this.beat - 1)) {
                            b = 0;
                            measure++;
                            if (metronomeListener != null)
                                metronomeListener.onMeasureChange(measure);
                        }
                    }
                }
            }
            audioGenerator.writeSound(sound);
        } while (this.play);
    }

    public void stop() {
        this.play = false;
        notification.hideRunningNotification();
        audioGenerator.destroyAudioTrack();
    }


    public void playSample(int freq, int duration) {

        double[] tick =
                audioGenerator.getSineWave(duration, sampleRate, freq);

        double[] sound = new double[sampleRate];
        int t = 0, s = 0, b = 0;

        for (int i = 0; i < sound.length; i++) {
            if (t < duration) {
                sound[i] = tick[t];
                t++;
            }
        }
        audioGenerator.writeSound(sound);
    }

    public interface MetronomeListener {
        public void onMeasureChange(int current_measure);
    }

    public void setMetronomeListener(MetronomeListener listener) {
        this.metronomeListener = listener;
    }

    //Getters and setters

    public boolean isPlaying() {
        return this.play;
    }

    public void setBpm(double new_bpm) {
        bpm = new_bpm;
    }

    public double getBpm() {
        return bpm;
    }

    public void setBeat(int new_beat) {
        beat = new_beat;
    }

    public void setVolume(double new_volume) {
        audioGenerator.setVolume(new_volume);
    }

    public void setBeatSubdivisions(String new_subdivisions) {
        beatSubdivisions = new_subdivisions;
    }

}