package ac.mdiq.podcini.ui.fragment

import ac.mdiq.podcini.R
import ac.mdiq.podcini.databinding.FeedinfoBinding
import ac.mdiq.podcini.net.discovery.CombinedSearcher
import ac.mdiq.podcini.storage.DBTasks
import ac.mdiq.podcini.storage.model.feed.Feed
import ac.mdiq.podcini.storage.model.feed.FeedFunding
import ac.mdiq.podcini.ui.activity.MainActivity
import ac.mdiq.podcini.ui.dialog.EditUrlSettingsDialog
import ac.mdiq.podcini.ui.statistics.StatisticsFragment
import ac.mdiq.podcini.ui.statistics.feed.FeedStatisticsFragment
import ac.mdiq.podcini.ui.view.ToolbarIconTintManager
import ac.mdiq.podcini.util.IntentUtils
import ac.mdiq.podcini.util.Logd
import ac.mdiq.podcini.util.ShareUtils
import ac.mdiq.podcini.util.syndication.HtmlToPlainText
import android.R.string
import android.app.Activity
import android.content.*
import android.content.res.Configuration
import android.graphics.LightingColorFilter
import android.net.Uri
import android.os.Build
import android.os.Bundle
import android.text.TextUtils
import android.util.Log
import android.view.LayoutInflater
import android.view.MenuItem
import android.view.View
import android.view.ViewGroup
import android.widget.ImageView
import android.widget.TextView
import android.widget.Toast
import androidx.activity.result.contract.ActivityResultContracts
import androidx.appcompat.content.res.AppCompatResources
import androidx.appcompat.widget.Toolbar
import androidx.documentfile.provider.DocumentFile
import androidx.fragment.app.Fragment
import androidx.media3.common.util.UnstableApi
import coil.load
import com.google.android.material.appbar.AppBarLayout
import com.google.android.material.appbar.CollapsingToolbarLayout
import com.google.android.material.appbar.MaterialToolbar
import com.google.android.material.dialog.MaterialAlertDialogBuilder
import com.google.android.material.snackbar.Snackbar
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.launch
import kotlinx.coroutines.withContext
import org.apache.commons.lang3.StringUtils

/**
 * Displays information about a feed.
 */
@UnstableApi
class FeedInfoFragment : Fragment(), Toolbar.OnMenuItemClickListener {
    private var _binding: FeedinfoBinding? = null
    private val binding get() = _binding!!

    private var feed: Feed? = null
    private lateinit var imgvCover: ImageView
    private lateinit var txtvTitle: TextView
    private lateinit var txtvDescription: TextView
    private lateinit var txtvFundingUrl: TextView
    private lateinit var lblSupport: TextView
    private lateinit var txtvUrl: TextView
    private lateinit var txtvAuthorHeader: TextView
    private lateinit var imgvBackground: ImageView
    private lateinit var infoContainer: View
    private lateinit var header: View
    private lateinit var toolbar: MaterialToolbar

    private val addLocalFolderLauncher = registerForActivityResult<Uri?, Uri>(AddLocalFolder()) {
        uri: Uri? -> this.addLocalFolderResult(uri)
    }

    private val copyUrlToClipboard = View.OnClickListener {
        if (feed != null && feed!!.download_url != null) {
            val url: String = feed!!.download_url!!
            val clipData: ClipData = ClipData.newPlainText(url, url)
            val cm = requireContext().getSystemService(Context.CLIPBOARD_SERVICE) as ClipboardManager
            cm.setPrimaryClip(clipData)
            if (Build.VERSION.SDK_INT <= 32) (activity as MainActivity).showSnackbarAbovePlayer(R.string.copied_to_clipboard, Snackbar.LENGTH_SHORT)
        }
    }

    override fun onCreateView(inflater: LayoutInflater, container: ViewGroup?, savedInstanceState: Bundle?): View {
        _binding = FeedinfoBinding.inflate(inflater)

        Logd(TAG, "fragment onCreateView")
        toolbar = binding.toolbar
        toolbar.title = ""
        toolbar.inflateMenu(R.menu.feedinfo)
        toolbar.setNavigationOnClickListener { parentFragmentManager.popBackStack() }
        toolbar.setOnMenuItemClickListener(this)
        refreshToolbarState()

        val appBar: AppBarLayout = binding.appBar
        val collapsingToolbar: CollapsingToolbarLayout = binding.collapsingToolbar
        val iconTintManager: ToolbarIconTintManager =
            object : ToolbarIconTintManager(requireContext(), toolbar, collapsingToolbar) {
                override fun doTint(themedContext: Context) {
                    toolbar.menu.findItem(R.id.visit_website_item).setIcon(AppCompatResources.getDrawable(themedContext, R.drawable.ic_web))
                    toolbar.menu.findItem(R.id.share_item).setIcon(AppCompatResources.getDrawable(themedContext, R.drawable.ic_share))
                }
            }
        iconTintManager.updateTint()
        appBar.addOnOffsetChangedListener(iconTintManager)

        imgvCover = binding.header.imgvCover
        txtvTitle = binding.header.txtvTitle
        txtvAuthorHeader = binding.header.txtvAuthor
        imgvBackground = binding.imgvBackground
        header = binding.header.root
        infoContainer = binding.infoContainer
//        binding.header.butShowInfo.visibility = View.INVISIBLE
        binding.header.butShowSettings.visibility = View.INVISIBLE
        binding.header.butFilter.visibility = View.INVISIBLE
        // https://github.com/bumptech/glide/issues/529
        imgvBackground.colorFilter = LightingColorFilter(-0x7d7d7e, 0x000000)

        txtvDescription = binding.txtvDescription
        txtvUrl = binding.txtvUrl
        lblSupport = binding.lblSupport
        txtvFundingUrl = binding.txtvFundingUrl

        binding.btnvRelatedFeeds.setOnClickListener {
            val fragment = OnlineSearchFragment.newInstance(CombinedSearcher::class.java, "${txtvAuthorHeader.text} podcasts")
            (activity as MainActivity).loadChildFragment(fragment, TransitionEffect.SLIDE)
        }

        txtvUrl.setOnClickListener(copyUrlToClipboard)

//        val feedId = requireArguments().getLong(EXTRA_FEED_ID)
        val feedId = feed!!.id
        parentFragmentManager.beginTransaction().replace(R.id.statisticsFragmentContainer,
            FeedStatisticsFragment.newInstance(feedId, false), "feed_statistics_fragment")
            .commitAllowingStateLoss()

        binding.btnvOpenStatistics.setOnClickListener {
            val fragment = StatisticsFragment()
            (activity as MainActivity).loadChildFragment(fragment, TransitionEffect.SLIDE)
        }

        showFeed()
        return binding.root
    }

    override fun onConfigurationChanged(newConfig: Configuration) {
        super.onConfigurationChanged(newConfig)
        val horizontalSpacing = resources.getDimension(R.dimen.additional_horizontal_spacing).toInt()
        header.setPadding(horizontalSpacing, header.paddingTop, horizontalSpacing, header.paddingBottom)
        infoContainer.setPadding(horizontalSpacing, infoContainer.paddingTop, horizontalSpacing, infoContainer.paddingBottom)
    }

    private fun showFeed() {
        if (feed == null) return
        Logd(TAG, "Language is " + feed!!.language)
        Logd(TAG, "Author is " + feed!!.author)
        Logd(TAG, "URL is " + feed!!.download_url)

//        TODO: need to generate blurred image for background
        imgvCover.load(feed!!.imageUrl) {
            placeholder(R.color.light_gray)
            error(R.mipmap.ic_launcher)
        }

        txtvTitle.text = feed!!.title
        txtvTitle.setMaxLines(6)

        val description: String = HtmlToPlainText.getPlainText(feed!!.description?:"")

        txtvDescription.text = description

        if (!feed!!.author.isNullOrEmpty()) txtvAuthorHeader.text = feed!!.author

        txtvUrl.text = feed!!.download_url
        txtvUrl.setCompoundDrawablesRelativeWithIntrinsicBounds(0, 0, R.drawable.ic_paperclip, 0)

        if (feed!!.paymentLinks.isEmpty()) {
            lblSupport.visibility = View.GONE
            txtvFundingUrl.visibility = View.GONE
        } else {
            lblSupport.visibility = View.VISIBLE
            val fundingList: ArrayList<FeedFunding> = feed!!.paymentLinks

            // Filter for duplicates, but keep items in the order that they have in the feed.
            val i: MutableIterator<FeedFunding> = fundingList.iterator()
            while (i.hasNext()) {
                val funding: FeedFunding = i.next()
                for (other in fundingList) {
                    if (TextUtils.equals(other.url, funding.url)) {
                        if (other.content != null && funding.content != null && other.content!!.length > funding.content!!.length) {
                            i.remove()
                            break
                        }
                    }
                }
            }

            var str = StringBuilder()
            for (funding in fundingList) {
                str.append(if (funding.content == null || funding.content!!.isEmpty()) requireContext().resources.getString(R.string.support_podcast)
                else funding.content).append(" ").append(funding.url)
                str.append("\n")
            }
            str = StringBuilder(StringUtils.trim(str.toString()))
            txtvFundingUrl.text = str.toString()
        }

        refreshToolbarState()
    }

    fun setFeed(feed_: Feed) {
        feed = feed_
    }

    override fun onDestroyView() {
        super.onDestroyView()
        _binding = null
        feed = null
    }

    private fun refreshToolbarState() {
        toolbar.menu?.findItem(R.id.reconnect_local_folder)?.setVisible(feed != null && feed!!.isLocalFeed)
        toolbar.menu?.findItem(R.id.share_item)?.setVisible(feed != null && !feed!!.isLocalFeed)
        toolbar.menu?.findItem(R.id.visit_website_item)
            ?.setVisible(feed != null && feed!!.link != null && IntentUtils.isCallable(requireContext(), Intent(Intent.ACTION_VIEW, Uri.parse(feed!!.link))))
        toolbar.menu?.findItem(R.id.edit_feed_url_item)?.setVisible(feed != null && !feed!!.isLocalFeed)
    }

    override fun onMenuItemClick(item: MenuItem): Boolean {
        if (feed == null) {
            (activity as MainActivity).showSnackbarAbovePlayer(R.string.please_wait_for_data, Toast.LENGTH_LONG)
            return false
        }
        when (item.itemId) {
            R.id.visit_website_item -> if (feed!!.link != null) IntentUtils.openInBrowser(requireContext(), feed!!.link!!)
            R.id.share_item -> ShareUtils.shareFeedLink(requireContext(), feed!!)
            R.id.reconnect_local_folder -> {
                val alert = MaterialAlertDialogBuilder(requireContext())
                alert.setMessage(R.string.reconnect_local_folder_warning)
                alert.setPositiveButton(string.ok) { _: DialogInterface?, _: Int ->
                    try {
                        addLocalFolderLauncher.launch(null)
                    } catch (e: ActivityNotFoundException) {
                        Log.e(TAG, "No activity found. Should never happen...")
                    }
                }
                alert.setNegativeButton(string.cancel, null)
                alert.show()
            }
            R.id.edit_feed_url_item -> {
                object : EditUrlSettingsDialog(activity as Activity, feed!!) {
                    override fun setUrl(url: String?) {
                        feed!!.download_url = url
                        txtvUrl.text = feed!!.download_url
                        txtvUrl.setCompoundDrawablesRelativeWithIntrinsicBounds(0, 0, R.drawable.ic_paperclip, 0)
                    }
                }.show()
            }
            else -> return false
        }
        return true
    }

    @UnstableApi private fun addLocalFolderResult(uri: Uri?) {
        if (uri == null) return
        reconnectLocalFolder(uri)
    }

    @UnstableApi private fun reconnectLocalFolder(uri: Uri) {
        if (feed == null) return

//        Completable.fromAction {
//            requireActivity().contentResolver.takePersistableUriPermission(uri, Intent.FLAG_GRANT_READ_URI_PERMISSION)
//            val documentFile = DocumentFile.fromTreeUri(requireContext(), uri)
//            requireNotNull(documentFile) { "Unable to retrieve document tree" }
//            feed!!.download_url = Feed.PREFIX_LOCAL_FOLDER + uri.toString()
//            DBTasks.updateFeed(requireContext(), feed!!, true)
//        }
//            .subscribeOn(Schedulers.io())
//            .observeOn(AndroidSchedulers.mainThread())
//            .subscribe({ (activity as MainActivity).showSnackbarAbovePlayer(string.ok, Snackbar.LENGTH_SHORT) },
//                { error: Throwable -> (activity as MainActivity).showSnackbarAbovePlayer(error.localizedMessage, Snackbar.LENGTH_LONG) })

        val scope = CoroutineScope(Dispatchers.Main)
        scope.launch {
            try {
                withContext(Dispatchers.IO) {
                    requireActivity().contentResolver.takePersistableUriPermission(uri, Intent.FLAG_GRANT_READ_URI_PERMISSION)
                    val documentFile = DocumentFile.fromTreeUri(requireContext(), uri)
                    requireNotNull(documentFile) { "Unable to retrieve document tree" }
                    feed!!.download_url = Feed.PREFIX_LOCAL_FOLDER + uri.toString()
                    DBTasks.updateFeed(requireContext(), feed!!, true)
                }
                withContext(Dispatchers.Main) {
                    (activity as MainActivity).showSnackbarAbovePlayer(string.ok, Snackbar.LENGTH_SHORT)
                }
            } catch (e: Throwable) {
                withContext(Dispatchers.Main) {
                    (activity as MainActivity).showSnackbarAbovePlayer(e.localizedMessage, Snackbar.LENGTH_LONG)
                }
            }
        }
    }

    private class AddLocalFolder : ActivityResultContracts.OpenDocumentTree() {
        override fun createIntent(context: Context, input: Uri?): Intent {
            return super.createIntent(context, input).addFlags(Intent.FLAG_GRANT_READ_URI_PERMISSION)
        }
    }

    companion object {
        private const val TAG = "FeedInfoActivity"

        fun newInstance(feed: Feed): FeedInfoFragment {
            val fragment = FeedInfoFragment()
            fragment.setFeed(feed)
            return fragment
        }
    }
}
