package ac.mdiq.podcini.net.sync.queue

import ac.mdiq.podcini.net.sync.LockingAsyncExecutor
import ac.mdiq.podcini.net.sync.SynchronizationSettings.isProviderConnected
import ac.mdiq.podcini.net.sync.SynchronizationSettings.lastSyncAttempt
import ac.mdiq.podcini.net.sync.model.EpisodeAction
import ac.mdiq.podcini.storage.model.Episode

object SynchronizationQueueSink {
    // To avoid a dependency loop of every class to SyncService, and from SyncService back to every class.
    private var serviceStarterImpl = {}

    fun setServiceStarterImpl(serviceStarter: ()->Unit) {
        serviceStarterImpl = serviceStarter
    }

    private fun syncNow() {
        serviceStarterImpl()
    }

    fun syncNowIfNotSyncedRecently() {
        if (System.currentTimeMillis() - lastSyncAttempt > 1000 * 60 * 10) syncNow()
    }

    
    fun clearQueue() {
        LockingAsyncExecutor.executeLockedAsync { SynchronizationQueueStorage().clearQueue() }
    }

    fun enqueueFeedAddedIfSyncActive(downloadUrl: String) {
        if (!isProviderConnected) return
        LockingAsyncExecutor.executeLockedAsync {
            SynchronizationQueueStorage().enqueueFeedAdded(downloadUrl)
            syncNow()
        }
    }

    fun enqueueFeedRemovedIfSyncActive(downloadUrl: String) {
        if (!isProviderConnected) return
        LockingAsyncExecutor.executeLockedAsync {
            SynchronizationQueueStorage().enqueueFeedRemoved(downloadUrl)
            syncNow()
        }
    }

    fun enqueueEpisodeActionIfSyncActive(action: EpisodeAction) {
        if (!isProviderConnected) return
        LockingAsyncExecutor.executeLockedAsync {
            SynchronizationQueueStorage().enqueueEpisodeAction(action)
            syncNow()
        }
    }

    fun enqueueEpisodePlayedIfSyncActive(media: Episode, completed: Boolean) {
        if (!isProviderConnected) return
        if (media.feed?.isLocalFeed == true) return
        if (media.startPosition < 0 || (!completed && media.startPosition >= media.position)) return
        val action = EpisodeAction.Builder(media, EpisodeAction.PLAY)
            .currentTimestamp()
            .started(media.startPosition / 1000)
            .position((if (completed) media.duration else media.position) / 1000)
            .total(media.duration / 1000)
            .build()
        enqueueEpisodeActionIfSyncActive(action)
    }
}
