package ac.mdiq.podcini.storage.database

import ac.mdiq.podcini.PodciniApp.Companion.getAppContext
import ac.mdiq.podcini.automation.autodownloadForQueue
import ac.mdiq.podcini.automation.autoenqueueForQueue
import ac.mdiq.podcini.playback.base.InTheatre.VIRTUAL_QUEUE_SIZE
import ac.mdiq.podcini.playback.base.InTheatre.actQueue
import ac.mdiq.podcini.playback.base.InTheatre.curEpisode
import ac.mdiq.podcini.playback.base.InTheatre.savePlayerStatus
import ac.mdiq.podcini.playback.service.PlaybackService.Companion.episodeChangedWhenScreenOff
import ac.mdiq.podcini.storage.model.Episode
import ac.mdiq.podcini.storage.model.PlayQueue
import ac.mdiq.podcini.storage.model.QueueEntry
import ac.mdiq.podcini.storage.model.VIRTUAL_QUEUE_ID
import ac.mdiq.podcini.storage.specs.EnqueueLocation
import ac.mdiq.podcini.storage.specs.EpisodeSortOrder
import ac.mdiq.podcini.storage.specs.EpisodeSortOrder.Companion.getPermutor
import ac.mdiq.podcini.storage.specs.EpisodeState
import ac.mdiq.podcini.utils.EventFlow
import ac.mdiq.podcini.utils.FlowEvent.QueueEvent
import ac.mdiq.podcini.utils.Logd
import ac.mdiq.podcini.utils.Loge
import ac.mdiq.podcini.utils.Logt
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.setValue
import io.github.xilinjia.krdb.notifications.ResultsChange
import io.github.xilinjia.krdb.notifications.UpdatedResults
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.launch
import java.util.Date
import kotlin.math.min
import kotlin.random.Random

private const val TAG: String = "Queues"

const val QUEUE_POSITION_DELTA = 10000L

val queuesFlow = realm.query(PlayQueue::class).sort("name").asFlow()
var queuesLive = realm.query(PlayQueue::class).sort("name").find()
    private set

val queuesJob = CoroutineScope(Dispatchers.Default).launch {
    queuesFlow.collect { changes: ResultsChange<PlayQueue> ->
        queuesLive = changes.list
        val q = queuesLive.find { it.id == actQueue.id }
        if (q != null) actQueue = q
        Logd(TAG, "queuesLive updated")
        when (changes) {
            is UpdatedResults -> {
                when {
                    changes.insertions.isNotEmpty() -> {}
                    changes.changes.isNotEmpty() -> {}
                    changes.deletions.isNotEmpty() -> {}
                    else -> {}
                }
            }
            else -> {}
        }
    }
}

fun cancelQueuesJob() {
    queuesJob.cancel()
}

var virQueue by mutableStateOf(realm.query(PlayQueue::class).query("id == $VIRTUAL_QUEUE_ID").first().find() ?:
run {
    val vq = PlayQueue()
    vq.id = VIRTUAL_QUEUE_ID
    vq.name = "Virtual"
    upsertBlk(vq) {}
})

var curIndexInActQueue = -1

fun inQueueEpisodeIdSet(): Set<Long> {
    Logd(TAG, "getQueueIDList() called")
    return realm.query(QueueEntry::class).find().map { it.episodeId }.toSet()
}

suspend fun sortQueue(queue: PlayQueue) {
    val queueEntries = queueEntriesOf(queue)
    val episodes = realm.query(Episode::class).query("id IN $0", queueEntries.map { it.episodeId }).find().toMutableList()
    getPermutor(queue.sortOrder).reorder(episodes)
    realm.write {
        for (i in episodes.indices) {
            val e = episodes[i]
            val qe = queueEntries.find { it.episodeId == e.id }
            if (qe == null) {
                Loge(TAG, "Can't find queueEntry for episode: ${e.title}")
                continue
            }
            findLatest(qe)?.position = (i+1) * QUEUE_POSITION_DELTA
        }
    }
}

fun queueEntriesOf(queue: PlayQueue): List<QueueEntry> {
    return realm.query(QueueEntry::class).query("queueId == ${queue.id}").sort("position").find()
}

suspend fun addToAssQueue(episodes: List<Episode>) {
    Logd(TAG, "addToQueueSync( ... ) called")
    val mapByFeed = episodes.groupBy { it.feedId }
    for (en in mapByFeed.entries) {
        val fid = en.key ?: continue
        val f = getFeed(fid) ?: continue
        val q = f.queue ?: continue
        val episodes = en.value
        addToQueue(episodes, q)
    }
}

suspend fun addToQueue(episodes: List<Episode>, queue: PlayQueue) {
    Logd(TAG, "addToQueueSync( ... ) called")
    if (queue.isVirtual()) {
        Loge(TAG, "Current queue is virtual, ignored")
        return
    }
    val time = System.currentTimeMillis()
    var i = 0L
    var qes = queueEntriesOf(queue)
    realm.write {
        for (e in episodes) {
            if (qes.indexOfFirst { it.episodeId == e.id } >= 0) continue
            val insertPosition = if (queue.autoSort) 0 else {
                qes = queueEntriesOf(queue)
                calcPosition(qes, EnqueueLocation.fromCode(queue.enqueueLocation), (if (queue.id == actQueue.id) curEpisode else null))
            }
            Logd(TAG, "addToQueueSync insertPosition: $insertPosition")
            val qe = QueueEntry().apply {
                id = time + i++
                queueId = queue.id
                episodeId = e.id
                position = insertPosition
            }
            copyToRealm(qe)
        }
    }
    for (e in episodes) if (e.playState < EpisodeState.QUEUE.code) setPlayState(EpisodeState.QUEUE, e, false)
    if (queue.autoSort) sortQueue(queue)
}

suspend fun queueToVirtual(episode: Episode, episodes: List<Episode>, listIdentity: String, sortOrder: EpisodeSortOrder, playInSequence: Boolean = true) {
    Logd(TAG, "queueToVirtual ${virQueue.identity} $listIdentity ${episodes.size}")
    virQueue = queuesLive.find { it.id == VIRTUAL_QUEUE_ID } ?: return
    if (virQueue.identity != listIdentity) {
        if (virQueue.contains(episode)) {
            Logd(TAG, "VirQueue has the episode, ignore")
            actQueue = virQueue
            return
        }
        val index = episodes.indexOfFirst { it.id == episode.id }
        if (index >= 0) {
            Logd(TAG, "queueToVirtual index: $index")
            realm.write {
                val qes = query(QueueEntry::class).query("queueId == $VIRTUAL_QUEUE_ID").find()
                delete(qes)
            }
            val eIdsToQueue = episodes.subList(index, min(episodes.size, index + VIRTUAL_QUEUE_SIZE)).map { it.id }
            virQueue = upsert(virQueue) { q ->
                q.identity = listIdentity
                q.playInSequence = playInSequence
                q.sortOrder = sortOrder
            }
            val time = System.currentTimeMillis()
            var i = 0L
            var ip = QUEUE_POSITION_DELTA
            realm.write {
                for (eid in eIdsToQueue) {
                    val qe = QueueEntry().apply {
                        id = time + i++
                        queueId = virQueue.id
                        episodeId = eid
                        position = ip
                    }
                    copyToRealm(qe)
                    ip += QUEUE_POSITION_DELTA
                }
            }
            actQueue = virQueue
            Logt(TAG, "first ${virQueue.size()} episodes are added to the Virtual queue")
        }
    } else actQueue = virQueue
}

suspend fun smartRemoveFromAllQueues(item_: Episode) {
    Logd(TAG, "smartRemoveFromAllQueues: ${item_.title}")
    var item = item_
    val almostEnded = item.hasAlmostEnded()
    if (almostEnded && item.playState < EpisodeState.PLAYED.code && !stateToPreserve(item.playState)) item = setPlayState(EpisodeState.PLAYED, item, resetMediaPosition = true, removeFromQueue = false)
    if (almostEnded) item = upsert(item) { it.playbackCompletionDate = Date() }
    if (item.playState < EpisodeState.SKIPPED.code && !stateToPreserve(item.playState)) {
        val stat = if (item.lastPlayedTime > 0L) EpisodeState.SKIPPED else EpisodeState.PASSED
        item = setPlayState(stat, item, resetMediaPosition = false, removeFromQueue = false)
    }
    for (q in queuesLive) {
        if (q.id != actQueue.id && q.contains(item)) removeFromQueue(q, listOf(item))
    }
    //        ensure actQueue is last updated
    Logd(TAG, "actQueue: [${actQueue.name}]")
    val qes = queueEntriesOf(actQueue)
    if (curEpisode != null) curIndexInActQueue = qes.indexOfFirst { it.episodeId == curEpisode!!.id }
    if (actQueue.size() > 0 && actQueue.contains(item)) removeFromQueue(actQueue, listOf(item))
    else upsertBlk(actQueue) { it.update() }
    if (actQueue.size() == 0 && !actQueue.isVirtual()) {
        autoenqueueForQueue(actQueue)
        if(actQueue.launchAutoEQDlWhenEmpty) autodownloadForQueue(getAppContext(), actQueue)
    }
}

fun trimBin(queue: PlayQueue) {
    if (queue.binLimit <= 0) return
    if (queue.idsBinList.size > queue.binLimit * 1.2) {
        val newSize = (0.2 * queue.binLimit).toInt()
        val subList = queue.idsBinList.subList(0, newSize)
        queue.idsBinList.clear()
        queue.idsBinList.addAll(subList)
    }
}

fun removeFromAllQueues(episodes: Collection<Episode>, playState: EpisodeState? = null) {
    Logd(TAG, "removeFromAllQueuesSync called ")
    for (e in episodes) {
        for (q in queuesLive) {
            if (q.id != actQueue.id && q.contains(e)) removeFromQueue(q, listOf(e), playState)
        }
    }
    //        ensure actQueue is last updated
    val qes = queueEntriesOf(actQueue)
    for (e in episodes) {
        if (curEpisode != null && e.id == curEpisode!!.id) curIndexInActQueue = qes.indexOfFirst { it.episodeId == curEpisode!!.id }
        if (actQueue.size() > 0 && actQueue.contains(e)) removeFromQueue(actQueue, listOf(e), playState)
    }
    upsertBlk(actQueue) { it.update() }
    if (actQueue.size() == 0 && !actQueue.isVirtual()) {
        autoenqueueForQueue(actQueue)
        if(actQueue.launchAutoEQDlWhenEmpty) autodownloadForQueue(getAppContext(), actQueue)
    }
}

/**
 * @param queue_    if null, use actQueue
 */
internal fun removeFromQueue(queue_: PlayQueue?, episodes: Collection<Episode>, playState: EpisodeState? = null) {
    Logd(TAG, "removeFromQueue called ")
    if (episodes.isEmpty()) return
    val queue = queue_ ?: actQueue
    if (queue.size() == 0) {
        if (!queue.isVirtual()) {
            autoenqueueForQueue(queue)
            if (queue.launchAutoEQDlWhenEmpty) autodownloadForQueue(getAppContext(), queue)
        }
        return
    }
    val removeFromActQueue = mutableListOf<Episode>()
    val indicesToRemove: MutableList<Int> = mutableListOf()
    val qItems = queue.episodes.toMutableList()
    val eList = episodes.toList()
    realm.writeBlocking {
        val qes = query(QueueEntry::class).query("queueId == $0 AND episodeId IN $1", queue.id, episodes.map { it.id }).find()
        delete(qes)
    }
    for (e in eList) {
        val i = qItems.indexWithId(e.id)
        if (i >= 0) {
            indicesToRemove.add(i)
            upsertBlk(e) {
                if (playState != null && it.playState == EpisodeState.QUEUE.code) it.setPlayState(playState)
            }
            if (queue.id == actQueue.id) removeFromActQueue.add(e)
        }
    }
    if (indicesToRemove.isNotEmpty()) {
        val queueNew = upsertBlk(queue) {
            for (i in indicesToRemove.indices.reversed()) {
                val id = qItems[indicesToRemove[i]].id
                it.idsBinList.remove(id)
                it.idsBinList.add(id)
            }
            trimBin(it)
            it.update()
        }
        EventFlow.postEvent(QueueEvent.removed(removeFromActQueue))
        if (queueNew.size() == 0 && !queueNew.isVirtual()) {
            autoenqueueForQueue(queueNew)
            if(queueNew.launchAutoEQDlWhenEmpty) autodownloadForQueue(getAppContext(), queueNew)
        }
    } else Logd(TAG, "Queue was not modified by call to removeQueueItem")
}

suspend fun removeFromAllQueuesQuiet(episodeIds: List<Long>, updateState: Boolean = true) {
    Logd(TAG, "removeFromAllQueuesQuiet called ")

    suspend fun doit(q: PlayQueue, isActQueue: Boolean = false) {
        if (q.size() == 0) {
            if (isActQueue) upsert(q) { it.update() }
            if (!q.isVirtual()) {
                autoenqueueForQueue(q)
                if (q.launchAutoEQDlWhenEmpty) autodownloadForQueue(getAppContext(), q)
            }
            return
        }
        val qes = realm.query(QueueEntry::class).query("queueId == $0 AND episodeId IN $1", q.id, episodeIds).find()
        val idsInQueuesToRemove = qes.map { it.episodeId }
        if (idsInQueuesToRemove.isNotEmpty()) {
            realm.write {
                for (qe in qes) {
                    val qe_ = findLatest(qe)
                    if (qe_ != null) delete (qe_)
                }
            }
            val eList = realm.query(Episode::class).query("id IN $0", idsInQueuesToRemove).find()
            for (e in eList) {
                if (updateState && e.playState < EpisodeState.SKIPPED.code && !stateToPreserve(e.playState))
                    setPlayState(EpisodeState.SKIPPED, e, false)
            }
            val qNew = upsert(q) {
                it.idsBinList.removeAll(idsInQueuesToRemove)
                it.idsBinList.addAll(idsInQueuesToRemove)
                trimBin(it)
                it.update()
            }
            if (qNew.size() == 0 && !q.isVirtual()) {
                autoenqueueForQueue(qNew)
                if(qNew.launchAutoEQDlWhenEmpty) autodownloadForQueue(getAppContext(), qNew)
            }
//            if (isActQueue) actQueue = qNew
        }
    }

    for (q in queuesLive) {
        if (q.id == actQueue.id) continue
        doit(q)
    }
    //        ensure actQueue is last updated
    doit(actQueue, true)
}

fun getNextInQueue(currentMedia: Episode?): Episode? {
    Logd(TAG, "getNextInQueue called currentMedia: ${currentMedia?.getEpisodeTitle()}")
    if (!actQueue.playInSequence) {
        Logd(TAG, "getNextInQueue(), but follow queue is not enabled.")
        savePlayerStatus(null)
        return null
    }
    val qes = queueEntriesOf(actQueue)
    if (qes.isEmpty()) {
        Logd(TAG, "getNextInQueue queue is empty")
        savePlayerStatus(null)
        return null
    }
    var curIndex = if (currentMedia != null) qes.indexOfFirst { it.episodeId == currentMedia.id } else 0
    if (curIndex < 0 && curIndexInActQueue >= 0) {
        curIndex = curIndexInActQueue
        curIndexInActQueue = -1
    }
    Logd(TAG, "getNextInQueue curIndexInQueue: $curIndex ${qes.size}")
    val nextQE = if (curIndex >= 0 && curIndex < qes.size) {
        when {
            qes[curIndex].episodeId != currentMedia?.id -> qes[curIndex]
            qes.size == 1 -> return null
            else -> {
                val j = if (curIndex < qes.size - 1) curIndex + 1 else 0
                Logd(TAG, "getNextInQueue next j: $j")
                qes[j]
            }
        }
    } else qes[0]
    var nextItem = getEpisode(nextQE.episodeId) ?: return null
    Logd(TAG, "getNextInQueue nextItem ${nextItem.title}")
    nextItem = checkAndMarkDuplicates(nextItem)
    episodeChangedWhenScreenOff = true
    return nextItem
}

private fun calcPosition(queueEntries: List<QueueEntry>, loc: EnqueueLocation, currentPlaying: Episode?): Long {
    if (queueEntries.isEmpty()) return 0

    fun getCurrentlyPlayingPosition(): Int {
        if (currentPlaying == null) return -1
        val curPlayingItemId = currentPlaying.id
        for (i in queueEntries.indices) if (curPlayingItemId == queueEntries[i].episodeId) return i
        return -1
    }
    val size = queueEntries.size
    return when (loc) {
        EnqueueLocation.BACK -> queueEntries[size-1].position + QUEUE_POSITION_DELTA
        EnqueueLocation.FRONT -> QUEUE_POSITION_DELTA / 2
        EnqueueLocation.AFTER_CURRENTLY_PLAYING -> queueEntries[getCurrentlyPlayingPosition()].position + QUEUE_POSITION_DELTA / 2
        EnqueueLocation.RANDOM -> queueEntries[Random.nextInt(queueEntries.size)].position + QUEUE_POSITION_DELTA / 2
    }
}

